<?php

/**
 * @package     Comdev.Module
 * @subpackage  mod_onecore_items
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Module\OnecoreItems\Site\Helper;

use Comdev\Component\Onecore\Site\Helper\ContentHelper;
use Comdev\Component\Onecore\Site\Helper\CustomFieldsHelper;
use Joomla\CMS\Factory;
use Joomla\Database\DatabaseAwareInterface;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Database\ParameterType;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Helper class for mod_onecore_items
 *
 * @since  1.0.0
 */
class ItemsHelper implements DatabaseAwareInterface
{
	use DatabaseAwareTrait;

	/**
	 * Get list of items
	 *
	 * @param   array  $params  Module parameters
	 *
	 * @return  array  Array of item objects
	 *
	 * @since   1.0.0
	 */
	public function getItems(array $params): array
	{
		$db = $this->getDatabase();
		$query = $db->createQuery();

		$count = (int) ($params['count'] ?? 6);
		$showLatest = (int) ($params['show_latest'] ?? 1);
		$featuredOnly = (int) ($params['featured_only'] ?? 0);

		// Select the required fields from the table
		$query->select([
			$db->quoteName('a.id'),
			$db->quoteName('a.title'),
			$db->quoteName('a.alias'),
			$db->quoteName('a.introtext'),
			$db->quoteName('a.fulltext'),
			$db->quoteName('a.published'),
			$db->quoteName('a.created'),
			$db->quoteName('a.created_by'),
			$db->quoteName('a.modified'),
			$db->quoteName('a.hits'),
			$db->quoteName('a.featured'),
			$db->quoteName('a.language'),
			$db->quoteName('a.images'),
			$db->quoteName('a.video_link'),
			$db->quoteName('a.access'),
		])
			->from($db->quoteName('#__one_content', 'a'));

		// Join on user table
		$query->select($db->quoteName('u.name', 'author'))
			->leftJoin($db->quoteName('#__users', 'u'), $db->quoteName('u.id') . ' = ' . $db->quoteName('a.created_by'));

		// Filter by published state
		$query->where($db->quoteName('a.published') . ' = 1');

		// Filter by featured
		if ($featuredOnly) {
			$query->where($db->quoteName('a.featured') . ' = 1');
		}

		// Filter by access level
		$user = Factory::getUser();
		if (!$user->authorise('core.admin')) {
			$groups = $user->getAuthorisedViewLevels();
			$query->whereIn($db->quoteName('a.access'), $groups);
		}

		// Ordering - show latest first if enabled
		if ($showLatest) {
			$query->order($db->quoteName('a.created') . ' DESC');
		} else {
			$query->order($db->quoteName('a.ordering') . ' ASC');
		}

		// Limit results
		$query->setLimit($count);

		$db->setQuery($query);

		try {
			$items = $db->loadObjectList() ?: [];

			// Process items - add images and categories
			foreach ($items as &$item) {
				// Get first image
				try {
					$item->image = ContentHelper::getFirstImage($item->images ?? '');
				} catch (\Exception $e) {
					$item->image = null;
				}
				
				// Get all categories for this item
				try {
					$item->categories = ContentHelper::getCategories($item->id);
				} catch (\Exception $e) {
					$item->categories = [];
				}
			}

			// Attach custom fields for module context ("module") in both positions
			$ids = array_map(static fn($it) => (int) ($it->id ?? 0), $items ?: []);
			$ids = array_values(array_filter($ids));

			if (!empty($ids)) {
				$fieldsUnderTitle = CustomFieldsHelper::getFieldsForContentIds($ids, 'module', 'under_title');
				$fieldsUnderIntro = CustomFieldsHelper::getFieldsForContentIds($ids, 'module', 'under_intro');

				foreach ($items as &$it) {
					$cid = (int) ($it->id ?? 0);
					$it->customfields_under_title = $fieldsUnderTitle[$cid] ?? [];
					$it->customfields_under_intro = $fieldsUnderIntro[$cid] ?? [];
				}
			}

			return $items;
		} catch (\Exception $e) {
			// Log error for debugging
			Factory::getApplication()->getLogger()->error('Error in mod_onecore_items helper: ' . $e->getMessage(), ['exception' => $e]);
			return [];
		}
	}
}
