<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Table;

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Table\Table;
use Joomla\Database\DatabaseDriver;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Review table class.
 *
 * @since  1.0.16
 */
class ReviewTable extends Table
{
	/**
	 * Constructor
	 *
	 * @param   DatabaseDriver  $db  Database connector object
	 *
	 * @since   1.0.16
	 */
	public function __construct(DatabaseDriver $db)
	{
		$this->typeAlias = 'com_onecore.review';

		parent::__construct('#__one_reviews', 'id', $db);
	}

	/**
	 * Method to perform sanity checks on the Table instance properties.
	 *
	 * @return  boolean  True if the instance is sane and able to be stored in the database.
	 *
	 * @since   1.0.16
	 */
	public function check()
	{
		// Enforce exactly one reference: content_id XOR event_id (other must be NULL for DB constraint)
		$contentId = (int) ($this->content_id ?? 0);
		$eventId = (int) ($this->event_id ?? 0);
		if ($contentId > 0 && $eventId > 0) {
			$this->setError(Text::_('COM_ONECORE_REVIEW_EITHER_CONTENT_OR_EVENT'));
			return false;
		}
		if ($contentId === 0 && $eventId === 0) {
			$this->setError(Text::_('COM_ONECORE_REVIEW_EITHER_CONTENT_OR_EVENT'));
			return false;
		}
		if ($contentId > 0) {
			$this->event_id = null;
		} else {
			$this->content_id = null;
		}

		try {
			parent::check();
		} catch (\Exception $e) {
			$this->setError($e->getMessage());
			return false;
		}

		// Rating 1-5
		$rating = (int) ($this->rating ?? 0);
		if ($rating < 1 || $rating > 5) {
			$this->setError(Text::_('COM_ONECORE_REVIEW_RATING_INVALID'));
			return false;
		}

		if (trim((string) ($this->name ?? '')) === '') {
			$this->setError(Text::_('COM_ONECORE_REVIEW_NAME_REQUIRED'));
			return false;
		}

		if (trim((string) ($this->email ?? '')) === '') {
			$this->setError(Text::_('COM_ONECORE_REVIEW_EMAIL_REQUIRED'));
			return false;
		}

		if (empty($this->created) || $this->created === '0000-00-00 00:00:00') {
			$this->created = Factory::getDate()->toSql();
		}

		return true;
	}
}
