<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Table;

use Joomla\CMS\Table\Nested;
use Joomla\Database\DatabaseDriver;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Category table class.
 *
 * @since  1.0.0
 */
class CategoryTable extends Nested
{
	/**
	 * Disable Joomla asset tracking to avoid locking other tables.
	 *
	 * @var bool
	 */
	protected $_trackAssets = false;

	/**
	 * Disable table locking to avoid LOCK TABLES issues with other tables (e.g. #__extensions).
	 *
	 * @return  boolean
	 */
	protected function _lock()
	{
		return true;
	}

	/**
	 * Disable unlock (paired with _lock override).
	 *
	 * @return  boolean
	 */
	protected function _unlock()
	{
		return true;
	}

	/**
	 * Constructor
	 *
	 * @param   DatabaseDriver  $db  Database connector object
	 *
	 * @since   1.0.0
	 */
	public function __construct(DatabaseDriver $db)
	{
		$this->typeAlias = 'com_onecore.category';

		// Use component-specific table
		parent::__construct('#__one_categories', 'id', $db);

		// Explicitly disable asset tracking for this custom table
		$this->_trackAssets = false;
	}

	/**
	 * Override bind to ensure extension is set to com_onecore
	 *
	 * @param   array|object  $src     An associative array or object to bind to the Table instance.
	 * @param   array|string   $ignore  An optional array or space separated list of properties to ignore while binding.
	 *
	 * @return  boolean  True on success.
	 *
	 * @since   1.0.0
	 */
	public function bind($src, $ignore = [])
	{
		// Ensure extension is always set to com_onecore
		if (is_array($src)) {
			$src['extension'] = 'com_onecore';
		} elseif (is_object($src)) {
			$src->extension = 'com_onecore';
		}

		return parent::bind($src, $ignore);
	}

	/**
	 * Method to compute the default name of the asset.
	 * The default name is in the form table_name.id
	 * where id is the value of the primary key of the table.
	 *
	 * @return  string
	 *
	 * @since   1.0.0
	 */
	protected function _getAssetName()
	{
		$k = $this->_tbl_key;

		return 'com_onecore.category.' . (int) $this->$k;
	}

	/**
	 * Method to return the title to use for the asset table.
	 *
	 * @return  string
	 *
	 * @since   1.0.0
	 */
	protected function _getAssetTitle()
	{
		return $this->title;
	}

	/**
	 * Rebuild the nested set tree. Override to support multiple roots (parent_id = 0).
	 *
	 * @param   integer  $parentId  The root of the tree to rebuild.
	 * @param   integer  $leftId    The left id to start with.
	 * @param   integer  $level     The level to assign to the current nodes.
	 * @param   string   $path      The path to the current nodes.
	 *
	 * @return  integer|boolean  Right value of the root on success, false on failure.
	 *
	 * @since   1.0.0
	 */
	public function rebuild($parentId = null, $leftId = 0, $level = 0, $path = '')
	{
		$db = $this->getDatabase();
		$k  = $this->_tbl_key;

		if ($parentId === null) {
			// Get all roots (parent_id = 0) ordered by id
			$query = $db->createQuery()
				->select($k)
				->from($this->_tbl)
				->where('parent_id = 0')
				->order($k . ' ASC');
			$db->setQuery($query);
			$rootIds = $db->loadColumn();

			if (empty($rootIds)) {
				return false;
			}

			$rightId = $leftId;
			foreach ($rootIds as $rootId) {
				$rightId = parent::rebuild((int) $rootId, $rightId, 0, '');
				if ($rightId === false) {
					return false;
				}
			}

			return $rightId;
		}

		return parent::rebuild($parentId, $leftId, $level, $path);
	}

	/**
	 * Override check function to allow parent_id = 0 for top-level categories
	 * Similar to Joomla\CMS\Table\Category::check()
	 *
	 * @return  boolean
	 *
	 * @see     Nested::check()
	 * @since   1.0.0
	 */
	public function check()
	{
		try {
			// Call Table::check() instead of Nested::check() to avoid parent_id validation
			\Joomla\CMS\Table\Table::check();
		} catch (\Exception $e) {
			$this->setError($e->getMessage());
			return false;
		}

		// Check for a title.
		if (trim($this->title) == '') {
			$this->setError(\Joomla\CMS\Language\Text::_('JLIB_DATABASE_ERROR_MUSTCONTAIN_A_TITLE_CATEGORY'));
			return false;
		}

		$this->alias = trim($this->alias ?? '');

		if (empty($this->alias)) {
			$this->alias = $this->title;
		}

		$this->alias = \Joomla\CMS\Application\ApplicationHelper::stringURLSafe($this->alias, $this->language ?? '*');

		if (trim(str_replace('-', '', $this->alias)) == '') {
			$this->alias = \Joomla\CMS\Factory::getDate()->format('Y-m-d-H-i-s');
		}

		// Validate parent_id if it's not 0
		$this->parent_id = (int) $this->parent_id;

		if ($this->parent_id > 0) {
			$db = $this->getDatabase();
			$query = $db->createQuery()
				->select('1')
				->from($this->_tbl)
				->where($this->_tbl_key . ' = :parentId')
				->bind(':parentId', $this->parent_id, \Joomla\Database\ParameterType::INTEGER);

			if (!$db->setQuery($query)->loadResult()) {
				$this->setError(\Joomla\CMS\Language\Text::sprintf('JLIB_DATABASE_ERROR_INVALID_PARENT_ID', $this->parent_id));
				return false;
			}
		}

		return true;
	}

	/**
	 * Overridden Table::store to set created/modified and user id.
	 * Also handles extension filtering for nested set operations.
	 *
	 * @param   boolean  $updateNulls  True to update fields even if they are null.
	 *
	 * @return  boolean  True on success.
	 *
	 * @since   1.0.0
	 */
	public function store($updateNulls = false)
	{
		$k = $this->_tbl_key;
		$date = \Joomla\CMS\Factory::getDate()->toSql();
		$user = \Joomla\CMS\Factory::getApplication()->getIdentity();

		// Ensure extension is set
		if (empty($this->extension)) {
			$this->extension = 'com_onecore';
		}

		// Ensure there is a root category for this extension
		$rootId = $this->getExtensionRootId($this->extension, $user->id, $date);

		// For new records, if no location set, insert as last child of root
		if (empty($this->$k) && $this->_location_id == 0) {
			$this->setLocation($rootId, 'last-child');
		}

		// Set created if not set.
		if (empty($this->created) || $this->created === '0000-00-00 00:00:00') {
			$this->created = $date;
		}

		if ($this->id) {
			// Existing category
			$this->modified_by = $user->id;
			$this->modified    = $date;
		} else {
			if (empty($this->modified) || $this->modified === '0000-00-00 00:00:00') {
				$this->modified = $this->created;
			}

			// Field created_by can be set by the user, so we don't touch it if it's set.
			if (empty($this->created_by)) {
				$this->created_by = $user->id;
			}

			if (empty($this->modified_by)) {
				$this->modified_by = $this->created_by;
			}
		}

		// Verify that the alias is unique (only within same extension and parent)
		$db = $this->getDatabase();
		$query = $db->createQuery()
			->select($db->quoteName('id'))
			->from($db->quoteName($this->_tbl))
			->where($db->quoteName('alias') . ' = :alias')
			->where($db->quoteName('parent_id') . ' = :parentId')
			->where($db->quoteName('extension') . ' = :extension')
			->bind(':alias', $this->alias)
			->bind(':parentId', $this->parent_id, \Joomla\Database\ParameterType::INTEGER)
			->bind(':extension', $this->extension);

		$db->setQuery($query);
		$existingId = $db->loadResult();

		if ($existingId && ($existingId != $this->id || $this->id == 0)) {
			// Check if existing category is trashed
			$query = $db->createQuery()
				->select($db->quoteName('published'))
				->from($db->quoteName($this->_tbl))
				->where($db->quoteName('id') . ' = :id')
				->bind(':id', $existingId, \Joomla\Database\ParameterType::INTEGER);
			
			$db->setQuery($query);
			$published = $db->loadResult();

			$this->setError(\Joomla\CMS\Language\Text::_('JLIB_DATABASE_ERROR_CATEGORY_UNIQUE_ALIAS'));

			if ($published === -2) {
				$this->setError(\Joomla\CMS\Language\Text::_('JLIB_DATABASE_ERROR_CATEGORY_UNIQUE_ALIAS_TRASHED'));
			}

			return false;
		}

		// Call parent store which handles nested set
		$result = parent::store($updateNulls);

		// Rebuild path after store
		if ($result && $this->id) {
			$this->rebuildPath($this->id);
		}

		return $result;
	}

	/**
	 * Get the parent asset id for the record
	 *
	 * @param   \Joomla\CMS\Table\Table|null  $table  A Table object for the asset parent.
	 * @param   integer|null                    $id     The id for the asset
	 *
	 * @return  integer  The id of the asset's parent
	 *
	 * @since   1.0.0
	 */
	protected function _getAssetParentId($table = null, $id = null)
	{
		// No asset tree for component-specific table; fallback to parent
		return parent::_getAssetParentId($table, $id);
	}

	/**
	 * Ensure root node exists for given extension and return its id.
	 */
	private function getExtensionRootId(string $extension, int $userId, string $date): int
	{
		$db = $this->getDatabase();

		$query = $db->createQuery()
			->select($db->quoteName('id'))
			->from($db->quoteName($this->_tbl))
			->where($db->quoteName('parent_id') . ' = 0')
			->where($db->quoteName('extension') . ' = :extension')
			->bind(':extension', $extension);

		$db->setQuery($query);
		$rootId = (int) $db->loadResult();

		if ($rootId) {
			return $rootId;
		}

		// Create root node for this extension
		$root = new \stdClass();
		$root->title             = 'ROOT';
		$root->alias             = 'root';
		$root->path              = 'root';
		$root->extension         = $extension;
		$root->parent_id         = 0;
		$root->level             = 0;
		$root->lft               = 0;
		$root->rgt               = 1;
		$root->published         = 1;
		$root->language          = '*';
		$root->access            = 1;
		$root->params            = '{}';
		$root->metadata          = '{}';
		$root->created           = $date;
		$root->modified          = $date;
		$root->created_by        = $userId;
		$root->modified_by       = $userId;

		$db->insertObject($this->_tbl, $root);
		return (int) $db->insertid();
	}
}
