<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Model\Reviews;

use Joomla\CMS\MVC\Model\ListModel;
use Joomla\Database\ParameterType;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Reviews model class.
 *
 * @since  1.0.16
 */
class ReviewsModel extends ListModel
{
	/**
	 * Constructor.
	 *
	 * @param   array  $config  An optional associative array of configuration settings.
	 *
	 * @since   1.0.16
	 */
	public function __construct($config = [])
	{
		if (empty($config['filter_fields'])) {
			$config['filter_fields'] = [
				'id', 'a.id',
				'name', 'a.name',
				'email', 'a.email',
				'title', 'a.title',
				'rating', 'a.rating',
				'approved', 'a.approved',
				'published', 'a.published',
				'created', 'a.created',
				'content_id', 'a.content_id',
				'event_id', 'a.event_id',
			];
		}

		parent::__construct($config);
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * @param   string  $ordering   An optional ordering field.
	 * @param   string  $direction  An optional direction (asc|desc).
	 *
	 * @return  void
	 *
	 * @since   1.0.16
	 */
	protected function populateState($ordering = 'a.created', $direction = 'DESC')
	{
		parent::populateState($ordering, $direction);
	}

	/**
	 * Method to build an SQL query to load the list data.
	 *
	 * @return  \Joomla\Database\DatabaseQuery
	 *
	 * @since   1.0.16
	 */
	protected function getListQuery()
	{
		$db = $this->getDatabase();
		$query = $db->getQuery(true);

		$query->select(
			$this->getState(
				'list.select',
				'a.*'
			)
		)
			->from($db->quoteName('#__one_reviews', 'a'));

		// Filter by published
		$published = $this->getState('filter.published');
		if (is_numeric($published)) {
			$query->where($db->quoteName('a.published') . ' = :published')
				->bind(':published', $published, ParameterType::INTEGER);
		} elseif ($published === '') {
			$query->where('(' . $db->quoteName('a.published') . ' IN (0, 1))');
		}

		// Filter by approved
		$approved = $this->getState('filter.approved');
		if (is_numeric($approved)) {
			$query->where($db->quoteName('a.approved') . ' = :approved')
				->bind(':approved', $approved, ParameterType::INTEGER);
		}

		// Filter by type (content vs event)
		$type = $this->getState('filter.type');
		if ($type === 'content') {
			$query->where($db->quoteName('a.content_id') . ' > 0');
		} elseif ($type === 'event') {
			$query->where($db->quoteName('a.event_id') . ' > 0');
		}

		// Search
		$search = $this->getState('filter.search');
		if (!empty($search)) {
			$search = '%' . str_replace(' ', '%', $db->escape(trim($search), true)) . '%';
			$query->where(
				'(' . $db->quoteName('a.name') . ' LIKE :search1 OR ' .
				$db->quoteName('a.email') . ' LIKE :search2 OR ' .
				$db->quoteName('a.title') . ' LIKE :search3 OR ' .
				$db->quoteName('a.body') . ' LIKE :search4)'
			)
				->bind([':search1', ':search2', ':search3', ':search4'], $search);
		}

		$orderCol = $this->state->get('list.ordering', 'a.created');
		$orderDirn = $this->state->get('list.direction', 'DESC');
		$query->order($db->escape($orderCol) . ' ' . $db->escape($orderDirn));

		return $query;
	}
}
