<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Model;

use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use Joomla\CMS\Factory;
use Joomla\Database\ParameterType;
use Joomla\CMS\Application\ApplicationHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Import/Export model class.
 *
 * @since  1.0.13
 */
class ImportexportModel extends BaseDatabaseModel
{
	/**
	 * Export data to CSV format
	 *
	 * @param   string  $type  Type of data to export (content or events)
	 *
	 * @return  array  Array of CSV rows
	 *
	 * @since   1.0.13
	 */
	public function exportToCsv($type = 'content')
	{
		$db = $this->getDatabase();
		$rows = [];

		if ($type === 'content') {
			$rows = $this->exportContent($db);
		} elseif ($type === 'events') {
			$rows = $this->exportEvents($db);
		}

		return $rows;
	}

	/**
	 * Export content items to CSV
	 *
	 * @param   \Joomla\Database\DatabaseInterface  $db  Database object
	 *
	 * @return  array  Array of CSV rows
	 *
	 * @since   1.0.13
	 */
	protected function exportContent($db)
	{
		// Get all custom fields
		$customFields = $this->getCustomFields($db, 'content');

		// Build header row
		$headers = [
			'id',
			'title',
			'alias',
			'category_path',
			'introtext',
			'fulltext',
			'published',
			'featured',
			'access',
			'language',
			'created',
			'created_by',
			'modakey',
			'metadesc',
			'author',
			'robots',
			'xreference',
			'images',
			'video_link',
			'user',
			'publish_up',
			'publish_down',
		];

		// Add custom field columns
		foreach ($customFields as $field) {
			$headers[] = 'cf_' . $field->alias;
		}

		$rows = [$headers];

		// Get all content items
		$query = $db->getQuery(true)
			->select('c.*')
			->from($db->quoteName('#__one_content', 'c'))
			->order('c.id ASC');

		$items = $db->setQuery($query)->loadObjectList();

		// Get category paths for all items
		$categoryPaths = $this->getCategoryPathsForContent($db, array_column($items, 'id'));

		// Get custom field values for all items
		$customFieldValues = $this->getCustomFieldValues($db, 'content', array_column($items, 'id'), $customFields);

		foreach ($items as $item) {
			$row = [
				$item->id,
				$item->title,
				$item->alias,
				$categoryPaths[$item->id] ?? '',
				$item->introtext ?? '',
				$item->fulltext ?? '',
				$item->published ?? 0,
				$item->featured ?? 0,
				$item->access ?? 1,
				$item->language ?? '*',
				$item->created ?? '',
				$item->created_by ?? 0,
				$item->metakey ?? '',
				$item->metadesc ?? '',
				$item->author ?? '',
				$item->robots ?? '',
				$item->xreference ?? '',
				$item->images ?? '',
				$item->video_link ?? '',
				$item->user ?? 0,
				$item->publish_up ?? '',
				$item->publish_down ?? '',
			];

			// Add custom field values
			foreach ($customFields as $field) {
				$fieldId = $field->id;
				$value = $customFieldValues[$item->id][$fieldId] ?? '';
				
				// Handle multiple values (for multiselect, etc.)
				if (is_array($value)) {
					$value = implode('|', $value);
				}
				
				$row[] = $value;
			}

			$rows[] = $row;
		}

		return $rows;
	}

	/**
	 * Export events to CSV
	 *
	 * @param   \Joomla\Database\DatabaseInterface  $db  Database object
	 *
	 * @return  array  Array of CSV rows
	 *
	 * @since   1.0.13
	 */
	protected function exportEvents($db)
	{
		// Get all custom fields
		$customFields = $this->getCustomFields($db, 'events');

		// Build header row
		$headers = [
			'id',
			'title',
			'alias',
			'category_path',
			'description',
			'start_date',
			'end_date',
			'all_day',
			'published',
			'access',
			'language',
			'created',
			'created_by',
			'metakey',
			'metadesc',
			'author',
			'robots',
			'xreference',
			'images',
			'video_link',
			'publish_up',
			'publish_down',
		];

		// Add custom field columns
		foreach ($customFields as $field) {
			$headers[] = 'cf_' . $field->alias;
		}

		$rows = [$headers];

		// Get all events
		$query = $db->getQuery(true)
			->select('e.*')
			->from($db->quoteName('#__one_events', 'e'))
			->order('e.id ASC');

		$items = $db->setQuery($query)->loadObjectList();

		// Get category paths for all items
		$categoryPaths = $this->getCategoryPathsForEvents($db, array_column($items, 'id'));

		// Get custom field values for all items
		$customFieldValues = $this->getCustomFieldValues($db, 'events', array_column($items, 'id'), $customFields);

		foreach ($items as $item) {
			$row = [
				$item->id,
				$item->title,
				$item->alias,
				$categoryPaths[$item->id] ?? '',
				$item->description ?? '',
				$item->start_date ?? '',
				$item->end_date ?? '',
				$item->all_day ?? 0,
				$item->published ?? 0,
				$item->access ?? 1,
				$item->language ?? '*',
				$item->created ?? '',
				$item->created_by ?? 0,
				$item->metakey ?? '',
				$item->metadesc ?? '',
				$item->author ?? '',
				$item->robots ?? '',
				$item->xreference ?? '',
				$item->images ?? '',
				$item->video_link ?? '',
				$item->publish_up ?? '',
				$item->publish_down ?? '',
			];

			// Add custom field values
			foreach ($customFields as $field) {
				$fieldId = $field->id;
				$value = $customFieldValues[$item->id][$fieldId] ?? '';
				
				// Handle multiple values
				if (is_array($value)) {
					$value = implode('|', $value);
				}
				
				$row[] = $value;
			}

			$rows[] = $row;
		}

		return $rows;
	}

	/**
	 * Get custom fields for a given entity type
	 *
	 * @param   \Joomla\Database\DatabaseInterface  $db   Database object
	 * @param   string                                $type Entity type (content or events)
	 *
	 * @return  array  Array of custom field objects
	 *
	 * @since   1.0.13
	 */
	protected function getCustomFields($db, $type)
	{
		$query = $db->getQuery(true)
			->select('f.id, f.title, f.type, f.options')
			->from($db->quoteName('#__one_customfields', 'f'))
			->join('INNER', $db->quoteName('#__one_customfield_groups', 'g'), 'g.id = f.group_id')
			->where('f.published = 1')
			->where('g.published = 1')
			->where('(g.entity_type = ' . $db->quote($type) . ' OR g.entity_type IS NULL)')
			->order('f.ordering ASC');

		$fields = $db->setQuery($query)->loadObjectList();

		// Generate alias for each field (used as column name in CSV)
		foreach ($fields as $field) {
			// Use title to generate consistent alias
			$field->alias = ApplicationHelper::stringURLSafe($field->title);
		}

		return $fields;
	}

	/**
	 * Get category paths for content items
	 *
	 * @param   \Joomla\Database\DatabaseInterface  $db      Database object
	 * @param   array                                 $itemIds Array of item IDs
	 *
	 * @return  array  Associative array [item_id => category_path]
	 *
	 * @since   1.0.13
	 */
	protected function getCategoryPathsForContent($db, $itemIds)
	{
		if (empty($itemIds)) {
			return [];
		}

		$query = $db->getQuery(true)
			->select('cc.content_id, GROUP_CONCAT(c.path ORDER BY c.path SEPARATOR ", ") AS paths')
			->from($db->quoteName('#__one_content_categories', 'cc'))
			->join('LEFT', $db->quoteName('#__one_categories', 'c'), 'c.id = cc.category_id')
			->whereIn('cc.content_id', $itemIds)
			->group('cc.content_id');

		$results = $db->setQuery($query)->loadObjectList();

		$paths = [];
		foreach ($results as $result) {
			$paths[$result->content_id] = $result->paths;
		}

		return $paths;
	}

	/**
	 * Get category paths for events
	 *
	 * @param   \Joomla\Database\DatabaseInterface  $db      Database object
	 * @param   array                                 $itemIds Array of item IDs
	 *
	 * @return  array  Associative array [item_id => category_path]
	 *
	 * @since   1.0.13
	 */
	protected function getCategoryPathsForEvents($db, $itemIds)
	{
		if (empty($itemIds)) {
			return [];
		}

		$query = $db->getQuery(true)
			->select('ec.event_id, GROUP_CONCAT(c.path ORDER BY c.path SEPARATOR ", ") AS paths')
			->from($db->quoteName('#__one_event_categories', 'ec'))
			->join('LEFT', $db->quoteName('#__one_categories', 'c'), 'c.id = ec.category_id')
			->whereIn('ec.event_id', $itemIds)
			->group('ec.event_id');

		$results = $db->setQuery($query)->loadObjectList();

		$paths = [];
		foreach ($results as $result) {
			$paths[$result->event_id] = $result->paths;
		}

		return $paths;
	}

	/**
	 * Get custom field values for items
	 *
	 * @param   \Joomla\Database\DatabaseInterface  $db           Database object
	 * @param   string                                $entityType   Entity type (content or events)
	 * @param   array                                 $itemIds      Array of item IDs
	 * @param   array                                 $customFields Array of custom field objects
	 *
	 * @return  array  Associative array [item_id => [field_id => value]]
	 *
	 * @since   1.0.13
	 */
	protected function getCustomFieldValues($db, $entityType, $itemIds, $customFields)
	{
		if (empty($itemIds) || empty($customFields)) {
			return [];
		}

		$fieldIds = array_column($customFields, 'id');
		$itemIdColumn = $entityType === 'content' ? 'content_id' : 'event_id';

		$query = $db->getQuery(true)
			->select($itemIdColumn . ', field_id, value')
			->from($db->quoteName('#__one_customfield_values'))
			->whereIn($itemIdColumn, $itemIds)
			->whereIn('field_id', $fieldIds)
			->order($itemIdColumn . ', field_id');

		$results = $db->setQuery($query)->loadObjectList();

		$values = [];
		foreach ($results as $result) {
			$itemId = $result->$itemIdColumn;
			$fieldId = $result->field_id;

			if (!isset($values[$itemId])) {
				$values[$itemId] = [];
			}

			// Handle multiple values for same field
			if (isset($values[$itemId][$fieldId])) {
				if (!is_array($values[$itemId][$fieldId])) {
					$values[$itemId][$fieldId] = [$values[$itemId][$fieldId]];
				}
				$values[$itemId][$fieldId][] = $result->value;
			} else {
				$values[$itemId][$fieldId] = $result->value;
			}
		}

		return $values;
	}

	/**
	 * Import data from CSV file
	 *
	 * @param   string  $filePath  Path to CSV file
	 * @param   string  $type      Type of data to import (content or events)
	 *
	 * @return  array  Result array with success, imported, total, errors, message
	 *
	 * @since   1.0.13
	 */
	public function importFromCsv($filePath, $type = 'content')
	{
		$db = $this->getDatabase();
		$imported = 0;
		$errors = 0;
		$total = 0;

		try {
			// Read CSV file
			$handle = fopen($filePath, 'r');
			if ($handle === false) {
				return [
					'success' => false,
					'message' => 'Could not open CSV file',
					'imported' => 0,
					'total' => 0,
					'errors' => 0,
				];
			}

			// Read header row
			$headers = fgetcsv($handle);
			if ($headers === false) {
				fclose($handle);
				return [
					'success' => false,
					'message' => 'CSV file is empty or invalid',
					'imported' => 0,
					'total' => 0,
					'errors' => 0,
				];
			}

			// Remove BOM if present
			$headers[0] = preg_replace('/^\xEF\xBB\xBF/', '', $headers[0]);

			// Parse headers to identify custom fields
			$customFieldColumns = [];
			$baseColumns = [];
			foreach ($headers as $index => $header) {
				if (strpos($header, 'cf_') === 0) {
					$customFieldColumns[$index] = substr($header, 3); // Remove 'cf_' prefix
				} else {
					$baseColumns[$header] = $index;
				}
			}

			// Get custom fields mapping
			$customFields = $this->getCustomFields($db, $type);
			$customFieldMap = [];
			foreach ($customFields as $field) {
				$alias = ApplicationHelper::stringURLSafe($field->title);
				$customFieldMap[$alias] = $field;
			}

			// Process each row
			while (($row = fgetcsv($handle)) !== false) {
				$total++;

				try {
					if ($type === 'content') {
						$this->importContentRow($db, $row, $baseColumns, $customFieldColumns, $customFieldMap);
					} elseif ($type === 'events') {
						$this->importEventRow($db, $row, $baseColumns, $customFieldColumns, $customFieldMap);
					}
					$imported++;
				} catch (\Exception $e) {
					$errors++;
					Factory::getApplication()->enqueueMessage(
						'Row ' . $total . ': ' . $e->getMessage(),
						'warning'
					);
				}
			}

			fclose($handle);

			return [
				'success' => true,
				'message' => '',
				'imported' => $imported,
				'total' => $total,
				'errors' => $errors,
			];
		} catch (\Exception $e) {
			return [
				'success' => false,
				'message' => $e->getMessage(),
				'imported' => $imported,
				'total' => $total,
				'errors' => $errors,
			];
		}
	}

	/**
	 * Import a content row
	 *
	 * @param   \Joomla\Database\DatabaseInterface  $db                 Database object
	 * @param   array                                $row                CSV row data
	 * @param   array                                $baseColumns        Base column mapping
	 * @param   array                                $customFieldColumns Custom field column mapping
	 * @param   array                                $customFieldMap     Custom field map
	 *
	 * @return  void
	 *
	 * @since   1.0.13
	 */
	protected function importContentRow($db, $row, $baseColumns, $customFieldColumns, $customFieldMap)
	{
		// Prepare base data
		$data = [];
		foreach ($baseColumns as $column => $index) {
			if (isset($row[$index])) {
				$data[$column] = $row[$index];
			}
		}

		// Handle ID - if exists, update; otherwise insert
		$isUpdate = !empty($data['id']) && $data['id'] > 0;
		$itemId = $isUpdate ? (int) $data['id'] : null;

		// Generate alias if empty
		if (empty($data['alias']) && !empty($data['title'])) {
			$data['alias'] = ApplicationHelper::stringURLSafe($data['title']);
		}

		// Set defaults
		if (empty($data['published'])) {
			$data['published'] = 0;
		}
		if (empty($data['featured'])) {
			$data['featured'] = 0;
		}
		if (empty($data['access'])) {
			$data['access'] = 1;
		}
		if (empty($data['language'])) {
			$data['language'] = '*';
		}

		// Insert or update content
		if ($isUpdate) {
			$query = $db->getQuery(true)
				->update($db->quoteName('#__one_content'))
				->where($db->quoteName('id') . ' = ' . (int) $itemId);

			foreach ($data as $key => $value) {
				if ($key !== 'id' && $key !== 'category_path') {
					$query->set($db->quoteName($key) . ' = ' . $db->quote($value));
				}
			}

			$db->setQuery($query)->execute();
		} else {
			// Set created date if not provided
			if (empty($data['created'])) {
				$data['created'] = Factory::getDate()->toSql();
			}

			$query = $db->getQuery(true)
				->insert($db->quoteName('#__one_content'));

			$columns = [];
			$values = [];
			foreach ($data as $key => $value) {
				if ($key !== 'category_path') {
					$columns[] = $db->quoteName($key);
					$values[] = $db->quote($value);
				}
			}

			$query->columns($columns)->values(implode(',', $values));
			$db->setQuery($query)->execute();
			$itemId = $db->insertid();
		}

		// Handle categories
		if (!empty($data['category_path'])) {
			$this->importCategories($db, 'content', $itemId, $data['category_path']);
		}

		// Handle custom fields
		foreach ($customFieldColumns as $colIndex => $fieldAlias) {
			if (isset($row[$colIndex]) && !empty($row[$colIndex]) && isset($customFieldMap[$fieldAlias])) {
				$field = $customFieldMap[$fieldAlias];
				$value = $row[$colIndex];

				// Handle multiple values (pipe-separated)
				if (strpos($value, '|') !== false) {
					$values = explode('|', $value);
				} else {
					$values = [$value];
				}

				// Delete existing values
				$query = $db->getQuery(true)
					->delete($db->quoteName('#__one_customfield_values'))
					->where($db->quoteName('content_id') . ' = ' . (int) $itemId)
					->where($db->quoteName('field_id') . ' = ' . (int) $field->id);
				$db->setQuery($query)->execute();

				// Insert new values
				foreach ($values as $val) {
					$query = $db->getQuery(true)
						->insert($db->quoteName('#__one_customfield_values'))
						->columns([$db->quoteName('content_id'), $db->quoteName('field_id'), $db->quoteName('value')])
						->values((int) $itemId . ', ' . (int) $field->id . ', ' . $db->quote(trim($val)));
					$db->setQuery($query)->execute();
				}
			}
		}
	}

	/**
	 * Import an event row
	 *
	 * @param   \Joomla\Database\DatabaseInterface  $db                 Database object
	 * @param   array                                $row                CSV row data
	 * @param   array                                $baseColumns        Base column mapping
	 * @param   array                                $customFieldColumns Custom field column mapping
	 * @param   array                                $customFieldMap     Custom field map
	 *
	 * @return  void
	 *
	 * @since   1.0.13
	 */
	protected function importEventRow($db, $row, $baseColumns, $customFieldColumns, $customFieldMap)
	{
		// Prepare base data
		$data = [];
		foreach ($baseColumns as $column => $index) {
			if (isset($row[$index])) {
				$data[$column] = $row[$index];
			}
		}

		// Handle ID - if exists, update; otherwise insert
		$isUpdate = !empty($data['id']) && $data['id'] > 0;
		$itemId = $isUpdate ? (int) $data['id'] : null;

		// Generate alias if empty
		if (empty($data['alias']) && !empty($data['title'])) {
			$data['alias'] = ApplicationHelper::stringURLSafe($data['title']);
		}

		// Set defaults
		if (empty($data['published'])) {
			$data['published'] = 0;
		}
		if (empty($data['all_day'])) {
			$data['all_day'] = 0;
		}
		if (empty($data['access'])) {
			$data['access'] = 1;
		}
		if (empty($data['language'])) {
			$data['language'] = '*';
		}

		// Insert or update event
		if ($isUpdate) {
			$query = $db->getQuery(true)
				->update($db->quoteName('#__one_events'))
				->where($db->quoteName('id') . ' = ' . (int) $itemId);

			foreach ($data as $key => $value) {
				if ($key !== 'id' && $key !== 'category_path') {
					$query->set($db->quoteName($key) . ' = ' . $db->quote($value));
				}
			}

			$db->setQuery($query)->execute();
		} else {
			// Set created date if not provided
			if (empty($data['created'])) {
				$data['created'] = Factory::getDate()->toSql();
			}

			$query = $db->getQuery(true)
				->insert($db->quoteName('#__one_events'));

			$columns = [];
			$values = [];
			foreach ($data as $key => $value) {
				if ($key !== 'category_path') {
					$columns[] = $db->quoteName($key);
					$values[] = $db->quote($value);
				}
			}

			$query->columns($columns)->values(implode(',', $values));
			$db->setQuery($query)->execute();
			$itemId = $db->insertid();
		}

		// Handle categories
		if (!empty($data['category_path'])) {
			$this->importCategories($db, 'events', $itemId, $data['category_path']);
		}

		// Handle custom fields
		foreach ($customFieldColumns as $colIndex => $fieldAlias) {
			if (isset($row[$colIndex]) && !empty($row[$colIndex]) && isset($customFieldMap[$fieldAlias])) {
				$field = $customFieldMap[$fieldAlias];
				$value = $row[$colIndex];

				// Handle multiple values (pipe-separated)
				if (strpos($value, '|') !== false) {
					$values = explode('|', $value);
				} else {
					$values = [$value];
				}

				// Delete existing values
				$query = $db->getQuery(true)
					->delete($db->quoteName('#__one_customfield_values'))
					->where($db->quoteName('event_id') . ' = ' . (int) $itemId)
					->where($db->quoteName('field_id') . ' = ' . (int) $field->id);
				$db->setQuery($query)->execute();

				// Insert new values
				foreach ($values as $val) {
					$query = $db->getQuery(true)
						->insert($db->quoteName('#__one_customfield_values'))
						->columns([$db->quoteName('event_id'), $db->quoteName('field_id'), $db->quoteName('value')])
						->values((int) $itemId . ', ' . (int) $field->id . ', ' . $db->quote(trim($val)));
					$db->setQuery($query)->execute();
				}
			}
		}
	}

	/**
	 * Import categories from path string
	 *
	 * @param   \Joomla\Database\DatabaseInterface  $db         Database object
	 * @param   string                              $entityType Entity type (content or events)
	 * @param   int                                 $itemId     Item ID
	 * @param   string                              $paths      Comma-separated category paths
	 *
	 * @return  void
	 *
	 * @since   1.0.13
	 */
	protected function importCategories($db, $entityType, $itemId, $paths)
	{
		// Delete existing category associations
		$table = $entityType === 'content' ? '#__one_content_categories' : '#__one_event_categories';
		$idColumn = $entityType === 'content' ? 'content_id' : 'event_id';

		$query = $db->getQuery(true)
			->delete($db->quoteName($table))
			->where($db->quoteName($idColumn) . ' = ' . (int) $itemId);
		$db->setQuery($query)->execute();

		// Parse paths (comma-separated)
		$pathArray = array_map('trim', explode(',', $paths));

		foreach ($pathArray as $path) {
			$path = trim($path);
			if (empty($path)) {
				continue;
			}

			// Find category by path
			$query = $db->getQuery(true)
				->select('id')
				->from($db->quoteName('#__one_categories'))
				->where($db->quoteName('path') . ' = ' . $db->quote($path))
				->setLimit(1);

			$categoryId = $db->setQuery($query)->loadResult();

			if ($categoryId) {
				// Insert category association
				$query = $db->getQuery(true)
					->insert($db->quoteName($table))
					->columns([$db->quoteName($idColumn), $db->quoteName('category_id')])
					->values((int) $itemId . ', ' . (int) $categoryId);
				$db->setQuery($query)->execute();
			}
		}
	}
}
