<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Model\EmailLogs;

use Joomla\CMS\MVC\Model\ListModel;
use Joomla\Database\ParameterType;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Email Logs model class.
 *
 * @since  1.0.17
 */
class EmailLogsModel extends ListModel
{
	/**
	 * Constructor.
	 *
	 * @param   array  $config  An optional associative array of configuration settings.
	 *
	 * @since   1.0.17
	 */
	public function __construct($config = [])
	{
		if (empty($config['filter_fields'])) {
			$config['filter_fields'] = [
				'id', 'a.id',
				'template_id', 'a.template_id',
				'recipient_email', 'a.recipient_email',
				'recipient_name', 'a.recipient_name',
				'status', 'a.status',
				'sent_at', 'a.sent_at',
				'created', 'a.created',
				'template_title', 't.title',
			];
		}

		parent::__construct($config);
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * @param   string  $ordering   An optional ordering field.
	 * @param   string  $direction  An optional direction (asc|desc).
	 *
	 * @return  void
	 *
	 * @since   1.0.17
	 */
	protected function populateState($ordering = 'a.created', $direction = 'DESC')
	{
		parent::populateState($ordering, $direction);
	}

	/**
	 * Method to build an SQL query to load the list data.
	 *
	 * @return  \Joomla\Database\DatabaseQuery
	 *
	 * @since   1.0.17
	 */
	protected function getListQuery()
	{
		$db = $this->getDatabase();
		$query = $db->getQuery(true);

		$query->select(
			$this->getState(
				'list.select',
				'a.*, t.title AS template_title'
			)
		)
			->from($db->quoteName('#__one_email_logs', 'a'))
			->leftJoin(
				$db->quoteName('#__one_email_templ', 't'),
				$db->quoteName('t.id') . ' = ' . $db->quoteName('a.template_id')
			);

		// Filter by template
		$templateId = $this->getState('filter.template_id');
		if ($templateId !== null && $templateId !== '' && is_numeric($templateId)) {
			$query->where($db->quoteName('a.template_id') . ' = :templateId')
				->bind(':templateId', $templateId, ParameterType::INTEGER);
		}

		// Filter by status
		$status = $this->getState('filter.status');
		if ($status !== null && $status !== '') {
			$query->where($db->quoteName('a.status') . ' = :status')
				->bind(':status', $status);
		}

		// Search
		$search = $this->getState('filter.search');
		if (!empty($search)) {
			$search = '%' . str_replace(' ', '%', $db->escape(trim($search), true)) . '%';
			$query->where(
				'(' . $db->quoteName('a.recipient_email') . ' LIKE :search1 OR ' .
				$db->quoteName('a.recipient_name') . ' LIKE :search2 OR ' .
				$db->quoteName('a.subject') . ' LIKE :search3 OR ' .
				$db->quoteName('t.title') . ' LIKE :search4)'
			)
				->bind([':search1', ':search2', ':search3', ':search4'], $search);
		}

		$orderCol = $this->state->get('list.ordering', 'a.created');
		$orderDirn = $this->state->get('list.direction', 'DESC');
		$query->order($db->escape($orderCol) . ' ' . $db->escape($orderDirn));

		return $query;
	}
}
