<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Model\Customfields;

use Comdev\Component\Onecore\Administrator\Helper\LicenseHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Model\AdminModel;
use Joomla\CMS\Table\Table;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Customfield item model (AdminModel).
 *
 * @since  1.0.4
 */
class ItemModel extends AdminModel
{
	/**
	 * Ensure custom fields tables exist (runtime fallback).
	 *
	 * @return void
	 *
	 * @since  1.0.5
	 */
	private function ensureTables(): void
	{
		$db = $this->getDatabase();
		$prefix = $db->getPrefix();

		try {
			$existing = $db->getTableList();
		} catch (\Exception $e) {
			return;
		}

		$need = [
			$prefix . 'one_customfields',
			$prefix . 'one_customfield_values',
		];

		if (empty(array_diff($need, $existing))) {
			return;
		}

		$queries = [
			"CREATE TABLE IF NOT EXISTS `{$prefix}one_customfields` (
				`id` int NOT NULL AUTO_INCREMENT,
				`category_id` int NOT NULL DEFAULT 0,
				`title` varchar(255) NOT NULL DEFAULT '',
				`type` varchar(20) NOT NULL DEFAULT 'input',
				`required` tinyint NOT NULL DEFAULT 0,
				`height` int NOT NULL DEFAULT 0,
				`ordering` int NOT NULL DEFAULT 0,
				`display_label` tinyint NOT NULL DEFAULT 1,
				`label_position` varchar(10) NOT NULL DEFAULT 'next',
				`searchable` tinyint NOT NULL DEFAULT 0,
				`pos_items` varchar(20) NOT NULL DEFAULT 'none',
				`pos_item` varchar(20) NOT NULL DEFAULT 'none',
				`pos_module` varchar(20) NOT NULL DEFAULT 'none',
				`published` tinyint NOT NULL DEFAULT 1,
				`created` datetime NOT NULL,
				`modified` datetime,
				PRIMARY KEY (`id`),
				KEY `idx_published` (`published`),
				KEY `idx_category_id` (`category_id`),
				KEY `idx_ordering` (`ordering`),
				KEY `idx_searchable` (`searchable`),
				KEY `idx_type` (`type`)
			) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 DEFAULT COLLATE=utf8mb4_unicode_ci",
			"CREATE TABLE IF NOT EXISTS `{$prefix}one_customfield_values` (
				`id` int NOT NULL AUTO_INCREMENT,
				`content_id` int NOT NULL,
				`field_id` int NOT NULL,
				`value` mediumtext NOT NULL,
				`created` datetime NOT NULL,
				`modified` datetime,
				PRIMARY KEY (`id`),
				UNIQUE KEY `idx_content_field` (`content_id`, `field_id`),
				KEY `idx_field_id` (`field_id`)
			) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 DEFAULT COLLATE=utf8mb4_unicode_ci",
		];

		foreach ($queries as $sql) {
			try {
				$db->setQuery($sql)->execute();
			} catch (\Exception $e) {
				// ignore
			}
		}
	}

	/**
	 * The prefix to use with controller messages.
	 *
	 * @var    string
	 * @since  1.0.4
	 */
	protected $text_prefix = 'COM_ONECORE_CUSTOMFIELD';

	/**
	 * Method to get a table object, load it if necessary.
	 *
	 * @param   string  $name     The table name. Optional.
	 * @param   string  $prefix   The class prefix. Optional.
	 * @param   array   $options  Configuration array for model. Optional.
	 *
	 * @return  Table  A Table object
	 *
	 * @since   1.0.4
	 */
	public function getTable($name = 'Customfield', $prefix = 'Administrator', $options = [])
	{
		$this->ensureTables();
		return parent::getTable($name, $prefix, $options);
	}

	/**
	 * Method to get the record form.
	 *
	 * @param   array    $data      Data for the form.
	 * @param   boolean  $loadData  True if the form is to load its own data (default case), false if not.
	 *
	 * @return  \Joomla\CMS\Form\Form|boolean  A Form object on success, false on failure
	 *
	 * @since   1.0.4
	 */
	public function getForm($data = [], $loadData = true)
	{
		$this->ensureTables();

		// Ensure form paths are registered
		\Joomla\CMS\Form\Form::addFormPath(JPATH_ADMINISTRATOR . '/components/com_onecore/forms');

		$form = $this->loadForm(
			'com_onecore.customfield',
			'customfield',
			['control' => 'jform', 'load_data' => $loadData]
		);

		if (empty($form)) {
			return false;
		}

		return $form;
	}

	/**
	 * Auto-populate state.
	 *
	 * @return  void
	 *
	 * @since  1.0.4
	 */
	protected function populateState()
	{
		$app = Factory::getApplication();
		$pk = $app->getInput()->getInt('id');
		$this->setState($this->getName() . '.id', $pk);
	}

	/**
	 * Method to get the data that should be injected in the form.
	 *
	 * @return  mixed  The data for the form.
	 *
	 * @since   1.0.4
	 */
	protected function loadFormData()
	{
		$app = Factory::getApplication();
		$data = $app->getUserState('com_onecore.edit.customfield.data', []);

		if (empty($data)) {
			$data = $this->getItem();
		}

		return $data;
	}


	/**
	 * Method to validate the form data.
	 *
	 * @param   \Joomla\CMS\Form\Form  $form   The form to validate against.
	 * @param   array                   $data   The data to validate.
	 * @param   string                  $group  The name of the field group to validate.
	 *
	 * @return  array|boolean  Array of filtered data if valid, false otherwise.
	 *
	 * @since   1.0.10
	 */
	public function validate($form, $data, $group = null)
	{
		// First run standard validation
		$validated = parent::validate($form, $data, $group);

		if ($validated === false) {
			return false;
		}

		// Check if trying to save select or multiselect field type without license
		$fieldType = isset($validated['type']) ? (string) $validated['type'] : '';
		
		if ($fieldType === 'select') {
			$isLicenseActive = LicenseHelper::isLicenseActive();
			
			if (!$isLicenseActive) {
				$this->setError(Text::_('COM_ONECORE_CUSTOMFIELD_SELECT_LICENSE_REQUIRED'));
				return false;
			}
		}
		
		if ($fieldType === 'multiselect') {
			$isLicenseActive = LicenseHelper::isLicenseActive();
			
			if (!$isLicenseActive) {
				$this->setError(Text::_('COM_ONECORE_CUSTOMFIELD_MULTISELECT_LICENSE_REQUIRED'));
				return false;
			}
		}

		return $validated;
	}

	/**
	 * Prepare and sanitise the table prior to saving.
	 *
	 * @param   \Joomla\CMS\Table\Table  $table  The Table object
	 *
	 * @return  void
	 *
	 * @since   1.0.4
	 */
	protected function prepareTable($table)
	{
		$date = Factory::getDate()->toSql();

		if (empty($table->id)) {
			$table->created = $date;
		} else {
			$table->modified = $date;
		}

		// Normalize fields
		$table->required = (int) ($table->required ?? 0);
		$table->height = (int) ($table->height ?? 0);
		$table->ordering = (int) ($table->ordering ?? 0);
		$table->display_label = (int) ($table->display_label ?? 1);
		
		// Disable searchable for textarea type
		if (isset($table->type) && $table->type === 'textarea') {
			$table->searchable = 0;
		} else {
			$table->searchable = (int) ($table->searchable ?? 0);
		}
		
		$table->published = (int) ($table->published ?? 1);
	}
}

