<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Model\Content;

use Joomla\CMS\MVC\Model\AdminModel;
use Joomla\CMS\Table\Table;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Content item model class.
 *
 * @since  1.0.0
 */
class ItemModel extends AdminModel
{
	/**
	 * The model name (overrides auto-detection from class name)
	 * Must be 'item' for FormController to find ID in state
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	protected $name = 'item';

	/**
	 * The prefix to use with controller messages.
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	protected $text_prefix = 'COM_ONECORE_CONTENT';

	/**
	 * Ensure custom fields schema exists for admin-side usage.
	 *
	 * @return void
	 *
	 * @since  1.0.6
	 */
	private function ensureCustomFieldsSchema(): void
	{
		$db = $this->getDatabase();
		$prefix = $db->getPrefix();

		try {
			$tables = $db->getTableList();
		} catch (\Exception $e) {
			return;
		}

		// Create tables if missing (best-effort)
		if (!in_array($prefix . 'one_customfields', $tables, true)) {
			try {
				$db->setQuery(
					"CREATE TABLE IF NOT EXISTS `{$prefix}one_customfields` (
						`id` int NOT NULL AUTO_INCREMENT,
						`category_id` int NOT NULL DEFAULT 0,
						`title` varchar(255) NOT NULL DEFAULT '',
						`type` varchar(20) NOT NULL DEFAULT 'input',
						`required` tinyint NOT NULL DEFAULT 0,
						`height` int NOT NULL DEFAULT 0,
						`ordering` int NOT NULL DEFAULT 0,
						`display_label` tinyint NOT NULL DEFAULT 1,
						`label_position` varchar(10) NOT NULL DEFAULT 'next',
						`searchable` tinyint NOT NULL DEFAULT 0,
						`pos_items` varchar(20) NOT NULL DEFAULT 'none',
						`pos_item` varchar(20) NOT NULL DEFAULT 'none',
						`pos_module` varchar(20) NOT NULL DEFAULT 'none',
						`published` tinyint NOT NULL DEFAULT 1,
						`created` datetime NOT NULL,
						`modified` datetime,
						PRIMARY KEY (`id`),
						KEY `idx_category_id` (`category_id`),
						KEY `idx_published` (`published`),
						KEY `idx_ordering` (`ordering`),
						KEY `idx_searchable` (`searchable`),
						KEY `idx_type` (`type`)
					) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 DEFAULT COLLATE=utf8mb4_unicode_ci"
				)->execute();
			} catch (\Exception $e) {
				// ignore
			}
		}

		if (!in_array($prefix . 'one_customfield_values', $tables, true)) {
			try {
				$db->setQuery(
					"CREATE TABLE IF NOT EXISTS `{$prefix}one_customfield_values` (
						`id` int NOT NULL AUTO_INCREMENT,
						`content_id` int NOT NULL,
						`field_id` int NOT NULL,
						`value` mediumtext NOT NULL,
						`created` datetime NOT NULL,
						`modified` datetime,
						PRIMARY KEY (`id`),
						UNIQUE KEY `idx_content_field` (`content_id`, `field_id`),
						KEY `idx_field_id` (`field_id`)
					) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 DEFAULT COLLATE=utf8mb4_unicode_ci"
				)->execute();
			} catch (\Exception $e) {
				// ignore
			}
		}

		// Ensure columns exist even if tables already existed from older versions
		try {
			$cols = $db->getTableColumns($prefix . 'one_customfields', false);
			if (!isset($cols['category_id'])) {
				try {
					$db->setQuery("ALTER TABLE `{$prefix}one_customfields` ADD COLUMN `category_id` int NOT NULL DEFAULT 0 AFTER `id`")->execute();
				} catch (\Exception $e) {
					// ignore (e.g. duplicate column)
				}
			}
		} catch (\Exception $e) {
			// ignore
		}

		try {
			$keys = $db->getTableKeys($prefix . 'one_customfields');
			$hasKey = false;
			foreach ($keys as $key) {
				if (isset($key->Key_name) && $key->Key_name === 'idx_category_id') {
					$hasKey = true;
					break;
				}
			}
			if (!$hasKey) {
				try {
					$db->setQuery("ALTER TABLE `{$prefix}one_customfields` ADD KEY `idx_category_id` (`category_id`)")->execute();
				} catch (\Exception $e) {
					// ignore
				}
			}
		} catch (\Exception $e) {
			// ignore
		}
	}

	/**
	 * Method to get a table object, load it if necessary.
	 *
	 * @param   string  $name     The table name. Optional.
	 * @param   string  $prefix   The class prefix. Optional.
	 * @param   array   $options  Configuration array for model. Optional.
	 *
	 * @return  Table  A Table object
	 *
	 * @since   1.0.0
	 */
	public function getTable($name = 'Content', $prefix = 'Administrator', $options = [])
	{
		return parent::getTable($name, $prefix, $options);
	}

	/**
	 * Method to get the record form.
	 *
	 * @param   array    $data      Data for the form.
	 * @param   boolean  $loadData  True if the form is to load its own data (default case), false if not.
	 *
	 * @return  \Joomla\CMS\Form\Form|boolean  A Form object on success, false on failure
	 *
	 * @since   1.0.0
	 */
	public function getForm($data = [], $loadData = true)
	{
		// Ensure form paths are registered
		\Joomla\CMS\Form\Form::addFormPath(JPATH_ADMINISTRATOR . '/components/com_onecore/forms');
		
		$form = $this->loadForm(
			'com_onecore.content',
			'content',
			['control' => 'jform', 'load_data' => $loadData]
		);

		if (empty($form)) {
			// Try to reload form with explicit path
			$formFactory = \Joomla\CMS\Factory::getContainer()->get(\Joomla\CMS\Form\FormFactoryInterface::class);
			$form = $formFactory->createForm('com_onecore.content', ['control' => 'jform']);
			
			if ($form->loadFile('content', false)) {
				// Form loaded successfully
				if ($loadData) {
					$formData = $this->loadFormData();
					$form->bind($formData);
				}
			} else {
				// Form file not found, return false
				$this->setError('Form file content.xml not found');
				return false;
			}
		}

		return $form;
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * Note. Calling getState in this method will result in recursion.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	protected function populateState()
	{
		$app = \Joomla\CMS\Factory::getApplication();

		// Load the User state.
		$pk = $app->getInput()->getInt('id');
		$this->setState($this->getName() . '.id', $pk);
	}

	/**
	 * Method to get the data that should be injected in the form.
	 *
	 * @return  mixed  The data for the form.
	 *
	 * @since   1.0.0
	 */
	protected function loadFormData()
	{
		$app = \Joomla\CMS\Factory::getApplication();
		$data = $app->getUserState('com_onecore.edit.content.data', []);

		if (empty($data)) {
			$data = $this->getItem();
		}

		$this->preprocessData('com_onecore.content', $data);

		return $data;
	}

	/**
	 * Prepare and sanitise the table prior to saving.
	 *
	 * @param   \Joomla\CMS\Table\Table  $table  The Table object
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	protected function prepareTable($table)
	{
		$date = \Joomla\CMS\Factory::getDate()->toSql();
		$user = \Joomla\CMS\Factory::getApplication()->getIdentity();

		// Set created date if not set
		if (empty($table->created) || $table->created === '0000-00-00 00:00:00') {
			$table->created = $date;
		}

		// Set modified date
		if (property_exists($table, 'modified')) {
			$table->modified = $date;
		}

		// Set created_by if new record
		if (empty($table->id)) {
			if (property_exists($table, 'created_by') && empty($table->created_by)) {
				$table->created_by = $user->id;
			}
		} else {
			// Set modified_by for existing record
			if (property_exists($table, 'modified_by')) {
				$table->modified_by = $user->id;
			}
		}

		// Set default values for fields that don't have defaults in database
		if (property_exists($table, 'params') && empty($table->params)) {
			$table->params = '{}';
		}

		if (property_exists($table, 'metakey') && empty($table->metakey)) {
			$table->metakey = '';
		}

		if (property_exists($table, 'metadesc') && empty($table->metadesc)) {
			$table->metadesc = '';
		}

		if (property_exists($table, 'metadata') && empty($table->metadata)) {
			$table->metadata = '{}';
		}

		if (property_exists($table, 'ordering') && $table->ordering === null) {
			$table->ordering = 0;
		}

		if (property_exists($table, 'hits') && $table->hits === null) {
			$table->hits = 0;
		}

		if (property_exists($table, 'checked_out') && $table->checked_out === null) {
			$table->checked_out = 0;
		}

		if (property_exists($table, 'checked_out_time') && empty($table->checked_out_time)) {
			$table->checked_out_time = null;
		}

		if (property_exists($table, 'asset_id') && empty($table->asset_id)) {
			$table->asset_id = 0;
		}

		// Handle images field - convert array to JSON if needed (subform stores as JSON)
		if (property_exists($table, 'images')) {
			if (is_array($table->images)) {
				// Subform stores data as JSON array, so encode as JSON
				$table->images = json_encode($table->images);
			} elseif (is_string($table->images) && !empty($table->images)) {
				// If it's already a string, check if it's valid JSON
				$decoded = json_decode($table->images, true);
				if (json_last_error() !== JSON_ERROR_NONE) {
					// If not valid JSON, try to decode as Registry and convert to JSON
					try {
						$registry = new \Joomla\Registry\Registry($table->images);
						$imagesArray = $registry->toArray();
						// Convert Registry format to subform JSON format
						if (is_array($imagesArray)) {
							// If Registry has numeric keys, it's already in subform format
							$table->images = json_encode($imagesArray);
						} else {
							$table->images = '';
						}
					} catch (\Exception $e) {
						$table->images = '';
					}
				}
				// If it's valid JSON, keep it as is
			} elseif (empty($table->images)) {
				$table->images = '';
			}
		}

		// Handle video_link field
		if (property_exists($table, 'video_link') && empty($table->video_link)) {
			$table->video_link = '';
		}

		// Handle user field
		if (property_exists($table, 'user') && ($table->user === null || $table->user === '')) {
			$table->user = 0;
		}

		// Handle publish_up field
		if (property_exists($table, 'publish_up') && empty($table->publish_up)) {
			$table->publish_up = null;
		}

		// Handle publish_down field
		if (property_exists($table, 'publish_down') && empty($table->publish_down)) {
			$table->publish_down = null;
		}

		// Handle categories field - convert array to JSON if needed
		if (property_exists($table, 'categories')) {
			if (is_array($table->categories)) {
				$table->categories = json_encode($table->categories);
			} elseif (empty($table->categories)) {
				$table->categories = '[]';
			}
		}
	}

	/**
	 * Load published custom field definitions.
	 *
	 * @return  array  Array of field objects
	 *
	 * @since  1.0.4
	 */
	public function getCustomFieldDefinitions(): array
	{
		$this->ensureCustomFieldsSchema();

		$db = $this->getDatabase();

		// Get all published groups for content (entity_type='content' or NULL for backward compatibility)
		$groupQuery = $db->createQuery()
			->select($db->quoteName('id'))
			->from($db->quoteName('#__one_customfield_groups'))
			->where($db->quoteName('published') . ' = 1')
			->where('(' . $db->quoteName('entity_type') . ' = ' . $db->quote('content') . ' OR ' . $db->quoteName('entity_type') . ' IS NULL)');

		$db->setQuery($groupQuery);
		$groupIds = $db->loadColumn() ?: [];

		if (empty($groupIds)) {
			return [];
		}

		// Get all fields from all published groups, including group's category_id
		$query = $db->createQuery()
			->select([
				$db->quoteName('f.id'),
				$db->quoteName('f.group_id'),
				$db->quoteName('g.category_id', 'category_id'),
				$db->quoteName('f.title'),
				$db->quoteName('f.type'),
				$db->quoteName('f.required'),
				$db->quoteName('f.height'),
				$db->quoteName('f.options'),
				$db->quoteName('f.ordering'),
				$db->quoteName('f.display_label'),
				$db->quoteName('f.label_position'),
				$db->quoteName('f.searchable'),
				$db->quoteName('f.pos_items'),
				$db->quoteName('f.pos_item'),
				$db->quoteName('f.pos_module'),
				$db->quoteName('f.published'),
			])
			->from($db->quoteName('#__one_customfields', 'f'))
			->leftJoin(
				$db->quoteName('#__one_customfield_groups', 'g'),
				$db->quoteName('g.id') . ' = ' . $db->quoteName('f.group_id')
			)
			->where($db->quoteName('f.published') . ' = 1')
			->whereIn($db->quoteName('f.group_id'), $groupIds)
			->order($db->quoteName('f.group_id') . ' ASC, ' . $db->quoteName('f.ordering') . ' ASC, ' . $db->quoteName('f.id') . ' ASC');

		$db->setQuery($query);

		try {
			return $db->loadObjectList() ?: [];
		} catch (\Exception $e) {
			return [];
		}
	}

	/**
	 * Load published custom field definitions limited to selected categories.
	 *
	 * @param   array  $categoryIds  Category ids
	 *
	 * @return  array
	 *
	 * @since  1.0.5
	 */
	public function getCustomFieldDefinitionsForCategories(array $categoryIds): array
	{
		$this->ensureCustomFieldsSchema();

		$categoryIds = array_values(array_filter(array_map('intval', $categoryIds)));

		$db = $this->getDatabase();
		$groupIds = [];

		// Always get groups assigned to ROOT category (for content) - title='ROOT', entity_type='content' or NULL
		$rootGroupQuery = $db->createQuery()
			->select($db->quoteName('g.id'))
			->from($db->quoteName('#__one_customfield_groups', 'g'))
			->leftJoin(
				$db->quoteName('#__one_categories', 'gcat'),
				$db->quoteName('gcat.id') . ' = ' . $db->quoteName('g.category_id')
			)
			->where($db->quoteName('g.published') . ' = 1')
			->where('(' . $db->quoteName('g.entity_type') . ' = ' . $db->quote('content') . ' OR ' . $db->quoteName('g.entity_type') . ' IS NULL)')
			->where($db->quoteName('gcat.title') . ' = ' . $db->quote('ROOT'))
			->where($db->quoteName('gcat.extension') . ' = ' . $db->quote('com_onecore'));

		$db->setQuery($rootGroupQuery);
		try {
			$rootGroupIds = $db->loadColumn() ?: [];
			$groupIds = array_merge($groupIds, $rootGroupIds);
		} catch (\Exception $e) {
			// Ignore errors, continue with category-specific groups
		}

		// Get groups for selected categories (if any)
		if (!empty($categoryIds)) {
			$groupQuery = $db->createQuery()
				->select($db->quoteName('g.id'))
				->from($db->quoteName('#__one_customfield_groups', 'g'))
				->innerJoin(
					$db->quoteName('#__one_categories', 'csel'),
					$db->quoteName('csel.id') . ' IN (' . implode(',', $categoryIds) . ')'
				)
				->leftJoin(
					$db->quoteName('#__one_categories', 'gcat'),
					$db->quoteName('gcat.id') . ' = ' . $db->quoteName('g.category_id')
				)
				->where($db->quoteName('g.published') . ' = 1')
				->where('(' . $db->quoteName('g.entity_type') . ' = ' . $db->quote('content') . ' OR ' . $db->quoteName('g.entity_type') . ' IS NULL)')
				// Inheritance: group category is ancestor of selected category (or equals)
				// Exclude ROOT groups (title='ROOT') as they're already included above
				->where($db->quoteName('gcat.title') . ' != ' . $db->quote('ROOT'))
				->where('(' . $db->quoteName('gcat.lft') . ' <= ' . $db->quoteName('csel.lft') . ' AND ' . $db->quoteName('gcat.rgt') . ' >= ' . $db->quoteName('csel.rgt') . ')')
				->group($db->quoteName('g.id'));

			$db->setQuery($groupQuery);
			try {
				$categoryGroupIds = $db->loadColumn() ?: [];
				$groupIds = array_unique(array_merge($groupIds, $categoryGroupIds));
			} catch (\Exception $e) {
				// Ignore errors, use ROOT groups only
			}
		}

		// If no groups found, return empty array
		if (empty($groupIds)) {
			return [];
		}

		// Get all fields from the found groups, including group's category_id
		$fieldsQuery = $db->createQuery()
			->select([
				$db->quoteName('f.id'),
				$db->quoteName('f.group_id'),
				$db->quoteName('g.category_id', 'category_id'),
				$db->quoteName('f.title'),
				$db->quoteName('f.type'),
				$db->quoteName('f.required'),
				$db->quoteName('f.height'),
				$db->quoteName('f.options'),
				$db->quoteName('f.ordering'),
				$db->quoteName('f.display_label'),
				$db->quoteName('f.label_position'),
				$db->quoteName('f.searchable'),
				$db->quoteName('f.pos_items'),
				$db->quoteName('f.pos_item'),
				$db->quoteName('f.pos_module'),
				$db->quoteName('f.published'),
			])
			->from($db->quoteName('#__one_customfields', 'f'))
			->leftJoin(
				$db->quoteName('#__one_customfield_groups', 'g'),
				$db->quoteName('g.id') . ' = ' . $db->quoteName('f.group_id')
			)
			->where($db->quoteName('f.published') . ' = 1')
			->whereIn($db->quoteName('f.group_id'), $groupIds)
			->order($db->quoteName('f.group_id') . ' ASC, ' . $db->quoteName('f.ordering') . ' ASC, ' . $db->quoteName('f.id') . ' ASC');

		$db->setQuery($fieldsQuery);

		try {
			return $db->loadObjectList() ?: [];
		} catch (\Exception $e) {
			return [];
		}
	}

	/**
	 * Load custom field values for content item.
	 *
	 * @param   int  $contentId  Content id
	 *
	 * @return  array  field_id => value
	 *
	 * @since  1.0.4
	 */
	public function getCustomFieldValues(int $contentId): array
	{
		$this->ensureCustomFieldsSchema();

		if ($contentId <= 0) {
			return [];
		}

		$db = $this->getDatabase();
		$query = $db->createQuery()
			->select([$db->quoteName('field_id'), $db->quoteName('value')])
			->from($db->quoteName('#__one_customfield_values'))
			->where($db->quoteName('content_id') . ' = :contentId')
			->bind(':contentId', $contentId, \Joomla\Database\ParameterType::INTEGER);

		$db->setQuery($query);

		try {
			$rows = $db->loadObjectList() ?: [];
			$values = [];
			// Get field types to determine if value should be parsed as JSON
			$fieldTypes = [];
			if (!empty($rows)) {
				$fieldIds = array_map(function($row) { return (int) $row->field_id; }, $rows);
				$typeQuery = $db->createQuery()
					->select([$db->quoteName('id'), $db->quoteName('type')])
					->from($db->quoteName('#__one_customfields'))
					->whereIn($db->quoteName('id'), $fieldIds);
				$db->setQuery($typeQuery);
				$typeRows = $db->loadObjectList() ?: [];
				foreach ($typeRows as $typeRow) {
					$fieldTypes[(int) $typeRow->id] = (string) $typeRow->type;
				}
			}
			
			foreach ($rows as $row) {
				$fieldId = (int) $row->field_id;
				$fieldType = $fieldTypes[$fieldId] ?? '';
				$value = (string) $row->value;
				
				// For multiselect, parse JSON array
				if ($fieldType === 'multiselect' && !empty($value)) {
					$decoded = json_decode($value, true);
					$values[$fieldId] = is_array($decoded) ? $decoded : [$value];
				} else {
					$values[$fieldId] = $value;
				}
			}
			return $values;
		} catch (\Exception $e) {
			return [];
		}
	}

	/**
	 * Method to save the form data.
	 *
	 * @param   array  $data  The form data.
	 *
	 * @return  boolean  True on success, False on error.
	 *
	 * @since   1.0.0
	 */
	public function save($data)
	{
		$app = \Joomla\CMS\Factory::getApplication();
		$input = $app->getInput();

		// Extract custom fields (values) before saving
		$dataWithCustom = $data;
		$customValues = [];
		
		// Get values directly from input first (most reliable source)
		$inputCustomFields = $input->get('jform', [], 'array')['customfields'] ?? [];
		if (is_array($inputCustomFields) && !empty($inputCustomFields)) {
			// Normalize keys to integers for consistency
			foreach ($inputCustomFields as $k => $v) {
				$key = is_numeric($k) ? (int) $k : $k;
				// For multiselect, values come as arrays (from name="jform[customfields][fieldId][]")
				// Ensure we preserve arrays for multiselect fields
				if (is_array($v)) {
					// Filter out empty values
					$v = array_filter($v, function($item) {
						return $item !== '' && $item !== null;
					});
					// Only store if not empty
					if (!empty($v)) {
						$customValues[$key] = array_values($v); // Re-index array
					}
				} else {
					$customValues[$key] = $v;
				}
			}
		}
		
		// Fallback to $data if input doesn't have values
		if (empty($customValues) && isset($data['customfields']) && is_array($data['customfields'])) {
			foreach ($data['customfields'] as $k => $v) {
				$key = is_numeric($k) ? (int) $k : $k;
				if (!isset($customValues[$key])) {
					$customValues[$key] = $v;
				}
			}
		}
		
		// Remove from $data to avoid binding issues
		unset($data['customfields']);

		// Automatic handling of alias for empty fields (for both new and existing records)
		if (\in_array($input->get('task'), ['apply', 'save', 'save2new'])) {
			if (empty($data['alias']) || $data['alias'] == null || trim($data['alias']) == '') {
				if ($app->get('unicodeslugs') == 1) {
					$data['alias'] = \Joomla\CMS\Filter\OutputFilter::stringUrlUnicodeSlug($data['title']);
				} else {
					$data['alias'] = \Joomla\CMS\Filter\OutputFilter::stringURLSafe($data['title']);
				}

				$table = $this->getTable();

				// Check if alias already exists (excluding current record if editing)
				$existingId = isset($data['id']) ? (int) $data['id'] : 0;
				if ($table->load(['alias' => $data['alias']])) {
					if ($table->id != $existingId) {
						$msg = \Joomla\CMS\Language\Text::_('COM_ONECORE_SAVE_WARNING');
					}
				}

				[$title, $alias] = $this->generateNewTitle(0, $data['alias'], $data['title']);
				$data['alias'] = $alias;

				if (isset($msg)) {
					$app->enqueueMessage($msg, 'warning');
				}
			}
		}

		// Extract address data before saving
		$addressData = [];
		$addressFields = [
			'address',
			'address_street',
			'address_street_number',
			'address_postal_code',
			'address_city',
			'address_country',
			'latitude',
			'longitude'
		];
		
		foreach ($addressFields as $field) {
			if (isset($data[$field])) {
				$addressData[$field] = $data[$field];
				// Remove from $data to avoid binding issues (these fields don't exist in #__one_content anymore)
				unset($data[$field]);
			}
		}

		// Extract categories before saving
		$categories = [];
		if (isset($data['categories'])) {
			$categories = is_array($data['categories']) ? $data['categories'] : [$data['categories']];
			$categories = array_filter(array_map('intval', $categories));
			// Remove categories from data to avoid binding issues
			unset($data['categories']);
		}

		// Validate required custom fields (only for selected categories)
		$definitions = $this->getCustomFieldDefinitionsForCategories($categories);
		foreach ($definitions as $def) {
			if (!empty($def->required)) {
				$fieldId = (int) ($def->id ?? 0);
				if ($fieldId <= 0) {
					continue;
				}
				
				// Try both integer and string keys (form may send either)
				$val = $customValues[$fieldId] ?? ($customValues[(string) $fieldId] ?? null);
				
				// If value is not set at all, check if it might be in raw data or input
				if ($val === null && isset($dataWithCustom['customfields'][$fieldId])) {
					$val = $dataWithCustom['customfields'][$fieldId];
				}
				if ($val === null && isset($dataWithCustom['customfields'][(string) $fieldId])) {
					$val = $dataWithCustom['customfields'][(string) $fieldId];
				}
				if ($val === null) {
					// Last resort: check input directly
					$inputVal = $input->get('jform', [], 'array')['customfields'][$fieldId] ?? 
					            $input->get('jform', [], 'array')['customfields'][(string) $fieldId] ?? null;
					if ($inputVal !== null) {
						$val = $inputVal;
					}
				}
				
				// Get field type for proper validation
				$fieldType = $def->type ?? 'input';
				
				// Handle different value types
				// For multiselect, check if array is not empty
				if ($fieldType === 'multiselect') {
					if (is_array($val)) {
						// Filter out empty values
						$val = array_filter(array_map('trim', $val), function($item) {
							return $item !== '' && $item !== null;
						});
						// Check if array is empty after filtering
						if (empty($val)) {
							$fieldTitle = $def->title ?? 'Field ' . $fieldId;
							$app->enqueueMessage(\Joomla\CMS\Language\Text::sprintf('JLIB_FORM_VALIDATE_FIELD_REQUIRED', $fieldTitle), 'error');
							$app->setUserState('com_onecore.edit.content.data', $dataWithCustom);
							return false;
						}
						// Store multiselect values as JSON array (will be converted later during save)
						// Keep as array for now to preserve structure
					} else {
						// If not array and required, it's empty
						$fieldTitle = $def->title ?? 'Field ' . $fieldId;
						$app->enqueueMessage(\Joomla\CMS\Language\Text::sprintf('JLIB_FORM_VALIDATE_FIELD_REQUIRED', $fieldTitle), 'error');
						$app->setUserState('com_onecore.edit.content.data', $dataWithCustom);
						return false;
					}
				} else {
					// For other field types, handle as before
					if (is_array($val)) {
						// For other array values, join them
						$val = implode('', $val);
					} elseif ($val === null) {
						$val = '';
					} else {
						$val = (string) $val;
					}
					
					// Normalize: trim whitespace and check if truly empty
					$valTrimmed = trim($val);
					
					// Check if value is empty
					if ($valTrimmed === '') {
						$fieldTitle = $def->title ?? 'Field ' . $fieldId;
						$app->enqueueMessage(\Joomla\CMS\Language\Text::sprintf('JLIB_FORM_VALIDATE_FIELD_REQUIRED', $fieldTitle), 'error');
						$app->setUserState('com_onecore.edit.content.data', $dataWithCustom);
						return false;
					}
				}
			}
		}

		$isNew = empty($data['id']) || (int) $data['id'] === 0;
		
		// Get table BEFORE save to check ID after
		$table = $this->getTable();
		$key = $table->getKeyName();
		
		// Save the main content record
		$result = parent::save($data);

		if ($result) {
			// AdminModel::save() sets ID in state as getName() . '.id'
			// getName() now returns 'item' (not 'contentitem'), so state key is 'item.id'
			$contentId = (int) $this->getState($this->getName() . '.id');
			
			// Check if table has ID set (insertObject should set it)
			if (isset($table->$key) && (int) $table->$key > 0) {
				$contentId = (int) $table->$key;
				// Ensure state is set
				if ($contentId != $this->getState($this->getName() . '.id')) {
					$this->setState($this->getName() . '.id', $contentId);
				}
			}
			
			// If AdminModel didn't set ID, try to get it from table
			if ($contentId <= 0) {
				$contentId = (int) ($table->$key ?? 0);
				if ($contentId > 0) {
					$this->setState($this->getName() . '.id', $contentId);
				}
			}
			
			// Fallback: get from data (for existing records)
			if ($contentId <= 0) {
				$contentId = (int) ($dataWithCustom['id'] ?? $data['id'] ?? 0);
			}
			
			// Fallback for new records: query database directly for last insert ID
			if ($contentId <= 0 && $isNew) {
				$db = $this->getDatabase();
				// Try insertid() first
				$insertId = (int) $db->insertid();
				
				// If insertid() returns 0, try querying the table directly
				if ($insertId <= 0) {
					$query = $db->createQuery()
						->select('MAX(' . $db->quoteName($key) . ')')
						->from($db->quoteName($table->getTableName()));
					$db->setQuery($query);
					$maxId = (int) $db->loadResult();
					
					// If max ID is greater than what we had, use it
					if ($maxId > 0) {
						$insertId = $maxId;
					}
				}
				
				if ($insertId > 0) {
					$contentId = $insertId;
					// Update table and state
					$table->$key = $contentId;
					$this->setState($this->getName() . '.id', $contentId);
				}
			}
			
			if ($contentId > 0) {
				// Save categories to junction table
				$db = $this->getDatabase();
				
				// Delete existing category associations
				$query = $db->createQuery()
					->delete($db->quoteName('#__one_content_categories'))
					->where($db->quoteName('content_id') . ' = :contentId')
					->bind(':contentId', $contentId, \Joomla\Database\ParameterType::INTEGER);
				$db->setQuery($query);
				$db->execute();

				// Insert new category associations
				if (!empty($categories)) {
					foreach ($categories as $categoryId) {
						if ($categoryId > 0) {
							$query = $db->createQuery()
								->insert($db->quoteName('#__one_content_categories'))
								->columns([$db->quoteName('content_id'), $db->quoteName('category_id')])
								->values(':contentId, :categoryId')
								->bind(':contentId', $contentId, \Joomla\Database\ParameterType::INTEGER)
								->bind(':categoryId', $categoryId, \Joomla\Database\ParameterType::INTEGER);
							$db->setQuery($query);
							$db->execute();
						}
					}
				}

				// Save custom field values
				try {
					$this->ensureCustomFieldsSchema();
					$db = $this->getDatabase();

					// Remove existing values first
					$query = $db->createQuery()
						->delete($db->quoteName('#__one_customfield_values'))
						->where($db->quoteName('content_id') . ' = :contentId')
						->bind(':contentId', $contentId, \Joomla\Database\ParameterType::INTEGER);
					$db->setQuery($query);
					$db->execute();

					$now = \Joomla\CMS\Factory::getDate()->toSql();

					// Get all field IDs that were submitted from $customValues
					$submittedFieldIds = array_values(array_filter(array_map('intval', array_keys($customValues))));
					
					if (!empty($submittedFieldIds)) {
						// Get allowed groups first (ROOT + selected categories)
						$allowedGroupIds = [];
						
						// Always allow groups assigned to ROOT category (for content) - title='ROOT', entity_type='content' or NULL
						$rootGroupQuery = $db->createQuery()
							->select($db->quoteName('g.id'))
							->from($db->quoteName('#__one_customfield_groups', 'g'))
							->leftJoin(
								$db->quoteName('#__one_categories', 'gcat'),
								$db->quoteName('gcat.id') . ' = ' . $db->quoteName('g.category_id')
							)
							->where($db->quoteName('g.published') . ' = 1')
							->where('(' . $db->quoteName('g.entity_type') . ' = ' . $db->quote('content') . ' OR ' . $db->quoteName('g.entity_type') . ' IS NULL)')
							->where($db->quoteName('gcat.title') . ' = ' . $db->quote('ROOT'))
							->where($db->quoteName('gcat.extension') . ' = ' . $db->quote('com_onecore'));
						$db->setQuery($rootGroupQuery);
						$rootGroupIds = $db->loadColumn() ?: [];
						$allowedGroupIds = array_merge($allowedGroupIds, $rootGroupIds);
						
						// Also allow groups for selected categories (if any)
						if (!empty($categories)) {
							$groupQuery = $db->createQuery()
								->select($db->quoteName('g.id'))
								->from($db->quoteName('#__one_customfield_groups', 'g'))
								->innerJoin(
									$db->quoteName('#__one_categories', 'csel'),
									$db->quoteName('csel.id') . ' IN (' . implode(',', array_map('intval', $categories)) . ')'
								)
								->leftJoin(
									$db->quoteName('#__one_categories', 'gcat'),
									$db->quoteName('gcat.id') . ' = ' . $db->quoteName('g.category_id')
								)
								->where($db->quoteName('g.published') . ' = 1')
								->where('(' . $db->quoteName('g.entity_type') . ' = ' . $db->quote('content') . ' OR ' . $db->quoteName('g.entity_type') . ' IS NULL)')
								// Exclude ROOT groups (title='ROOT') as they're already included
								->where($db->quoteName('gcat.title') . ' != ' . $db->quote('ROOT'))
								->where($db->quoteName('gcat.lft') . ' <= ' . $db->quoteName('csel.lft'))
								->where($db->quoteName('gcat.rgt') . ' >= ' . $db->quoteName('csel.rgt'))
								->group($db->quoteName('g.id'));
							$db->setQuery($groupQuery);
							$categoryGroupIds = $db->loadColumn() ?: [];
							$allowedGroupIds = array_unique(array_merge($allowedGroupIds, $categoryGroupIds));
						}
						
						// Get all fields from allowed groups that were submitted
						if (!empty($allowedGroupIds)) {
							$fieldsQuery = $db->createQuery()
								->select($db->quoteName('f.id'))
								->from($db->quoteName('#__one_customfields', 'f'))
								->where($db->quoteName('f.published') . ' = 1')
								->whereIn($db->quoteName('f.group_id'), $allowedGroupIds)
								->whereIn($db->quoteName('f.id'), $submittedFieldIds);
							$db->setQuery($fieldsQuery);
							$allowedFieldIds = $db->loadColumn() ?: [];
						} else {
							$allowedFieldIds = [];
						}
					} else {
						// If no submitted fields, use definitions as fallback
						$allowedFieldIds = [];
						foreach ($definitions as $def) {
							$fid = (int) ($def->id ?? 0);
							if ($fid > 0) {
								$allowedFieldIds[] = $fid;
							}
						}
					}

					// Insert current values (skip empty) - only for allowed fields
					$inserted = 0;
					$skippedDueToCategory = 0;
					$skippedDueToEmpty = 0;
					$nonEmptyValuesCount = 0;
					
					foreach ($submittedFieldIds as $fieldId) {
						// Get value from $customValues (already normalized with integer keys)
						$val = $customValues[$fieldId] ?? null;
						
						// Fallback: try string key
						if ($val === null) {
							$val = $customValues[(string) $fieldId] ?? null;
						}
						
						// Get field type first to handle values correctly
						$fieldType = null;
						foreach ($definitions as $def) {
							if ((int) ($def->id ?? 0) === $fieldId) {
								$fieldType = $def->type ?? 'input';
								break;
							}
						}
						
						// If not found in definitions, query directly from DB
						if ($fieldType === null) {
							try {
								$q = $db->createQuery()
									->select($db->quoteName('type'))
									->from($db->quoteName('#__one_customfields'))
									->where($db->quoteName('id') . ' = :fieldId')
									->bind(':fieldId', $fieldId, \Joomla\Database\ParameterType::INTEGER);
								$db->setQuery($q);
								$fieldType = $db->loadResult() ?: 'input';
							} catch (\Exception $e) {
								$fieldType = 'input'; // Default fallback
							}
						}
						
						// Handle different value types based on field type
						$originalVal = $val;
						if (is_array($val)) {
							// For multiselect, store as JSON array
							if ($fieldType === 'multiselect') {
								$val = json_encode(array_filter(array_map('trim', $val)));
								// Check if there are non-empty values in the array
								$hasNonEmpty = !empty(array_filter(array_map('trim', $originalVal)));
								if ($hasNonEmpty) {
									$nonEmptyValuesCount++;
								}
							} else {
								// For other array values (e.g., textarea), join with newlines
								$val = implode("\n", $val);
								if (!empty(trim($val))) {
									$nonEmptyValuesCount++;
								}
							}
						} elseif ($val === null) {
							$val = '';
						} else {
							$val = (string) $val;
							if (!empty(trim($val))) {
								$nonEmptyValuesCount++;
							}
						}
						
						// Check if value is non-empty before checking category assignment
						$isNonEmpty = false;
						if ($fieldType === 'textarea') {
							$isNonEmpty = ($val !== '' && $val !== null);
						} elseif ($fieldType === 'multiselect') {
							$isNonEmpty = (!empty($originalVal) && is_array($originalVal) && !empty(array_filter(array_map('trim', $originalVal))));
						} else {
							$isNonEmpty = !empty(trim($val));
						}
						
						// Check if field is allowed for these categories
						if (!in_array($fieldId, $allowedFieldIds, true)) {
							if ($isNonEmpty) {
								$skippedDueToCategory++;
							}
							continue; // Skip fields not allowed for these categories
						}
						
						// Skip only if completely empty (not just whitespace for textarea)
						if ($fieldType === 'textarea') {
							// For textarea, allow whitespace and newlines - only skip if truly empty
							if ($val === '' || $val === null) {
								$skippedDueToEmpty++;
								continue;
							}
						} elseif ($fieldType === 'multiselect') {
							// For multiselect, check if array has non-empty values
							if (!is_array($originalVal) || empty(array_filter(array_map('trim', $originalVal)))) {
								$skippedDueToEmpty++;
								continue;
							}
						} else {
							// For input, trim and skip if empty after trimming
							if (trim($val) === '') {
								$skippedDueToEmpty++;
								continue;
							}
						}

						// Use raw SQL for NULL value to avoid binding issues
						$sql = 'INSERT INTO ' . $db->quoteName('#__one_customfield_values') . 
							' (' . $db->quoteName('content_id') . ', ' . 
							$db->quoteName('event_id') . ', ' . 
							$db->quoteName('field_id') . ', ' . 
							$db->quoteName('value') . ', ' . 
							$db->quoteName('created') . ') VALUES (' .
							(int) $contentId . ', ' .
							'NULL, ' .
							(int) $fieldId . ', ' .
							$db->quote($val) . ', ' .
							$db->quote($now) . ')';
						$db->setQuery($sql);
						$db->execute();
						$inserted++;
					}
					
					// Only show warning if non-empty values were submitted but not saved due to category mismatch
					if ($inserted === 0 && $nonEmptyValuesCount > 0 && $skippedDueToCategory > 0) {
						$app->enqueueMessage('Warning: Custom field values were submitted but none were saved. Check category assignments.', 'warning');
					}
				} catch (\Exception $e) {
					// Non-fatal: still consider main save successful
					$app->enqueueMessage('Custom field values could not be saved: ' . $e->getMessage(), 'warning');
				}

				// Save address data to separate table
				try {
					$db = $this->getDatabase();
					$now = \Joomla\CMS\Factory::getDate()->toSql();
					
					// Check if address record exists
					$query = $db->createQuery()
						->select($db->quoteName('id'))
						->from($db->quoteName('#__one_content_addresses'))
						->where($db->quoteName('content_id') . ' = :contentId')
						->bind(':contentId', $contentId, \Joomla\Database\ParameterType::INTEGER);
					$db->setQuery($query);
					$existingAddressId = $db->loadResult();
					
					// Prepare address data
					$latitude = null;
					$longitude = null;
					
					if (!empty($addressData['latitude'])) {
						$latitude = is_numeric($addressData['latitude']) ? (string) $addressData['latitude'] : null;
					}
					
					if (!empty($addressData['longitude'])) {
						$longitude = is_numeric($addressData['longitude']) ? (string) $addressData['longitude'] : null;
					}
					
					$addressRecord = [
						'content_id' => $contentId,
						'address' => $addressData['address'] ?? '',
						'address_street' => $addressData['address_street'] ?? '',
						'address_street_number' => $addressData['address_street_number'] ?? '',
						'address_postal_code' => $addressData['address_postal_code'] ?? '',
						'address_city' => $addressData['address_city'] ?? '',
						'address_country' => $addressData['address_country'] ?? '',
						'latitude' => $latitude,
						'longitude' => $longitude,
						'modified' => $now
					];
					
					if ($existingAddressId) {
						// Update existing record
						$query = $db->createQuery()
							->update($db->quoteName('#__one_content_addresses'))
							->set($db->quoteName('address') . ' = :address')
							->set($db->quoteName('address_street') . ' = :address_street')
							->set($db->quoteName('address_street_number') . ' = :address_street_number')
							->set($db->quoteName('address_postal_code') . ' = :address_postal_code')
							->set($db->quoteName('address_city') . ' = :address_city')
							->set($db->quoteName('address_country') . ' = :address_country')
							->set($db->quoteName('latitude') . ' = :latitude')
							->set($db->quoteName('longitude') . ' = :longitude')
							->set($db->quoteName('modified') . ' = :modified')
							->where($db->quoteName('id') . ' = :id')
							->bind(':address', $addressRecord['address'])
							->bind(':address_street', $addressRecord['address_street'])
							->bind(':address_street_number', $addressRecord['address_street_number'])
							->bind(':address_postal_code', $addressRecord['address_postal_code'])
							->bind(':address_city', $addressRecord['address_city'])
							->bind(':address_country', $addressRecord['address_country'])
							->bind(':latitude', $addressRecord['latitude'], \Joomla\Database\ParameterType::STRING)
							->bind(':longitude', $addressRecord['longitude'], \Joomla\Database\ParameterType::STRING)
							->bind(':modified', $addressRecord['modified'])
							->bind(':id', $existingAddressId, \Joomla\Database\ParameterType::INTEGER);
						$db->setQuery($query);
						$db->execute();
					} else {
						// Insert new record
						$addressRecord['created'] = $now;
						$query = $db->createQuery()
							->insert($db->quoteName('#__one_content_addresses'))
							->columns([
								$db->quoteName('content_id'),
								$db->quoteName('address'),
								$db->quoteName('address_street'),
								$db->quoteName('address_street_number'),
								$db->quoteName('address_postal_code'),
								$db->quoteName('address_city'),
								$db->quoteName('address_country'),
								$db->quoteName('latitude'),
								$db->quoteName('longitude'),
								$db->quoteName('created'),
								$db->quoteName('modified')
							])
							->values(':contentId, :address, :address_street, :address_street_number, :address_postal_code, :address_city, :address_country, :latitude, :longitude, :created, :modified')
							->bind(':contentId', $addressRecord['content_id'], \Joomla\Database\ParameterType::INTEGER)
							->bind(':address', $addressRecord['address'])
							->bind(':address_street', $addressRecord['address_street'])
							->bind(':address_street_number', $addressRecord['address_street_number'])
							->bind(':address_postal_code', $addressRecord['address_postal_code'])
							->bind(':address_city', $addressRecord['address_city'])
							->bind(':address_country', $addressRecord['address_country'])
							->bind(':latitude', $addressRecord['latitude'], \Joomla\Database\ParameterType::STRING)
							->bind(':longitude', $addressRecord['longitude'], \Joomla\Database\ParameterType::STRING)
							->bind(':created', $addressRecord['created'])
							->bind(':modified', $addressRecord['modified']);
						$db->setQuery($query);
						$db->execute();
					}
				} catch (\Exception $e) {
					// Non-fatal: still consider main save successful
					$app->enqueueMessage('Address data could not be saved: ' . $e->getMessage(), 'warning');
				}
			} else {
				// Debug: log if contentId is missing
				$app->enqueueMessage('Warning: Content ID could not be determined after save. Custom fields were not saved.', 'warning');
			}

			// Clear stale edit state to avoid showing old empty custom fields after successful save
			$app->setUserState('com_onecore.edit.content.data', null);
		}

		return $result;
	}

	/**
	 * Method to get a single record.
	 *
	 * @param   integer  $pk  The id of the primary key.
	 *
	 * @return  \stdClass|false  Object on success, false on failure.
	 *
	 * @since   1.0.0
	 */
	public function getItem($pk = null)
	{
		$item = parent::getItem($pk);

		if (!$item) {
			$item = (object) ['id' => 0, 'categories' => [], 'images' => []];
		} elseif ($item->id) {
			// Load categories from junction table
			$db = $this->getDatabase();
			$query = $db->createQuery()
				->select($db->quoteName('category_id'))
				->from($db->quoteName('#__one_content_categories'))
				->where($db->quoteName('content_id') . ' = :contentId')
				->bind(':contentId', $item->id, \Joomla\Database\ParameterType::INTEGER);
			$db->setQuery($query);
			$categories = $db->loadColumn();
			
			$item->categories = $categories ?: [];
			
			// Decode images field if it's a JSON string
			if (property_exists($item, 'images') && !empty($item->images) && is_string($item->images)) {
				$decoded = json_decode($item->images, true);
				if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
					$item->images = $decoded;
				} else {
					// Try Registry format
					try {
						$registry = new \Joomla\Registry\Registry($item->images);
						$item->images = $registry->toArray();
					} catch (\Exception $e) {
						$item->images = [];
					}
				}
			} elseif (empty($item->images)) {
				$item->images = [];
			}

			// Attach custom field values (used by the edit template)
			$item->customfield_values = $this->getCustomFieldValues((int) $item->id);
			
			// Load address data from separate table
			$db = $this->getDatabase();
			$query = $db->createQuery()
				->select([
					$db->quoteName('address'),
					$db->quoteName('address_street'),
					$db->quoteName('address_street_number'),
					$db->quoteName('address_postal_code'),
					$db->quoteName('address_city'),
					$db->quoteName('address_country'),
					$db->quoteName('latitude'),
					$db->quoteName('longitude')
				])
				->from($db->quoteName('#__one_content_addresses'))
				->where($db->quoteName('content_id') . ' = :contentId')
				->bind(':contentId', $item->id, \Joomla\Database\ParameterType::INTEGER);
			$db->setQuery($query);
			$addressData = $db->loadObject();
			
			if ($addressData) {
				$item->address = $addressData->address ?? '';
				$item->address_street = $addressData->address_street ?? '';
				$item->address_street_number = $addressData->address_street_number ?? '';
				$item->address_postal_code = $addressData->address_postal_code ?? '';
				$item->address_city = $addressData->address_city ?? '';
				$item->address_country = $addressData->address_country ?? '';
				$item->latitude = $addressData->latitude ?? '';
				$item->longitude = $addressData->longitude ?? '';
			} else {
				// Set empty defaults
				$item->address = '';
				$item->address_street = '';
				$item->address_street_number = '';
				$item->address_postal_code = '';
				$item->address_city = '';
				$item->address_country = '';
				$item->latitude = '';
				$item->longitude = '';
			}
		} else {
			// New record
			$item->categories = [];
			if (!property_exists($item, 'images') || empty($item->images)) {
				$item->images = [];
			}
			$item->customfield_values = [];
			// Set empty defaults for address fields
			$item->address = '';
			$item->address_street = '';
			$item->address_street_number = '';
			$item->address_postal_code = '';
			$item->address_city = '';
			$item->address_country = '';
			$item->latitude = '';
			$item->longitude = '';
		}

		return $item;
	}

	/**
	 * Method to change the title & alias.
	 *
	 * @param   integer  $parentId  The id of the parent (not used for content, kept for compatibility).
	 * @param   string   $alias     The alias.
	 * @param   string   $title     The title.
	 *
	 * @return  array  Contains the modified title and alias.
	 *
	 * @since   1.0.0
	 */
	protected function generateNewTitle($parentId, $alias, $title)
	{
		// Alter the title & alias
		$table = $this->getTable();

		while ($table->load(['alias' => $alias])) {
			if ($title == $table->title) {
				$title = \Joomla\String\StringHelper::increment($title);
			}

			$alias = \Joomla\String\StringHelper::increment($alias, 'dash');
		}

		return [$title, $alias];
	}
}

