<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Model\Content;

use Joomla\CMS\MVC\Model\ListModel;
use Joomla\Database\ParameterType;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Content model class.
 *
 * @since  1.0.0
 */
class ContentModel extends ListModel
{
	/**
	 * Constructor.
	 *
	 * @param   array  $config  An optional associative array of configuration settings.
	 *
	 * @since   1.0.0
	 */
	public function __construct($config = [])
	{
		if (empty($config['filter_fields'])) {
			$config['filter_fields'] = [
				'id', 'a.id',
				'title', 'a.title',
				'alias', 'a.alias',
				'published', 'a.published',
				'featured', 'a.featured',
				'ordering', 'a.ordering',
				'access', 'a.access',
				'created', 'a.created',
				'created_by', 'a.created_by',
				'user', 'a.user',
			];
		}

		parent::__construct($config);
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * @param   string  $ordering   An optional ordering field.
	 * @param   string  $direction  An optional direction (asc|desc).
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	protected function populateState($ordering = 'a.created', $direction = 'DESC')
	{
		parent::populateState($ordering, $direction);
	}

	/**
	 * Method to build an SQL query to load the list data.
	 *
	 * @return  \Joomla\Database\DatabaseQuery
	 *
	 * @since   1.0.0
	 */
	protected function getListQuery()
	{
		$db = $this->getDatabase();
		$query = $db->getQuery(true);

		// Subquery to get categories as comma-separated string
		$subQueryCategories = $db->createQuery()
			->select('GROUP_CONCAT(' . $db->quoteName('c.title') . ' SEPARATOR ' . $db->quote(', ') . ')')
			->from($db->quoteName('#__one_content_categories', 'cc'))
			->join('LEFT', $db->quoteName('#__one_categories', 'c'), $db->quoteName('c.id') . ' = ' . $db->quoteName('cc.category_id'))
			->where($db->quoteName('cc.content_id') . ' = ' . $db->quoteName('a.id'));

		// Subquery to get custom field groups as comma-separated string
		// Groups are linked to content through categories (one-to-many: content -> categories -> groups)
		$subQueryGroups = $db->createQuery()
			->select('GROUP_CONCAT(DISTINCT ' . $db->quoteName('g.title') . ' SEPARATOR ' . $db->quote(', ') . ')')
			->from($db->quoteName('#__one_content_categories', 'cc'))
			->join('LEFT', $db->quoteName('#__one_customfield_groups', 'g'), $db->quoteName('g.category_id') . ' = ' . $db->quoteName('cc.category_id'))
			->where($db->quoteName('cc.content_id') . ' = ' . $db->quoteName('a.id'))
			->where($db->quoteName('g.published') . ' = 1')
			->where('(' . $db->quoteName('g.entity_type') . ' = ' . $db->quote('content') . ' OR ' . $db->quoteName('g.entity_type') . ' IS NULL)');

		$query->select(
			$this->getState(
				'list.select',
				'a.*, (' . $subQueryCategories . ') AS categories, (' . $subQueryGroups . ') AS customfield_groups'
			)
		)
			->from($db->quoteName('#__one_content', 'a'));

		// Filter by published state
		$published = $this->getState('filter.published');

		if (is_numeric($published)) {
			$query->where($db->quoteName('a.published') . ' = :published')
				->bind(':published', $published, ParameterType::INTEGER);
		} elseif ($published === '') {
			$query->where('(' . $db->quoteName('a.published') . ' IN (0, 1))');
		}

		// Filter by search
		$search = $this->getState('filter.search');

		if (!empty($search)) {
			$search = '%' . str_replace(' ', '%', $db->escape(trim($search), true) . '%');
			$query->where(
				'(' . $db->quoteName('a.title') . ' LIKE :search1 OR ' .
				$db->quoteName('a.alias') . ' LIKE :search2)'
			)
				->bind([':search1', ':search2'], $search);
		}

		// Add the list ordering clause
		$orderCol = $this->state->get('list.ordering', 'a.ordering');
		$orderDirn = $this->state->get('list.direction', 'ASC');

		$query->order($db->escape($orderCol) . ' ' . $db->escape($orderDirn));

		return $query;
	}

	/**
	 * Method to get a list of items.
	 *
	 * @return  mixed  An array of objects on success, false on failure.
	 *
	 * @since   1.0.0
	 */
	public function getItems()
	{
		$items = parent::getItems();

		if ($items) {
			$db = $this->getDatabase();
			
			// Get ROOT category groups (always visible for all content)
			$rootGroupsQuery = $db->createQuery()
				->select($db->quoteName('g.title'))
				->from($db->quoteName('#__one_customfield_groups', 'g'))
				->leftJoin(
					$db->quoteName('#__one_categories', 'gcat'),
					$db->quoteName('gcat.id') . ' = ' . $db->quoteName('g.category_id')
				)
				->where($db->quoteName('g.published') . ' = 1')
				->where('(' . $db->quoteName('g.entity_type') . ' = ' . $db->quote('content') . ' OR ' . $db->quoteName('g.entity_type') . ' IS NULL)')
				->where($db->quoteName('gcat.title') . ' = ' . $db->quote('ROOT'))
				->where($db->quoteName('gcat.extension') . ' = ' . $db->quote('com_onecore'));
			$db->setQuery($rootGroupsQuery);
			$rootGroups = $db->loadColumn() ?: [];
			$rootGroupsStr = !empty($rootGroups) ? implode(', ', $rootGroups) : '';
			
			foreach ($items as $item) {
				// Extract first image from images field (JSON or Registry)
				$item->thumbnail = $this->getFirstImage($item->images ?? '');
				
				// Add ROOT groups to customfield_groups (always visible)
				$existingGroups = !empty($item->customfield_groups) ? $item->customfield_groups : '';
				if (!empty($rootGroupsStr)) {
					$item->customfield_groups = !empty($existingGroups) 
						? $existingGroups . ', ' . $rootGroupsStr 
						: $rootGroupsStr;
				} else {
					$item->customfield_groups = $existingGroups;
				}
			}
		}

		return $items;
	}

	/**
	 * Extract first image from images JSON field.
	 *
	 * @param   string  $imagesJson  JSON string with images data
	 *
	 * @return  string|null  First image path or null
	 *
	 * @since   1.0.0
	 */
	protected function getFirstImage($imagesJson)
	{
		if (empty($imagesJson)) {
			return null;
		}

		// Handle if it's already an array (decoded earlier)
		if (is_array($imagesJson)) {
			if (isset($imagesJson[0]) && is_array($imagesJson[0])) {
				// Subform array format: [{"image":"path"}, ...]
				foreach ($imagesJson as $imageData) {
					if (isset($imageData['image']) && !empty($imageData['image'])) {
						$imagePath = $imageData['image'];
						// Handle new Joomla 4+ format: path#joomlaImage://...
						if (strpos($imagePath, '#') !== false) {
							$imagePath = explode('#', $imagePath)[0];
						}
						return $imagePath;
					}
				}
			}
			if (isset($imagesJson['image']) && !empty($imagesJson['image'])) {
				$imagePath = $imagesJson['image'];
				// Handle new Joomla 4+ format: path#joomlaImage://...
				if (strpos($imagePath, '#') !== false) {
					$imagePath = explode('#', $imagePath)[0];
				}
				return $imagePath;
			}
			return null;
		}

		// If it's not a string, return null
		if (!is_string($imagesJson)) {
			return null;
		}

		// If it's already a string path (not JSON/Registry), return it
		$trimmed = trim($imagesJson);
		if (!empty($trimmed) && !preg_match('/^[\{\[]/', $trimmed)) {
			// Doesn't start with { or [, so it's likely a plain path
			// Handle new Joomla 4+ format: path#joomlaImage://...
			if (strpos($trimmed, '#') !== false) {
				$trimmed = explode('#', $trimmed)[0];
			}
			return $trimmed;
		}

		try {
			// First, try to decode as JSON array (subform format: [{"image":"path"}, ...])
			$images = json_decode($imagesJson, true);
			
			if (is_array($images) && json_last_error() === JSON_ERROR_NONE) {
				// If it's a subform array (repeatable-table format)
				if (isset($images[0]) && is_array($images[0])) {
					foreach ($images as $imageData) {
						if (isset($imageData['image']) && !empty($imageData['image'])) {
							$imagePath = $imageData['image'];
							// Handle new Joomla 4+ format: path#joomlaImage://...
							if (strpos($imagePath, '#') !== false) {
								$imagePath = explode('#', $imagePath)[0];
							}
							return $imagePath;
						}
					}
				}
				// If it's a flat array with 'image' key
				if (isset($images['image']) && !empty($images['image'])) {
					$imagePath = $images['image'];
					// Handle new Joomla 4+ format: path#joomlaImage://...
					if (strpos($imagePath, '#') !== false) {
						$imagePath = explode('#', $imagePath)[0];
					}
					return $imagePath;
				}
				// Check for any key containing 'image'
				foreach ($images as $key => $value) {
					if (is_string($key) && stripos($key, 'image') !== false && !empty($value) && is_string($value)) {
						$imagePath = $value;
						// Handle new Joomla 4+ format: path#joomlaImage://...
						if (strpos($imagePath, '#') !== false) {
							$imagePath = explode('#', $imagePath)[0];
						}
						return $imagePath;
					}
				}
			}
			
			// If JSON decode failed, try Registry format (INI-like format)
			try {
				$registry = new \Joomla\Registry\Registry($imagesJson);
				$images = $registry->toArray();
				
				if (is_array($images) && !empty($images)) {
					// Look for image fields in Registry format
					foreach ($images as $key => $value) {
						if (is_string($key) && stripos($key, 'image') !== false && !empty($value)) {
							// Check if it's an array with 'image' key
							if (is_array($value) && isset($value['image'])) {
								$imagePath = $value['image'];
								// Handle new Joomla 4+ format: path#joomlaImage://...
								if (strpos($imagePath, '#') !== false) {
									$imagePath = explode('#', $imagePath)[0];
								}
								return $imagePath;
							}
							// Or direct value
							if (is_string($value)) {
								$imagePath = $value;
								// Handle new Joomla 4+ format: path#joomlaImage://...
								if (strpos($imagePath, '#') !== false) {
									$imagePath = explode('#', $imagePath)[0];
								}
								return $imagePath;
							}
						}
					}
					
					// Also check for numeric keys (subform array in Registry)
					foreach ($images as $key => $value) {
						if (is_numeric($key) && is_array($value) && isset($value['image']) && !empty($value['image'])) {
							$imagePath = $value['image'];
							// Handle new Joomla 4+ format: path#joomlaImage://...
							if (strpos($imagePath, '#') !== false) {
								$imagePath = explode('#', $imagePath)[0];
							}
							return $imagePath;
						}
					}
					
					// Check if Registry has direct image path
					if (isset($images['image']) && !empty($images['image']) && is_string($images['image'])) {
						$imagePath = $images['image'];
						// Handle new Joomla 4+ format: path#joomlaImage://...
						if (strpos($imagePath, '#') !== false) {
							$imagePath = explode('#', $imagePath)[0];
						}
						return $imagePath;
					}
				}
			} catch (\Exception $e) {
				// Registry parsing failed, continue
			}
		} catch (\Exception $e) {
			// If parsing fails, return null
		}

		return null;
	}

	/**
	 * Method to test whether a record can have its state changed.
	 *
	 * @param   object  $record  A record object.
	 *
	 * @return  boolean  True if allowed to change the state of the record. Defaults to the permission for the component.
	 *
	 * @since   1.0.0
	 */
	protected function canEditState($record)
	{
		return $this->getCurrentUser()->authorise('core.edit.state', 'com_onecore');
	}

	/**
	 * Method to change the published state of one or more records.
	 *
	 * @param   array    &$pks   A list of the primary keys to change.
	 * @param   integer  $value  The value of the published state.
	 *
	 * @return  boolean  True on success.
	 *
	 * @since   1.0.0
	 */
	public function publish(&$pks, $value = 1)
	{
		$user = $this->getCurrentUser();
		$table = $this->getTable();
		$pks = (array) $pks;

		// Access checks
		foreach ($pks as $i => $pk) {
			$table->reset();

			if ($table->load($pk)) {
				if (!$this->canEditState($table)) {
					// Prune items that you can't change
					unset($pks[$i]);
					$this->setError(\Joomla\CMS\Language\Text::_('JLIB_APPLICATION_ERROR_EDITSTATE_NOT_PERMITTED'));
					return false;
				}

				// If the table is checked out by another user, drop it
				if ($table->hasField('checked_out') && $table->checked_out && ($table->checked_out != $user->id)) {
					// Prune items that you can't change
					unset($pks[$i]);
				}

				// Prune items that are already at the given state
				$publishedColumnName = $table->getColumnAlias('published');
				if (property_exists($table, $publishedColumnName) && ($table->$publishedColumnName ?? $value) == $value) {
					unset($pks[$i]);
				}
			}
		}

		// Check if there are items to change
		if (!\count($pks)) {
			return true;
		}

		// Attempt to change the state of the records
		if (!$table->publish($pks, $value, $user->id)) {
			$this->setError($table->getError());
			return false;
		}

		// Clear the component's cache
		$this->cleanCache();

		return true;
	}

	/**
	 * Method to toggle the featured setting of one or more records.
	 *
	 * @param   array    &$pks   A list of the primary keys to change.
	 * @param   integer  $value  The value of the featured state.
	 *
	 * @return  boolean  True on success.
	 *
	 * @since   1.0.0
	 */
	public function featured(&$pks, $value = 1)
	{
		$user = $this->getCurrentUser();
		$table = $this->getTable();
		$pks = (array) $pks;

		// Access checks
		foreach ($pks as $i => $pk) {
			$table->reset();

			if ($table->load($pk)) {
				if (!$this->canEditState($table)) {
					// Prune items that you can't change
					unset($pks[$i]);
					$this->setError(\Joomla\CMS\Language\Text::_('JLIB_APPLICATION_ERROR_EDITSTATE_NOT_PERMITTED'));
					return false;
				}

				// If the table is checked out by another user, drop it
				if ($table->hasField('checked_out') && $table->checked_out && ($table->checked_out != $user->id)) {
					// Prune items that you can't change
					unset($pks[$i]);
				}

				// Prune items that are already at the given state
				if (property_exists($table, 'featured') && ($table->featured ?? $value) == $value) {
					unset($pks[$i]);
				}
			}
		}

		// Check if there are items to change
		if (!\count($pks)) {
			return true;
		}

		// Update the featured state
		$db = $this->getDatabase();
		$query = $db->getQuery(true)
			->update($db->quoteName('#__one_content'))
			->set($db->quoteName('featured') . ' = :featured')
			->whereIn($db->quoteName('id'), $pks)
			->bind(':featured', $value, ParameterType::INTEGER);

		$db->setQuery($query);

		try {
			$db->execute();
		} catch (\RuntimeException $e) {
			$this->setError($e->getMessage());
			return false;
		}

		// Clear the component's cache
		$this->cleanCache();

		return true;
	}
}

