<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Model\Categories;

use Joomla\CMS\MVC\Model\AdminModel;
use Joomla\CMS\Table\Table;
use Joomla\Utilities\ArrayHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Category model class.
 *
 * @since  1.0.0
 */
class CategoryModel extends AdminModel
{
	/**
	 * The prefix to use with controller messages.
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	protected $text_prefix = 'COM_ONECORE_CATEGORY';

	/**
	 * Method to get a table object, load it if necessary.
	 *
	 * @param   string  $name     The table name. Optional.
	 * @param   string  $prefix   The class prefix. Optional.
	 * @param   array   $options  Configuration array for model. Optional.
	 *
	 * @return  Table  A Table object
	 *
	 * @since   1.0.0
	 */
	public function getTable($name = 'Category', $prefix = 'Administrator', $options = [])
	{
		try {
			return parent::getTable($name, $prefix, $options);
		} catch (\Exception $e) {
			// Try direct instantiation if parent fails
			$className = 'Comdev\\Component\\Onecore\\Administrator\\Table\\CategoryTable';
			if (class_exists($className)) {
				$db = $this->getDatabase();
				$table = new $className($db);
				return $table;
			}
			// Re-throw original exception if direct instantiation also fails
			throw $e;
		}
	}

	/**
	 * Method to get the record form.
	 *
	 * @param   array    $data      Data for the form.
	 * @param   boolean  $loadData  True if the form is to load its own data (default case), false if not.
	 *
	 * @return  \Joomla\CMS\Form\Form|boolean  A Form object on success, false on failure
	 *
	 * @since   1.0.0
	 */
	public function getForm($data = [], $loadData = true)
	{
		// Ensure form paths are registered
		\Joomla\CMS\Form\Form::addFormPath(JPATH_ADMINISTRATOR . '/components/com_onecore/forms');
		
		$form = $this->loadForm(
			'com_onecore.category',
			'category',
			['control' => 'jform', 'load_data' => $loadData]
		);

		if (empty($form)) {
			return false;
		}

		// Get the current category ID to exclude it from parent list
		$pk = (!empty($data['id'])) ? $data['id'] : (int) $this->getState($this->getName() . '.id');
		
		if ($pk > 0) {
			// Modify the parent_id field query to exclude current category
			$parentField = $form->getField('parent_id');
			if ($parentField && $parentField->getAttribute('type') === 'sql') {
				$db = $this->getDatabase();
				$query = $parentField->getAttribute('query');
				
				// Add WHERE clause to exclude current category
				if (strpos($query, 'WHERE') !== false) {
					$query .= ' AND ' . $db->quoteName('id') . ' != ' . (int) $pk;
				} else {
					$query .= ' WHERE ' . $db->quoteName('id') . ' != ' . (int) $pk;
				}
				
				// SqlField does not provide setAttribute; use setQuery if available, otherwise set the property directly
				if (method_exists($parentField, 'setQuery')) {
					$parentField->setQuery($query);
				} else {
					$parentField->query = $query;
				}
			}
		}

		return $form;
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * Note. Calling getState in this method will result in recursion.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	protected function populateState()
	{
		$app = \Joomla\CMS\Factory::getApplication();

		// Load the User state.
		$pk = $app->getInput()->getInt('id');
		$this->setState($this->getName() . '.id', $pk);

		// Set extension to com_onecore
		$this->setState('category.extension', 'com_onecore');
	}

	/**
	 * Method to get a single record.
	 *
	 * @param   integer  $pk  The id of the primary key.
	 *
	 * @return  \stdClass|false  Object on success, false on failure.
	 *
	 * @since   1.0.0
	 */
	public function getItem($pk = null)
	{
		$pk = (!empty($pk)) ? $pk : (int) $this->getState($this->getName() . '.id');
		$table = $this->getTable();

		if ($pk > 0) {
			// Attempt to load the row.
			$return = $table->load($pk);

			// Check for a table object error.
			if ($return === false) {
				// If there was no underlying error, then the false means there simply was not a row in the db for this $pk.
				if (!$table->getError()) {
					$this->setError(\Joomla\CMS\Language\Text::_('JLIB_APPLICATION_ERROR_NOT_EXIST'));
				} else {
					$this->setError($table->getError());
				}

				return false;
			}
		}

		// Convert to \stdClass before adding other data
		// Use getProperties() to ensure all table columns are included
		$properties = $table->getProperties();
		$item = ArrayHelper::toObject($properties);

		// Ensure params is properly handled if it exists
		if (property_exists($item, 'params') && !empty($item->params)) {
			if (is_string($item->params)) {
				$registry = new \Joomla\Registry\Registry($item->params);
				$item->params = $registry->toArray();
			}
		}

		return $item;
	}

	/**
	 * Method to get the data that should be injected in the form.
	 *
	 * @return  mixed  The data for the form.
	 *
	 * @since   1.0.0
	 */
	protected function loadFormData()
	{
		$app = \Joomla\CMS\Factory::getApplication();
		$data = $app->getUserState('com_onecore.edit.category.data', []);

		if (empty($data)) {
			$data = $this->getItem();
			
			// Ensure data is an object/array with proper structure
			if ($data && is_object($data)) {
				// Convert object to array for form binding
				$dataArray = (array) $data;
				// Ensure all expected fields exist
				if (!isset($dataArray['id'])) {
					$dataArray['id'] = 0;
				}
				// Convert back to object for form
				$data = (object) $dataArray;
			}
		}

		return $data;
	}

	/**
	 * Method to save the form data.
	 *
	 * @param   array  $data  The form data.
	 *
	 * @return  boolean  True on success, False on error.
	 *
	 * @since   1.0.0
	 */
	public function save($data)
	{
		$table = $this->getTable();
		$input = \Joomla\CMS\Factory::getApplication()->getInput();
		$pk = (!empty($data['id'])) ? $data['id'] : (int) $this->getState($this->getName() . '.id');
		$isNew = true;

		// Ensure extension is set
		if (!isset($data['extension'])) {
			$data['extension'] = 'com_onecore';
		}

		// Load the row if saving an existing category.
		if ($pk > 0) {
			$table->load($pk);
			$isNew = false;
		}

		// Set the new parent id if parent id not matched OR while New/Save as Copy.
		// This must be done BEFORE bind() to set the location in nested set
		$parentId = isset($data['parent_id']) ? (int) $data['parent_id'] : 0;
		
		// For new records or when parent_id changes, set location
		if ($isNew || ($pk > 0 && $table->parent_id != $parentId)) {
			// If parent_id is 0, we need to find or create root node
			if ($parentId == 0) {
				// Get root node ID for this extension
				$db = $this->getDatabase();
				$query = $db->createQuery()
					->select($db->quoteName('id'))
					->from($db->quoteName('#__one_categories'))
					->where($db->quoteName('parent_id') . ' = 0')
					->where($db->quoteName('extension') . ' = ' . $db->quote('com_onecore'))
					->setLimit(1);
				$db->setQuery($query);
				$rootId = (int) $db->loadResult();
				
				if ($rootId) {
					$table->setLocation($rootId, 'last-child');
				} else {
					// Root doesn't exist yet, it will be created in store()
					$table->setLocation(0, 'last-child');
				}
			} else {
				$table->setLocation($parentId, 'last-child');
			}
		}

		// Alter the title for save as copy
		if ($input->get('task') == 'save2copy') {
			$origTable = $this->getTable();
			$origTable->load($input->getInt('id'));

			if ($data['title'] == $origTable->title) {
				list($title, $alias) = $this->generateNewTitle($data['parent_id'], $data['alias'], $data['title']);
				$data['title'] = $title;
				$data['alias'] = $alias;
			} else {
				if ($data['alias'] == $origTable->alias) {
					$data['alias'] = '';
				}
			}

			$data['published'] = 0;
		}

		// Bind the data.
		if (!$table->bind($data)) {
			$this->setError($table->getError());
			return false;
		}

		// Bind the rules.
		if (isset($data['rules'])) {
			$rules = new \Joomla\CMS\Access\Rules($data['rules']);
			$table->setRules($rules);
		}

		// Prepare the table (sets created_time, modified_time, etc.)
		$this->prepareTable($table);

		// Check the data.
		if (!$table->check()) {
			$this->setError($table->getError());
			return false;
		}

		// Store the data.
		if (!$table->store()) {
			$this->setError($table->getError());
			return false;
		}

		// Set the state with the new id
		$this->setState($this->getName() . '.id', $table->id);

		$this->cleanCache();

		return true;
	}

	/**
	 * Prepare and sanitise the table prior to saving.
	 *
	 * @param   \Joomla\CMS\Table\Table  $table  The Table object
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	protected function prepareTable($table)
	{
		$date = \Joomla\CMS\Factory::getDate()->toSql();
		$user = \Joomla\CMS\Factory::getApplication()->getIdentity();

		// Ensure extension is set to com_onecore
		$table->extension = 'com_onecore';

		// Use component table fields (#__one_categories): created/modified
		if (empty($table->created) || $table->created === '0000-00-00 00:00:00') {
			$table->created = $date;
		}

		// Set modified
		$table->modified = $date;

		// Set created_user_id if new record
		if (empty($table->id)) {
			if (empty($table->created_by)) {
				$table->created_by = $user->id;
			}
			if (empty($table->modified_by)) {
				$table->modified_by = $user->id;
			}
		} else {
			// Set modified_user_id for existing record
			$table->modified_by = $user->id;
		}

		// Set default values for fields that don't have defaults in database
		if (empty($table->params)) {
			$table->params = '{}';
		}

		if (empty($table->metakey)) {
			$table->metakey = '';
		}

		if (empty($table->metadesc)) {
			$table->metadesc = '';
		}

		if (empty($table->metadata)) {
			$table->metadata = '{}';
		}

		if (empty($table->description)) {
			$table->description = '';
		}

		// Ensure params is a JSON string if it's an array
		if (is_array($table->params)) {
			$table->params = json_encode($table->params);
		}

		// Ensure metadata is a JSON string if it's an array
		if (is_array($table->metadata)) {
			$table->metadata = json_encode($table->metadata);
		}
	}
}

