<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Model\Categories;

use Joomla\CMS\MVC\Model\ListModel;
use Joomla\Database\ParameterType;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Categories model class.
 *
 * @since  1.0.0
 */
class CategoriesModel extends ListModel
{
	/**
	 * Constructor.
	 *
	 * @param   array  $config  An optional associative array of configuration settings.
	 *
	 * @since   1.0.0
	 */
	public function __construct($config = [])
	{
		if (empty($config['filter_fields'])) {
			$config['filter_fields'] = [
				'id', 'a.id',
				'title', 'a.title',
				'alias', 'a.alias',
				'published', 'a.published',
				'parent_id', 'a.parent_id',
				'parent_title', 'p.title',
				'ordering', 'a.ordering',
				'access', 'a.access',
				'created', 'a.created',
				'created_by', 'a.created_by',
				'lft', 'a.lft',
				'rgt', 'a.rgt',
				'level', 'a.level',
			];
		}

		parent::__construct($config);
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * Note. Calling getState in this method will result in recursion.
	 *
	 * @param   string  $ordering   An optional ordering field.
	 * @param   string  $direction  An optional direction (asc|desc).
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	protected function populateState($ordering = 'a.lft', $direction = 'ASC')
	{
		$app = \Joomla\CMS\Factory::getApplication();

		// Adjust the context to support modal layouts
		if ($layout = $app->getInput()->get('layout')) {
			$this->context .= '.' . $layout;
		}

		// Filter by extension
		$extension = $app->getUserStateFromRequest($this->context . '.filter.extension', 'filter_extension', 'com_onecore', 'string');
		$this->setState('filter.extension', $extension);

		// Show full tree: no pagination (list.limit = 0 loads all categories)
		$this->setState('list.limit', 0);

		// List state information - parent handles filter array from request
		parent::populateState($ordering, $direction);
	}

	/**
	 * Method to build an SQL query to load the list data.
	 *
	 * @return  \Joomla\Database\DatabaseQuery
	 *
	 * @since   1.0.0
	 */
	protected function getListQuery()
	{
		$db = $this->getDatabase();
		$query = $db->getQuery(true);

		$query->select(
			$this->getState(
				'list.select',
				'a.*, p.title AS parent_title'
			)
		)
			->from($db->quoteName('#__one_categories', 'a'))
			->join('LEFT', $db->quoteName('#__one_categories', 'p') . ' ON ' . $db->quoteName('p.id') . ' = ' . $db->quoteName('a.parent_id'));

		// Filter by extension
		$extension = $this->getState('filter.extension', 'com_onecore');
		$query->where($db->quoteName('a.extension') . ' = :extension')
			->bind(':extension', $extension);

		// Filter by published state
		$published = $this->getState('filter.published');

		if (is_numeric($published)) {
			$query->where($db->quoteName('a.published') . ' = :published')
				->bind(':published', $published, ParameterType::INTEGER);
		}
		// If published filter is not set, show all categories (no WHERE clause)

		// Filter by search
		$search = $this->getState('filter.search');

		if (!empty($search)) {
			$search = '%' . str_replace(' ', '%', $db->escape(trim($search), true) . '%');
			$query->where(
				'(' . $db->quoteName('a.title') . ' LIKE :search1 OR ' .
				$db->quoteName('a.alias') . ' LIKE :search2)'
			)
				->bind([':search1', ':search2'], $search);
		}

		// Add the list ordering clause - use lft for nested set ordering
		$orderCol = $this->state->get('list.ordering', 'a.lft');
		$orderDirn = $this->state->get('list.direction', 'ASC');

		if ($orderCol) {
			$query->order($db->escape($orderCol) . ' ' . $db->escape($orderDirn));
		}

		return $query;
	}

	/**
	 * Method to get an array of data items.
	 *
	 * @return  mixed  An array of data items on success, false on failure.
	 *
	 * @since   1.0.0
	 */
	public function getItems()
	{
		$items = parent::getItems();

		if ($items) {
			// Ensure level is set from database (it should be already if lft/rgt/level columns exist)
			foreach ($items as $item) {
				// If level is not set, calculate it from parent_id chain
				if (!isset($item->level) || $item->level === null) {
					$level = 0;
					$currentParentId = $item->parent_id;
					$visited = [];

					// Build a map for quick lookup
					$itemsMap = [];
					foreach ($items as $it) {
						$itemsMap[$it->id] = $it;
					}

					// Traverse up the parent chain to calculate level
					while ($currentParentId > 0 && !isset($visited[$currentParentId])) {
						$visited[$currentParentId] = true;
						$level++;
						
						if (isset($itemsMap[$currentParentId])) {
							$currentParentId = $itemsMap[$currentParentId]->parent_id;
						} else {
							// Parent not in current result set
							break;
						}
					}

					$item->level = $level;
				}
			}
		}

		return $items;
	}

	/**
	 * Method to get a table object, load it if necessary.
	 *
	 * @param   string  $name     The table name. Optional.
	 * @param   string  $prefix   The class prefix. Optional.
	 * @param   array   $options  Configuration array for model. Optional.
	 *
	 * @return  \Joomla\CMS\Table\Table  A Table object
	 *
	 * @since   1.0.0
	 */
	public function getTable($name = 'Category', $prefix = 'Administrator', $options = [])
	{
		try {
			return parent::getTable($name, $prefix, $options);
		} catch (\Exception $e) {
			// Try direct instantiation if parent fails
			$className = 'Comdev\\Component\\Onecore\\Administrator\\Table\\CategoryTable';
			if (class_exists($className)) {
				$db = $this->getDatabase();
				$table = new $className($db);
				return $table;
			}
			// Re-throw original exception if direct instantiation also fails
			throw $e;
		}
	}

	/**
	 * Method to test whether a record can have its state changed.
	 *
	 * @param   object  $record  A record object.
	 *
	 * @return  boolean  True if allowed to change the state of the record. Defaults to the permission for the component.
	 *
	 * @since   1.0.0
	 */
	protected function canEditState($record)
	{
		return $this->getCurrentUser()->authorise('core.edit.state', 'com_onecore');
	}

	/**
	 * Method to change the published state of one or more records.
	 *
	 * @param   array    &$pks   A list of the primary keys to change.
	 * @param   integer  $value  The value of the published state.
	 *
	 * @return  boolean  True on success.
	 *
	 * @since   1.0.0
	 */
	public function publish(&$pks, $value = 1)
	{
		$user = $this->getCurrentUser();
		$table = $this->getTable();
		$pks = (array) $pks;

		// Access checks
		foreach ($pks as $i => $pk) {
			$table->reset();

			if ($table->load($pk)) {
				if (!$this->canEditState($table)) {
					// Prune items that you can't change
					unset($pks[$i]);
					$this->setError(\Joomla\CMS\Language\Text::_('JLIB_APPLICATION_ERROR_EDITSTATE_NOT_PERMITTED'));
					return false;
				}

				// If the table is checked out by another user, drop it
				if ($table->hasField('checked_out') && $table->checked_out && ($table->checked_out != $user->id)) {
					// Prune items that you can't change
					unset($pks[$i]);
				}

				// Prune items that are already at the given state
				$publishedColumnName = $table->getColumnAlias('published');
				if (property_exists($table, $publishedColumnName) && ($table->$publishedColumnName ?? $value) == $value) {
					unset($pks[$i]);
				}
			}
		}

		// Check if there are items to change
		if (!\count($pks)) {
			return true;
		}

		// Attempt to change the state of the records
		if (!$table->publish($pks, $value, $user->id)) {
			$this->setError($table->getError());
			return false;
		}

		// Clear the component's cache
		$this->cleanCache();

		return true;
	}
}

