<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Helper;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\Uri\Uri;
use Joomla\Registry\Registry;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * License helper class for OneCore component.
 *
 * @since  1.0.0
 */
class LicenseHelper
{
	/**
	 * WooCommerce API Manager endpoint URL for status check
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	private const API_ENDPOINT_STATUS = 'https://comdev.eu/wc-api/am/v2/license/status';
	
	/**
	 * WooCommerce API Manager base endpoint URL
	 *
	 * @var    string
	 * @since  1.0.10
	 */
	private const API_ENDPOINT_BASE = 'https://comdev.eu/wc-api';
	
	/**
	 * WooCommerce API Manager Consumer Key
	 *
	 * @var    string
	 * @since  1.0.10
	 */
	private const WC_AM_CONSUMER_KEY = 'ck_22505b7d767dc7a383b9859796e098a58007dd4b';
	
	/**
	 * WooCommerce API Manager Consumer Secret
	 *
	 * @var    string
	 * @since  1.0.10
	 */
	private const WC_AM_CONSUMER_SECRET = 'cs_1efc5a909eb5c5a4efb5151dd1a4405940c99015';
	
	/**
	 * WooCommerce API Manager Product IDs
	 * Multiple product IDs are supported for different license versions
	 *
	 * @var    array
	 * @since  1.0.12
	 */
	private const WC_AM_PRODUCT_IDS = [4044, 4054, 4085];

	/**
	 * Cache lifetime for license status (in seconds)
	 *
	 * @var    int
	 * @since  1.0.0
	 */
	private const CACHE_LIFETIME = 3600; // 1 hour

	/**
	 * Check if development mode is enabled
	 * Development mode bypasses license validation
	 *
	 * @return  bool  True if development mode is enabled
	 *
	 * @since   1.0.10
	 */
	private static function isDevelopmentMode(): bool
	{
		// Check environment variable via getenv()
		$devMode = getenv('ONECORE_DEV_MODE');
		if ($devMode === '1' || $devMode === 'true' || $devMode === 'on') {
			return true;
		}
		
		// Check $_ENV array (some PHP configurations)
		if (isset($_ENV['ONECORE_DEV_MODE'])) {
			$devMode = $_ENV['ONECORE_DEV_MODE'];
			if ($devMode === '1' || $devMode === 'true' || $devMode === 'on' || $devMode === 1 || $devMode === true) {
				return true;
			}
		}
		
		// Check $_SERVER array (fallback)
		if (isset($_SERVER['ONECORE_DEV_MODE'])) {
			$devMode = $_SERVER['ONECORE_DEV_MODE'];
			if ($devMode === '1' || $devMode === 'true' || $devMode === 'on' || $devMode === 1 || $devMode === true) {
				return true;
			}
		}
		
		// Check Joomla debug mode with additional flag
		$config = Factory::getConfig();
		if ($config->get('debug') == 1) {
			// In debug mode, check if ONECORE_DEV_LICENSE is set
			$devLicense = getenv('ONECORE_DEV_LICENSE');
			if ($devLicense === '1' || $devLicense === 'true' || $devLicense === 'on') {
				return true;
			}
			if (isset($_ENV['ONECORE_DEV_LICENSE']) && ($_ENV['ONECORE_DEV_LICENSE'] === '1' || $_ENV['ONECORE_DEV_LICENSE'] === 'true')) {
				return true;
			}
		}
		
		return false;
	}

	/**
	 * Check if license is active
	 *
	 * @return  bool  True if license is active, false otherwise
	 *
	 * @since   1.0.0
	 */
	public static function isLicenseActive(): bool
	{
		// Development mode: bypass license validation
		if (self::isDevelopmentMode()) {
			return true;
		}
		
		$params = ComponentHelper::getParams('com_onecore');
		$licenseKey = $params->get('license_key', '');
		$licenseStatus = $params->get('license_status', 'inactive');
		$licenseLastCheck = $params->get('license_last_check', 0);

		// If no license key, return false
		if (empty($licenseKey)) {
			return false;
		}

		// Check cache - if checked recently, return cached status
		$now = time();
		if ($licenseLastCheck > 0 && ($now - $licenseLastCheck) < self::CACHE_LIFETIME) {
			return $licenseStatus === 'active';
		}

		// Perform fresh license check
		$status = self::checkLicenseStatus($licenseKey);

		// If we already had active and API returned inactive, don't overwrite unless we're sure
		// (API might use different instance format or response structure - trust stored active)
		if ($licenseStatus === 'active' && $status === 'inactive') {
			// Refresh last_check time but keep status active
			$params = ComponentHelper::getParams('com_onecore');
			$params->set('license_last_check', time());
			$component = ComponentHelper::getComponent('com_onecore');
			$table = \Joomla\CMS\Table\Table::getInstance('Extension');
			$table->load($component->id);
			$table->params = $params->toString();
			$table->store();
			return true;
		}

		// Update component params with new status
		self::updateLicenseStatus($licenseKey, $status);

		return $status === 'active';
	}

	/**
	 * Check if license has expired
	 *
	 * @return  bool  True if license has expired, false otherwise
	 *
	 * @since   1.0.0
	 */
	public static function isLicenseExpired(): bool
	{
		$params = ComponentHelper::getParams('com_onecore');
		$licenseExpires = $params->get('license_expires', 0);
		
		if ($licenseExpires <= 0) {
			return false; // No expiration date set, assume not expired
		}
		
		$expiresTimestamp = is_numeric($licenseExpires) ? (int) $licenseExpires : strtotime($licenseExpires);
		if ($expiresTimestamp <= 0) {
			return false;
		}
		
		// Check if expiration date has passed
		return time() > $expiresTimestamp;
	}

	/**
	 * Check if new Plus features can be added (license is active and not expired)
	 *
	 * @return  bool  True if new Plus features can be added
	 *
	 * @since   1.0.0
	 */
	public static function canAddPlusFeatures(): bool
	{
		if (self::isDevelopmentMode()) {
			return true;
		}
		
		// Can add Plus features only if license is active and not expired
		return self::isLicenseActive() && !self::isLicenseExpired();
	}

	/**
	 * Get license type (free or plus)
	 * Returns 'plus' if license is active (even if expired - existing Plus features still work)
	 * Returns 'free' only if license was never activated or is inactive
	 *
	 * @return  string  'free' or 'plus'
	 *
	 * @since   1.0.0
	 */
	public static function getLicenseType(): string
	{
		if (self::isDevelopmentMode()) {
			return 'plus'; // Development mode always returns 'plus'
		}
		
		// Check if license was ever activated (has license_key and status was active)
		$params = ComponentHelper::getParams('com_onecore');
		$licenseKey = $params->get('license_key', '');
		$licenseStatus = $params->get('license_status', 'inactive');
		
		// If license was activated (even if expired), return 'plus' to allow existing Plus features
		if (!empty($licenseKey) && $licenseStatus === 'active') {
			return 'plus';
		}
		
		return 'free';
	}

	/**
	 * Get component version name (OneCore or OneCore Plus)
	 *
	 * @return  string  Component version name
	 *
	 * @since   1.0.0
	 */
	public static function getComponentVersionName(): string
	{
		$isActive = self::isLicenseActive();
		$baseVersion = 'OneCore';
		
		if (self::isDevelopmentMode()) {
			return $baseVersion . ' Plus (Dev Mode)';
		}

		return $isActive ? $baseVersion . ' Plus' : $baseVersion;
	}

	/**
	 * Check license status via WooCommerce API Manager
	 *
	 * @param   string  $licenseKey  The license key to check
	 *
	 * @return  string  License status: 'active', 'inactive', 'expired', or 'invalid'
	 *
	 * @since   1.0.0
	 */
	public static function checkLicenseStatus(string $licenseKey): string
	{
		if (empty($licenseKey)) {
			return 'invalid';
		}

		try {
			$http = HttpFactory::getHttp();
			$instance = self::getInstance();
			$logger = Factory::getApplication()->getLogger();
			
			// Instance is now a unique alphanumeric ID, not a URL - use as is
			$instanceClean = $instance;
			
			// Try each product_id until one works
			$lastResponse = null;
			$lastError = null;
			$successfulProductId = null;
			
			foreach (self::WC_AM_PRODUCT_IDS as $productId) {
				// Build query string according to WC AM API documentation (same format as activate)
				$queryParams = [
					'wc-api' => 'wc-am-api',
					'wc_am_action' => 'status',
					'api_key' => $licenseKey,
					'product_id' => $productId,
					'instance' => $instanceClean
				];
				
				// Build full URL with query string
				$statusUrl = self::API_ENDPOINT_BASE . '?' . http_build_query($queryParams);
				
				// GET request with query string (WC AM API standard format)
				$response = $http->get($statusUrl, [
					'User-Agent' => 'OneCore-Joomla/1.0',
					'Accept' => 'application/json'
				], 30);
				
				$lastResponse = $response;
				
				// If successful (200), break and use this product_id
				if ($response->code === 200) {
					$successfulProductId = $productId;
					break;
				}
				
				// Store error for logging if all fail
				if ($response->code !== 200) {
					$lastError = 'HTTP ' . $response->code . ' for product_id ' . $productId;
				}
			}
			
			$response = $lastResponse;

			// Store response for debugging
			self::$lastApiResponse = [
				'code' => $response->code,
				'body' => $response->body
			];

			if ($response->code !== 200) {
				return 'inactive';
			}

			$body = trim($response->body);
			
			// Try to decode JSON
			$result = json_decode($body, true);

			if (json_last_error() !== JSON_ERROR_NONE) {
				// If not JSON, try to check if it's a simple text response
				$bodyLower = strtolower($body);
				if (strpos($bodyLower, 'active') !== false || strpos($bodyLower, 'activated') !== false) {
					return 'active';
				}
				return 'inactive';
			}
			
			// Extract expiration date if available and update params
			$expires = null;
			if (is_array($result)) {
				// Extract expiration date from status response
				// Format: data.api_key_expirations.non_wc_subs_resources[0].friendly_api_key_expiration_date or next_payment
				if (isset($result['data']['api_key_expirations']['non_wc_subs_resources']) 
					&& is_array($result['data']['api_key_expirations']['non_wc_subs_resources'])
					&& !empty($result['data']['api_key_expirations']['non_wc_subs_resources'])) {
					
					$firstResource = $result['data']['api_key_expirations']['non_wc_subs_resources'][0];
					
					// Try friendly_api_key_expiration_date first (e.g., "January 21, 2026")
					if (isset($firstResource['friendly_api_key_expiration_date']) && !empty($firstResource['friendly_api_key_expiration_date'])) {
						$expires = $firstResource['friendly_api_key_expiration_date'];
					} elseif (isset($firstResource['next_payment']) && !empty($firstResource['next_payment'])) {
						$expires = $firstResource['next_payment'];
					}
				}
				
				// Fallback: try other possible expiration fields
				if (($expires === null || $expires === '' || $expires === '0') && isset($result['data'])) {
					if (isset($result['data']['expires'])) {
						$expires = $result['data']['expires'];
					} elseif (isset($result['data']['expires_date'])) {
						$expires = $result['data']['expires_date'];
					} elseif (isset($result['data']['access_expires'])) {
						$expires = $result['data']['access_expires'];
					} elseif (isset($result['data']['subscription_expires'])) {
						$expires = $result['data']['subscription_expires'];
					} elseif (isset($result['expires'])) {
						$expires = $result['expires'];
					} elseif (isset($result['expires_date'])) {
						$expires = $result['expires_date'];
					} elseif (isset($result['access_expires'])) {
						$expires = $result['access_expires'];
					}
				}
				
				// Check if license is lifetime (never expires)
				$isLifetime = false;
				if ($expires !== null && $expires !== '' && $expires !== '0') {
					$expiresLower = strtolower(trim((string) $expires));
					// Check for lifetime indicators
					if ($expiresLower === 'never' || 
						$expiresLower === 'lifetime' || 
						$expiresLower === 'null' ||
						$expiresLower === '0000-00-00' ||
						$expiresLower === '0000-00-00 00:00:00' ||
						strpos($expiresLower, 'never expires') !== false ||
						strpos($expiresLower, 'lifetime license') !== false) {
						$isLifetime = true;
					}
				}
				
				$params = ComponentHelper::getParams('com_onecore');
				
				if ($isLifetime || $expires === null || $expires === '' || $expires === '0') {
					// Lifetime license - set to 0 (never expires)
					$params->set('license_expires', 0);
				} else {
					// Parse date string like "January 21, 2026" to timestamp
					$expiresTimestamp = is_numeric($expires) ? (int) $expires : strtotime($expires);
					if ($expiresTimestamp !== false && $expiresTimestamp > 0) {
						$params->set('license_expires', $expiresTimestamp);
					} else {
						// Invalid date - treat as lifetime
						$params->set('license_expires', 0);
					}
				}
				
				$component = ComponentHelper::getComponent('com_onecore');
				$table = \Joomla\CMS\Table\Table::getInstance('Extension');
				$table->load($component->id);
				$table->params = $params->toString();
				$table->store();
			}

			// Check various response structures from WooCommerce API Manager
			// WC AM API returns status_check and data.activated - check these first

			// Format 0a: { "status_check": "active" } - official WC AM status response
			if (isset($result['status_check'])) {
				$statusCheck = strtolower(trim((string) $result['status_check']));
				if (in_array($statusCheck, ['active', 'activated', 'valid', '1', 'true'], true)) {
					return 'active';
				}
				if (in_array($statusCheck, ['expired', 'expires'], true)) {
					return 'expired';
				}
			}

			// Format 0b: { "data": { "activated": true } } - official WC AM status response
			if (isset($result['data']['activated'])) {
				$activated = $result['data']['activated'];
				if ($activated === true || $activated === 1 || strtolower((string) $activated) === 'yes' || strtolower((string) $activated) === 'true') {
					return 'active';
				}
			}

			// Format 1: { "status": "active" } or { "status": "inactive" }
			if (isset($result['status'])) {
				$status = strtolower(trim((string) $result['status']));
				
				// Map various status values to our standard values
				if (in_array($status, ['active', 'activated', 'valid', '1', 'true'], true)) {
					return 'active';
				} elseif (in_array($status, ['expired', 'expires'], true)) {
					return 'expired';
				} else {
					return 'inactive';
				}
			}

			// Format 2: { "activated": true } or { "activated": "yes" }
			if (isset($result['activated'])) {
				$activated = $result['activated'];
				if ($activated === true || $activated === 1 || strtolower((string) $activated) === 'yes' || strtolower((string) $activated) === 'true') {
					return 'active';
				}
			}

			// Format 3: { "success": true, "status": "active" }
			if (isset($result['success']) && $result['success'] === true) {
				if (isset($result['status'])) {
					$status = strtolower(trim((string) $result['status']));
					if (in_array($status, ['active', 'activated', 'valid'], true)) {
						return 'active';
					}
				} else {
					// If success is true but no status, assume active
					return 'active';
				}
			}

			// Format 4: { "license_status": "active" }
			if (isset($result['license_status'])) {
				$status = strtolower(trim((string) $result['license_status']));
				if (in_array($status, ['active', 'activated', 'valid'], true)) {
					return 'active';
				}
			}

			// Format 5: Check for any field containing "active"
			foreach ($result as $key => $value) {
				if (is_string($value) && strtolower($value) === 'active') {
					return 'active';
				}
				if (is_bool($value) && $value === true && (strpos(strtolower($key), 'active') !== false || strpos(strtolower($key), 'valid') !== false)) {
					return 'active';
				}
			}

			// If no clear status found, log and return inactive
			$logger->warning(
				'OneCore License Check: No recognized status field found in response: ' . json_encode($result),
				['category' => 'onecore']
			);
			return 'inactive';

		} catch (\Exception $e) {
			// Log exception but don't throw
			Factory::getApplication()->getLogger()->error(
				'OneCore License Check Exception: ' . $e->getMessage() . ', Trace: ' . $e->getTraceAsString(),
				['category' => 'onecore', 'exception' => $e]
			);
			return 'inactive';
		}
	}

	/**
	 * Activate license key
	 * First activates the license via /activate endpoint, then checks status
	 *
	 * @param   string  $licenseKey  The license key to activate
	 *
	 * @return  array  Result array with 'success' and 'message' keys
	 *
	 * @since   1.0.0
	 */
	public static function activateLicense(string $licenseKey): array
	{
		if (empty($licenseKey)) {
			return [
				'success' => false,
				'message' => 'License key cannot be empty'
			];
		}

		try {
			$http = HttpFactory::getHttp();
			$instance = self::getInstance();
			$logger = Factory::getApplication()->getLogger();
			
			// Instance is now a unique alphanumeric ID, not a URL - use as is
			$instanceClean = $instance;
			
			// First, check if license is already active for any product_id
			$status = self::checkLicenseStatus($licenseKey);
			if ($status === 'active') {
				// License is already active, return success
				return [
					'success' => true,
					'message' => 'License is already active',
					'status' => 'active'
				];
			}
			
			// Try each product_id until one works
			$activateResponse = null;
			$activateResult = null;
			$usedProductId = null;
			$allErrors = [];
			$successFound = false;
			
			foreach (self::WC_AM_PRODUCT_IDS as $productId) {
				// Build query string according to WC AM API documentation
				// Format: https://comdev.eu/wc-api?wc-api=wc-am-api&wc_am_action=activate&api_key=...&product_id=...&instance=...
				$queryParams = [
					'wc-api' => 'wc-am-api',
					'wc_am_action' => 'activate',
					'api_key' => $licenseKey,
					'product_id' => $productId,
					'instance' => $instanceClean
				];
				
				// Build full URL with query string
				$activateUrl = self::API_ENDPOINT_BASE . '?' . http_build_query($queryParams);
				
				// Approach 1: GET request with query string (WC AM API standard format)
				$response = $http->get($activateUrl, [
					'User-Agent' => 'OneCore-Joomla/1.0',
					'Accept' => 'application/json'
				], 30);
				
				// Approach 2: POST request with query string in URL (fallback)
				if ($response->code !== 200 && $response->code !== 201) {
					$response = $http->post($activateUrl, '', [
						'User-Agent' => 'OneCore-Joomla/1.0',
						'Accept' => 'application/json'
					], 30);
				}
				
				// Parse response to check if activation was successful
				$result = json_decode($response->body, true);
				$activated = false;
				$alreadyActivated = false;
				
				// Log full API response for debugging
				$logger->info(
					'OneCore License Activation API Response for product_id ' . $productId . ': ' . json_encode([
						'url' => $activateUrl,
						'http_code' => $response->code,
						'response' => $result,
						'raw_body' => substr($response->body, 0, 500)
					]),
					['category' => 'onecore']
				);
				
				// Check if activation was actually successful from response
				if ($response->code === 200 || $response->code === 201) {
					if (is_array($result)) {
						// Check for success indicators - activated must be true, not just success
						if (isset($result['activated']) && $result['activated'] === true) {
							$activated = true;
							// Also check if total_activations increased (verify actual activation occurred)
							if (isset($result['data']['total_activations'])) {
								$logger->info(
									'OneCore License Activation: total_activations = ' . $result['data']['total_activations'] . 
									', activations_remaining = ' . ($result['data']['activations_remaining'] ?? 'N/A'),
									['category' => 'onecore']
								);
							}
						} elseif (isset($result['success']) && $result['success'] === true) {
							// If success is true but activated is not set, still check
							if (isset($result['activated']) && $result['activated'] === true) {
								$activated = true;
							} elseif (!isset($result['error']) && !isset($result['data']['error'])) {
								// If success=true and no error, assume activated
								$activated = true;
							}
						}
					}
				}
				
				// Check if license is already activated (this is also a success case)
				if (!$activated && ($response->code === 200 || $response->code === 201)) {
					$errorText = '';
					if (is_array($result)) {
						if (isset($result['error'])) {
							$errorText = strtolower($result['error']);
						} elseif (isset($result['message'])) {
							$errorText = strtolower($result['message']);
						} elseif (isset($result['data']['error'])) {
							$errorText = strtolower($result['data']['error']);
						}
					} else {
						$errorText = strtolower($response->body);
					}
					
					// Check for "already activated" messages
					if (strpos($errorText, 'already been activated') !== false ||
						strpos($errorText, 'already activated') !== false ||
						strpos($errorText, 'has already been activated') !== false) {
						$alreadyActivated = true;
						$activated = true; // Treat as success
					}
				}
				
				// If activation was successful, break and use this product_id
				if ($activated) {
					$activateResponse = $response;
					$activateResult = $result;
					$usedProductId = $productId;
					$successFound = true;
					break; // Success - use this product_id
				}
				
				// Store error for logging if this product_id failed
				$errorMsg = 'HTTP ' . $response->code . ' for product_id ' . $productId;
				if (trim($response->body) === '-1') {
					$errorMsg .= ' (Invalid parameters)';
				} elseif (is_array($result)) {
					if (isset($result['error'])) {
						$errorMsg .= ': ' . $result['error'];
					} elseif (isset($result['message'])) {
						$errorMsg .= ': ' . $result['message'];
					}
				}
				$allErrors[] = $errorMsg;
				
				// Store last response for final error message if all fail
				if (!$successFound) {
					$activateResponse = $response;
					$activateResult = $result;
					$usedProductId = $productId; // Keep last tried product_id
				}
			}
			
			// If no success found after trying all product_ids, return error
			if (!$successFound) {
				$errorMsg = 'Activation failed';
				
				// Build comprehensive error message
				if ($activateResponse && $activateResponse->code !== 200 && $activateResponse->code !== 201) {
					$errorMsg .= ' with HTTP ' . $activateResponse->code;
				}
				
				// Response "-1" from WooCommerce API Manager usually means invalid parameters
				if ($activateResponse && (trim($activateResponse->body) === '-1' || $activateResponse->body === '-1')) {
					$productIdsList = implode(', ', self::WC_AM_PRODUCT_IDS);
					$errorMsg .= '. Invalid parameters or license inactive. Check: api key format, product IDs (' . $productIdsList . '), instance URL format, and if license is active in WooCommerce. Tried all product IDs. Response: -1';
				} elseif (is_array($activateResult)) {
					if (isset($activateResult['error'])) {
						$errorMsg .= ': ' . $activateResult['error'];
					} elseif (isset($activateResult['message'])) {
						$errorMsg .= ': ' . $activateResult['message'];
					}
				} elseif ($activateResponse) {
					$errorMsg .= '. Response: ' . substr($activateResponse->body, 0, 200);
				}
				
				// Add information about all tried product_ids
				if (!empty($allErrors)) {
					$productIdsList = implode(', ', self::WC_AM_PRODUCT_IDS);
					$errorMsg .= ' (Tried all product IDs: ' . $productIdsList . ')';
					
					// Log all errors for debugging
					$logger->error(
						'OneCore License Activation Failed - All attempts: ' . implode(' | ', $allErrors),
						['category' => 'onecore']
					);
				}
				
				$logger->error(
					'OneCore License Activation Failed: ' . $errorMsg,
					['category' => 'onecore']
				);
				
				return [
					'success' => false,
					'message' => $errorMsg,
					'status' => 'inactive'
				];
			}
			
			// Activation was successful - extract message
			$activationMessage = '';
			if (is_array($activateResult)) {
				// Check if it was "already activated" case
				$errorText = '';
				if (isset($activateResult['error'])) {
					$errorText = strtolower($activateResult['error']);
				} elseif (isset($activateResult['message'])) {
					$errorText = strtolower($activateResult['message']);
				}
				
				if (strpos($errorText, 'already been activated') !== false ||
					strpos($errorText, 'already activated') !== false ||
					strpos($errorText, 'has already been activated') !== false) {
					$activationMessage = 'License is already activated for this domain';
				} else {
					$activationMessage = $activateResult['message'] ?? 'License activated successfully';
				}
			} else {
				$activationMessage = 'License activated successfully';
			}
			
			// Process successful activation
			if ($successFound) {
				// Extract expiration date from response if available
				$expires = null;
				
				if (is_array($activateResult)) {
					// Try various paths for expiration date
					if (isset($activateResult['data']['expires'])) {
						$expires = $activateResult['data']['expires'];
					} elseif (isset($activateResult['data']['expires_date'])) {
						$expires = $activateResult['data']['expires_date'];
					} elseif (isset($activateResult['data']['access_expires'])) {
						$expires = $activateResult['data']['access_expires'];
					} elseif (isset($activateResult['data']['subscription_expires'])) {
						$expires = $activateResult['data']['subscription_expires'];
					} elseif (isset($activateResult['expires'])) {
						$expires = $activateResult['expires'];
					} elseif (isset($activateResult['expires_date'])) {
						$expires = $activateResult['expires_date'];
					} elseif (isset($activateResult['access_expires'])) {
						$expires = $activateResult['access_expires'];
					}
					
					// Always try status endpoint to get expiration date (activation response doesn't include it)
					// Use the same product_id that worked for activation
					$statusQueryParams = [
						'wc-api' => 'wc-am-api',
						'wc_am_action' => 'status',
						'api_key' => $licenseKey,
						'product_id' => $usedProductId ?? self::WC_AM_PRODUCT_IDS[0],
						'instance' => $instanceClean
					];
					$statusUrl = self::API_ENDPOINT_BASE . '?' . http_build_query($statusQueryParams);
					
					try {
						$statusResponse = $http->get($statusUrl, [
							'User-Agent' => 'OneCore-Joomla/1.0',
							'Accept' => 'application/json'
						], 30);
						
						if ($statusResponse->code === 200) {
							$statusResult = json_decode($statusResponse->body, true);
							if (is_array($statusResult)) {
								// Extract expiration date from status response
								// Format: data.api_key_expirations.non_wc_subs_resources[0].friendly_api_key_expiration_date or next_payment
								if (isset($statusResult['data']['api_key_expirations']['non_wc_subs_resources']) 
									&& is_array($statusResult['data']['api_key_expirations']['non_wc_subs_resources'])
									&& !empty($statusResult['data']['api_key_expirations']['non_wc_subs_resources'])) {
									
									$firstResource = $statusResult['data']['api_key_expirations']['non_wc_subs_resources'][0];
									
									// Try friendly_api_key_expiration_date first (e.g., "January 21, 2026")
									if (isset($firstResource['friendly_api_key_expiration_date']) && !empty($firstResource['friendly_api_key_expiration_date'])) {
										$expires = $firstResource['friendly_api_key_expiration_date'];
									} elseif (isset($firstResource['next_payment']) && !empty($firstResource['next_payment'])) {
										$expires = $firstResource['next_payment'];
									}
								}
								
								// Fallback: try other possible fields
								if (($expires === null || $expires === '' || $expires === '0') && isset($statusResult['data'])) {
									if (isset($statusResult['data']['expires'])) {
										$expires = $statusResult['data']['expires'];
									} elseif (isset($statusResult['data']['expires_date'])) {
										$expires = $statusResult['data']['expires_date'];
									} elseif (isset($statusResult['data']['access_expires'])) {
										$expires = $statusResult['data']['access_expires'];
									} elseif (isset($statusResult['expires'])) {
										$expires = $statusResult['expires'];
									}
								}
								
								// Check if license is lifetime (never expires)
								if ($expires !== null && $expires !== '' && $expires !== '0') {
									$expiresLower = strtolower(trim((string) $expires));
									if ($expiresLower === 'never' || 
										$expiresLower === 'lifetime' || 
										$expiresLower === 'null' ||
										$expiresLower === '0000-00-00' ||
										$expiresLower === '0000-00-00 00:00:00' ||
										strpos($expiresLower, 'never expires') !== false ||
										strpos($expiresLower, 'lifetime license') !== false) {
										// Set to null so updateLicenseStatus will treat it as lifetime
										$expires = null;
									}
								}
							}
						}
					} catch (\Exception $e) {
						// Ignore errors when fetching expiration date
					}
				}
				
				// Activation successful - update component params with expiration date
				// If expires is null or empty, updateLicenseStatus will set it to 0 (lifetime)
				self::updateLicenseStatus($licenseKey, 'active', $expires);
				
				return [
					'success' => true,
					'message' => $activationMessage ?: 'License activated successfully',
					'status' => 'active'
				];
			}
			
		} catch (\Exception $e) {
			$logger = Factory::getApplication()->getLogger();
			$logger->error(
				'OneCore License Activation Exception: ' . $e->getMessage(),
				['category' => 'onecore', 'exception' => $e]
			);
			
			return [
				'success' => false,
				'message' => 'License activation failed: ' . $e->getMessage(),
				'status' => 'inactive'
			];
		}
	}

	/**
	 * Deactivate license
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public static function deactivateLicense(): void
	{
		$params = ComponentHelper::getParams('com_onecore');
		$params->set('license_key', '');
		$params->set('license_status', 'inactive');
		$params->set('license_last_check', 0);
		$params->set('license_type', 'free');
		$params->set('license_expires', 0);

		// Save params
		$component = ComponentHelper::getComponent('com_onecore');
		$table = \Joomla\CMS\Table\Table::getInstance('Extension');
		$table->load($component->id);
		$table->params = $params->toString();
		$table->store();
	}

	/**
	 * Update license status in component params
	 *
	 * @param   string       $licenseKey  The license key
	 * @param   string       $status      The license status
	 * @param   string|null  $expires     The expiration date (timestamp or date string)
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public static function updateLicenseStatus(string $licenseKey, string $status, ?string $expires = null): void
	{
		$params = ComponentHelper::getParams('com_onecore');
		$params->set('license_key', $licenseKey);
		$params->set('license_status', $status);
		$params->set('license_last_check', time());
		$params->set('license_type', $status === 'active' ? 'plus' : 'free');
		
		// Save expiration date if provided
		if ($expires !== null && $expires !== '' && $expires !== '0') {
			// Check if license is lifetime (never expires)
			$expiresLower = strtolower(trim((string) $expires));
			$isLifetime = false;
			
			// Check for lifetime indicators
			if ($expiresLower === 'never' || 
				$expiresLower === 'lifetime' || 
				$expiresLower === 'null' ||
				$expiresLower === '0000-00-00' ||
				$expiresLower === '0000-00-00 00:00:00' ||
				strpos($expiresLower, 'never expires') !== false ||
				strpos($expiresLower, 'lifetime license') !== false) {
				$isLifetime = true;
			}
			
			if ($isLifetime) {
				// Lifetime license - set to 0 (never expires)
				$params->set('license_expires', 0);
			} else {
				// Convert to timestamp if it's a date string
				// Handle both numeric timestamps and date strings like "January 21, 2026"
				$expiresTimestamp = null;
				
				if (is_numeric($expires)) {
					$expiresTimestamp = (int) $expires;
					// If it's a timestamp in seconds (not milliseconds), use as is
					if ($expiresTimestamp > 1000000000) {
						// Valid Unix timestamp (after 2001-09-09)
						$expiresTimestamp = $expiresTimestamp;
					} elseif ($expiresTimestamp > 0) {
						// Might be milliseconds, convert to seconds
						$expiresTimestamp = (int) ($expiresTimestamp / 1000);
					}
				} else {
					// Parse date string (e.g., "January 21, 2026")
					$expiresTimestamp = strtotime($expires);
					if ($expiresTimestamp === false || $expiresTimestamp <= 0) {
						$expiresTimestamp = null;
					}
				}
				
				if ($expiresTimestamp !== null && $expiresTimestamp > 0) {
					// Ensure it's saved as integer, not string
					$params->set('license_expires', (int) $expiresTimestamp);
				} else {
					// Invalid date - treat as lifetime
					$params->set('license_expires', 0);
				}
			}
		} else {
			// Clear expiration if not provided - treat as lifetime (never expires)
			$params->set('license_expires', 0);
		}

		// Save params
		$component = ComponentHelper::getComponent('com_onecore');
		$table = \Joomla\CMS\Table\Table::getInstance('Extension');
		$table->load($component->id);
		$table->params = $params->toString();
		$table->store();
		
		// Clear session cache to force reload of formatted data
		$app = Factory::getApplication();
		$app->setUserState('com_config.edit.component.com_onecore.data', null);
	}

	/**
	 * Store last API response for debugging
	 *
	 * @var    array|null
	 * @since  1.0.10
	 */
	private static $lastApiResponse = null;

	/**
	 * Get last API response details
	 *
	 * @return  string  Details string
	 *
	 * @since   1.0.10
	 */
	private static function getLastApiResponseDetails(): string
	{
		if (self::$lastApiResponse === null) {
			return '';
		}

		$details = [];
		if (isset(self::$lastApiResponse['code'])) {
			$details[] = 'HTTP ' . self::$lastApiResponse['code'];
		}
		if (isset(self::$lastApiResponse['body'])) {
			$body = self::$lastApiResponse['body'];
			// Try to extract error message from response
			$result = json_decode($body, true);
			if (is_array($result)) {
				if (isset($result['error'])) {
					$details[] = 'Error: ' . $result['error'];
				}
				if (isset($result['message'])) {
					$details[] = 'Message: ' . $result['message'];
				}
				if (isset($result['data'])) {
					if (is_string($result['data'])) {
						$details[] = 'Data: ' . $result['data'];
					} elseif (is_array($result['data']) && isset($result['data']['message'])) {
						$details[] = 'Data: ' . $result['data']['message'];
					}
				}
			} else {
				// If not JSON, show first 100 chars
				$details[] = 'Response: ' . substr(strip_tags($body), 0, 100);
			}
		}

		return implode('. ', $details);
	}

	/**
	 * Get instance ID for this Joomla installation
	 * According to WooCommerce API Manager documentation, instance should be a unique alphanumeric string
	 * (not a URL). We generate it from URL hash to ensure it's unique and persistent per installation.
	 *
	 * @return  string  Instance ID (unique alphanumeric string)
	 *
	 * @since   1.0.0
	 */
	private static function getInstance(): string
	{
		// Get component params to store/retrieve instance ID
		$params = ComponentHelper::getParams('com_onecore');
		$instanceId = $params->get('api_instance_id', '');
		
		// Generate instance ID if not exists - use hash of site URL to ensure uniqueness and persistence
		if (empty($instanceId)) {
			$siteUrl = Uri::root();
			// Generate unique alphanumeric ID from URL hash (first 13 chars, similar to WooCommerce examples)
			$instanceId = substr(md5($siteUrl), 0, 13);
			
			// Save instance ID to params
			$params->set('api_instance_id', $instanceId);
			$component = ComponentHelper::getComponent('com_onecore');
			$table = \Joomla\CMS\Table\Table::getInstance('Extension');
			$table->load($component->id);
			$table->params = $params->toString();
			$table->store();
		}
		
		return $instanceId;
	}

	/**
	 * Get license information
	 *
	 * @return  array  License information array
	 *
	 * @since   1.0.0
	 */
	public static function getLicenseInfo(): array
	{
		$params = ComponentHelper::getParams('com_onecore');
		$licenseKey = $params->get('license_key', '');
		$licenseStatus = $params->get('license_status', 'inactive');
		$licenseLastCheck = $params->get('license_last_check', 0);
		$licenseExpires = $params->get('license_expires', 0);

		// Check if license expires within 1 month
		$showRenewalDiscount = false;
		$daysUntilExpiry = null;
		if ($licenseExpires > 0) {
			$now = time();
			$expiresTimestamp = is_numeric($licenseExpires) ? (int) $licenseExpires : strtotime($licenseExpires);
			if ($expiresTimestamp > $now) {
				$daysUntilExpiry = (int) (($expiresTimestamp - $now) / 86400); // days
				// Show discount if expires within 30 days
				$showRenewalDiscount = $daysUntilExpiry <= 30 && $daysUntilExpiry > 0;
			} elseif ($expiresTimestamp <= $now) {
				// Already expired
				$daysUntilExpiry = 0;
				$showRenewalDiscount = true;
			}
		}

		return [
			'key' => $licenseKey,
			'status' => $licenseStatus,
			'is_active' => self::isLicenseActive(),
			'type' => self::getLicenseType(),
			'version_name' => self::getComponentVersionName(),
			'last_check' => $licenseLastCheck,
			'last_check_formatted' => $licenseLastCheck > 0 
				? Factory::getDate($licenseLastCheck)->format('Y-m-d H:i:s') 
				: '',
			'expires' => $licenseExpires,
			'expires_formatted' => $licenseExpires > 0 
				? Factory::getDate($licenseExpires)->format('Y-m-d H:i:s') 
				: '',
			'show_renewal_discount' => $showRenewalDiscount,
			'days_until_expiry' => $daysUntilExpiry
		];
	}
}
