<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Helper;

use Joomla\CMS\Factory;
use Joomla\CMS\Mail\MailerFactoryInterface;
use Joomla\CMS\Mail\MailerInterface;
use Joomla\CMS\Mail\Exception\MailDisabledException;
use Joomla\CMS\Uri\Uri;
use Joomla\Database\DatabaseInterface;
use PHPMailer\PHPMailer\Exception as phpmailerException;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Email Helper class for sending emails using email templates.
 *
 * @since  1.0.17
 */
class EmailHelper
{
	/**
	 * Send email using email template.
	 *
	 * @param   int     $templateId  Email template ID
	 * @param   array   $variables   Variables to replace in template
	 * @param   string  $recipientEmail  Recipient email address
	 * @param   string  $recipientName   Recipient name
	 * @param   string  $ccEmail         CC email address (optional)
	 *
	 * @return  boolean  True on success, false on failure
	 *
	 * @since   1.0.17
	 */
	public static function sendEmail($templateId, $variables, $recipientEmail, $recipientName = '', $ccEmail = '')
	{
		try {
			$db = Factory::getContainer()->get(DatabaseInterface::class);
			
			// Get email template
			$query = $db->getQuery(true)
				->select('*')
				->from($db->quoteName('#__one_email_templ'))
				->where($db->quoteName('id') . ' = :templateId')
				->where($db->quoteName('published') . ' = 1')
				->bind(':templateId', $templateId, \Joomla\Database\ParameterType::INTEGER);
			
			$db->setQuery($query);
			$template = $db->loadObject();
			
			if (!$template) {
				return false;
			}
			
			// Replace variables in email title and content
			$emailTitle = str_replace(array_keys($variables), array_values($variables), $template->email_title);
			$emailContent = str_replace(array_keys($variables), array_values($variables), $template->email_content);
			
			// Get mailer
			$mailer = Factory::getContainer()->get(MailerFactoryInterface::class)->createMailer();
			
			// Set sender
			$app = Factory::getApplication();
			$mailer->setSender($app->get('mailfrom'), $app->get('fromname'));
			
			// Set recipient
			$mailer->addRecipient($recipientEmail, $recipientName);
			
			// Add CC if specified and template allows it
			if (!empty($ccEmail) && $template->send_cc_to_admin) {
				$mailer->addCc($ccEmail);
			}
			
			// Set subject and body
			$mailer->setSubject($emailTitle);
			$mailer->setBody($emailContent);
			
			// Set HTML mode if mailer supports it
			if ($mailer instanceof \Joomla\CMS\Mail\Mail) {
				$mailer->isHtml(true);
			}
			
			// Send email
			$sent = false;
			$errorMessage = null;
			
			try {
				if ($mailer instanceof \Joomla\CMS\Mail\Mail) {
					// Use Send() method which returns boolean
					$sent = $mailer->Send();
					if (!$sent) {
						$errorMessage = $mailer->ErrorInfo ?? 'Unknown error';
					}
				} else {
					// Use send() method from MailerInterface (throws exception)
					$mailer->send();
					$sent = true;
				}
			} catch (MailDisabledException | phpmailerException | \RuntimeException $e) {
				$sent = false;
				$errorMessage = $e->getMessage();
			}
			
			// Log email to database
			$logData = [
				'template_id' => $templateId,
				'recipient_email' => $recipientEmail,
				'recipient_name' => $recipientName,
				'cc_email' => $ccEmail ?: '',
				'subject' => $emailTitle,
				'body' => $emailContent,
				'status' => $sent === true ? 'sent' : 'failed',
				'error_message' => $errorMessage,
				'sent_at' => $sent === true ? Factory::getDate()->toSql() : null,
				'created' => Factory::getDate()->toSql(),
				'created_by' => Factory::getApplication()->getIdentity()->id,
			];
			
			$query = $db->getQuery(true)
				->insert($db->quoteName('#__one_email_logs'))
				->columns([
					$db->quoteName('template_id'),
					$db->quoteName('recipient_email'),
					$db->quoteName('recipient_name'),
					$db->quoteName('cc_email'),
					$db->quoteName('subject'),
					$db->quoteName('body'),
					$db->quoteName('status'),
					$db->quoteName('error_message'),
					$db->quoteName('sent_at'),
					$db->quoteName('created'),
					$db->quoteName('created_by'),
				])
				->values(
					':templateId, :recipientEmail, :recipientName, :ccEmail, :subject, :body, :status, :errorMessage, :sentAt, :created, :createdBy'
				);
			
			$query->bind(':templateId', $logData['template_id'], \Joomla\Database\ParameterType::INTEGER);
			$query->bind(':recipientEmail', $logData['recipient_email']);
			$query->bind(':recipientName', $logData['recipient_name']);
			$query->bind(':ccEmail', $logData['cc_email']);
			$query->bind(':subject', $logData['subject']);
			$query->bind(':body', $logData['body']);
			$query->bind(':status', $logData['status']);
			$query->bind(':errorMessage', $logData['error_message']);
			$query->bind(':sentAt', $logData['sent_at']);
			$query->bind(':created', $logData['created']);
			$query->bind(':createdBy', $logData['created_by'], \Joomla\Database\ParameterType::INTEGER);
			
			$db->setQuery($query)->execute();
			
			return $sent === true;
		} catch (\Exception $e) {
			// Log error
			try {
				$db = Factory::getContainer()->get(DatabaseInterface::class);
				$query = $db->getQuery(true)
					->insert($db->quoteName('#__one_email_logs'))
					->columns([
						$db->quoteName('template_id'),
						$db->quoteName('recipient_email'),
						$db->quoteName('recipient_name'),
						$db->quoteName('cc_email'),
						$db->quoteName('subject'),
						$db->quoteName('body'),
						$db->quoteName('status'),
						$db->quoteName('error_message'),
						$db->quoteName('created'),
						$db->quoteName('created_by'),
					])
					->values(
						':templateId, :recipientEmail, :recipientName, :ccEmail, :subject, :body, :status, :errorMessage, :created, :createdBy'
					);
				
				$query->bind(':templateId', $templateId, \Joomla\Database\ParameterType::INTEGER);
				$query->bind(':recipientEmail', $recipientEmail);
				$query->bind(':recipientName', $recipientName);
				$query->bind(':ccEmail', $ccEmail ?: '');
				$query->bind(':subject', $emailTitle ?? '');
				$query->bind(':body', $emailContent ?? '');
				$query->bind(':status', 'failed');
				$query->bind(':errorMessage', $e->getMessage());
				$query->bind(':created', Factory::getDate()->toSql());
				$query->bind(':createdBy', Factory::getApplication()->getIdentity()->id, \Joomla\Database\ParameterType::INTEGER);
				
				$db->setQuery($query)->execute();
			} catch (\Exception $logError) {
				// Silently fail logging
			}
			
			return false;
		}
	}
}
