<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Field;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Form\Field\SqlField;
use Joomla\CMS\Factory;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\Database\ParameterType;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Event Category field - filters categories based on root category from component options
 *
 * @since  1.0.0
 */
class EventCategoryField extends SqlField
{
	/**
	 * The form field type.
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	protected $type = 'EventCategory';

	/**
	 * Method to get the options to populate list
	 *
	 * @return  array  The field option objects.
	 *
	 * @since   1.0.0
	 */
	protected function getOptions()
	{
		// Get database - use getDatabase() if available, otherwise use Factory
		try {
			$db = $this->getDatabase();
		} catch (\Exception $e) {
			$db = Factory::getContainer()->get(\Joomla\Database\DatabaseInterface::class);
		}
		
		// Get root category from component options
		$params = ComponentHelper::getParams('com_onecore');
		$rootCategoryId = (int) $params->get('events_root_category', 0);
		
		$query = $db->getQuery(true);
		
		// Build base query
		$query->select([
			$db->quoteName('a.id', 'value'),
			$db->quoteName('a.title', 'text'),
			$db->quoteName('a.level'),
			$db->quoteName('a.lft'),
			$db->quoteName('a.rgt')
		])
		->from($db->quoteName('#__one_categories', 'a'))
		->where($db->quoteName('a.published') . ' IN (0, 1)');
		
		// If root category is set, filter to show only that category and its children
		if ($rootCategoryId > 0) {
			// Get lft and rgt of root category
			$subQuery = $db->getQuery(true)
				->select([
					$db->quoteName('lft'),
					$db->quoteName('rgt')
				])
				->from($db->quoteName('#__one_categories'))
				->where($db->quoteName('id') . ' = :rootId')
				->bind(':rootId', $rootCategoryId, ParameterType::INTEGER);
			
			$db->setQuery($subQuery);
			$rootCategory = $db->loadObject();
			
			if ($rootCategory) {
				// Filter: show root category and all its children (lft >= root.lft AND rgt <= root.rgt)
				$query->where($db->quoteName('a.lft') . ' >= :rootLft')
					->where($db->quoteName('a.rgt') . ' <= :rootRgt')
					->bind(':rootLft', $rootCategory->lft, ParameterType::INTEGER)
					->bind(':rootRgt', $rootCategory->rgt, ParameterType::INTEGER);
			}
		}
		
		// Order by lft to maintain hierarchy
		$query->order($db->quoteName('a.lft') . ' ASC');
		
		$db->setQuery($query);
		
		try {
			$items = $db->loadObjectList();
		} catch (\Exception $e) {
			// If query fails, return at least the XML options
			$items = [];
		}
		
		// Format options with indentation based on level
		$options = [];
		
		// Add "None" option first
		$options[] = \Joomla\CMS\HTML\HTMLHelper::_('select.option', '0', \Joomla\CMS\Language\Text::_('JNONE'));
		
		// Add categories with indentation based on level
		if (!empty($items)) {
			foreach ($items as $item) {
				$repeat = ($item->level > 0) ? $item->level - 1 : 0;
				$text = str_repeat('- ', $repeat) . $item->text;
				
				$options[] = \Joomla\CMS\HTML\HTMLHelper::_('select.option', $item->value, $text);
			}
		}
		
		// Parse XML options directly (skip parent::getOptions() because SqlField needs $this->query)
		// We already added "None" option above, so we can skip it in XML if present
		if ($this->element instanceof \SimpleXMLElement) {
			foreach ($this->element->xpath('option') as $xmlOption) {
				$xmlValue = (string) $xmlOption['value'];
				// Skip "None" (value=0) if we already have it
				if ($xmlValue == '0') {
					continue;
				}
				$xmlText = trim((string) $xmlOption) ?: $xmlValue;
				$options[] = \Joomla\CMS\HTML\HTMLHelper::_('select.option', $xmlValue, \Joomla\CMS\Language\Text::_($xmlText));
			}
		}
		
		return $options;
	}
}
