<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Field\Modal;

use Joomla\CMS\Factory;
use Joomla\CMS\Form\Field\ModalSelectField;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Session\Session;
use Joomla\CMS\Uri\Uri;
use Joomla\Database\ParameterType;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Supports a modal content picker for OneCore content items.
 *
 * @since  1.0.0
 */
class ContentField extends ModalSelectField
{
	/**
	 * The form field type.
	 *
	 * @var     string
	 * @since   1.0.0
	 */
	protected $type = 'Modal_Content';

	/**
	 * Method to attach a Form object to the field.
	 *
	 * @param   \SimpleXMLElement  $element  The SimpleXMLElement object representing the `<field>` tag for the form field object.
	 * @param   mixed              $value    The form field value to validate.
	 * @param   string             $group    The field name group control value.
	 *
	 * @return  boolean  True on success.
	 *
	 * @see     FormField::setup()
	 * @since   1.0.0
	 */
	public function setup(\SimpleXMLElement $element, $value, $group = null)
	{
		// Check if the value consist with id:alias, extract the id only
		if ($value && str_contains($value, ':')) {
			[$id]  = explode(':', $value, 2);
			$value = (int) $id;
		}

		$result = parent::setup($element, $value, $group);

		if (!$result) {
			return $result;
		}

		Factory::getApplication()->getLanguage()->load('com_onecore', JPATH_ADMINISTRATOR);

		// Prepare Urls - use OneCore's content view
		$linkItems = (new Uri())->setPath(Uri::base(true) . '/index.php');
		$linkItems->setQuery([
			'option'                => 'com_onecore',
			'view'                  => 'content',
			'layout'                => 'modal',
			'tmpl'                  => 'component',
			Session::getFormToken() => 1,
		]);
		$linkItem = clone $linkItems;
		$linkItem->setVar('view', 'item');
		$linkCheckin = (new Uri())->setPath(Uri::base(true) . '/index.php');
		$linkCheckin->setQuery([
			'option'                => 'com_onecore',
			'task'                  => 'content.checkin',
			'format'                => 'json',
			Session::getFormToken() => 1,
		]);

		$modalTitle = Text::_('COM_ONECORE_SELECT_CONTENT');

		$urlSelect = $linkItems;
		$urlEdit   = clone $linkItem;
		$urlEdit->setVar('task', 'item.edit');
		$urlNew    = clone $linkItem;
		$urlNew->setVar('task', 'item.add');

		$this->urls['select']  = (string) $urlSelect;
		$this->urls['new']     = (string) $urlNew;
		$this->urls['edit']    = (string) $urlEdit;
		$this->urls['checkin'] = (string) $linkCheckin;

		// Prepare titles
		$this->modalTitles['select']  = $modalTitle;
		$this->modalTitles['new']     = Text::_('COM_ONECORE_CONTENT_NEW');
		$this->modalTitles['edit']    = Text::_('COM_ONECORE_CONTENT_EDIT');

		$this->hint = $this->hint ?: Text::_('COM_ONECORE_SELECT_CONTENT');

		return $result;
	}

	/**
	 * Method to retrieve the title of selected item.
	 *
	 * @return string
	 *
	 * @since   1.0.0
	 */
	protected function getValueTitle()
	{
		$value = (int) $this->value ?: '';
		$title = '';

		if ($value) {
			try {
				$db    = $this->getDatabase();
				$query = $db->createQuery()
					->select($db->quoteName('title'))
					->from($db->quoteName('#__one_content'))
					->where($db->quoteName('id') . ' = :value')
					->bind(':value', $value, ParameterType::INTEGER);
				$db->setQuery($query);

				$title = $db->loadResult();
			} catch (\Throwable $e) {
				Factory::getApplication()->enqueueMessage($e->getMessage(), 'error');
			}
		}

		return $title ?: $value;
	}

	/**
	 * Method to get the data to be passed to the layout for rendering.
	 *
	 * @return  array
	 *
	 * @since 1.0.0
	 */
	protected function getLayoutData()
	{
		$data = parent::getLayoutData();

		return $data;
	}
}
