<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Field;

use Joomla\CMS\Factory;
use Joomla\CMS\Form\Field\TextField;
use Joomla\CMS\Language\Text;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * License Expires field - displays license expiration date in readable format
 *
 * @since  1.0.0
 */
class LicenseExpiresField extends TextField
{
	/**
	 * The form field type.
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	protected $type = 'LicenseExpires';

	/**
	 * Method to get the field input markup.
	 *
	 * @return  string  The field input markup.
	 *
	 * @since   1.0.0
	 */
	protected function getInput()
	{
		$value = $this->value;
		$displayValue = '';
		$statusHtml = '';

		// If value is empty or 0, show "Never expires"
		if (empty($value) || $value == 0 || $value === '0') {
			$displayValue = Text::_('COM_ONECORE_LICENSE_EXPIRES_NEVER');
		} else {
			// Convert timestamp to readable date
			try {
				// Value could be timestamp (integer) or date string
				if (is_numeric($value)) {
					$timestamp = (int) $value;
					$date = Factory::getDate($timestamp);
				} else {
					// Try to parse as date string
					$timestamp = strtotime($value);
					if ($timestamp !== false && $timestamp > 0) {
						$date = Factory::getDate($timestamp);
					} else {
						// Invalid date, show as-is
						$displayValue = htmlspecialchars($value);
						$date = null;
					}
				}

				if ($date !== null) {
					// Format as readable date: "January 25, 2027"
					$displayValue = $date->format('F d, Y', true);
					
					// Add "Expires in X days" or "Expired X days ago" as separate element
					$now = time();
					$diff = $timestamp - $now;
					$daysDiff = (int) ($diff / 86400);
					
					if ($daysDiff > 0) {
						$statusText = Text::sprintf('COM_ONECORE_LICENSE_EXPIRES_IN_DAYS', $daysDiff);
						$statusHtml = '<div style="margin-top: 5px; color: #10b981; font-size: 12px;">(' . htmlspecialchars($statusText) . ')</div>';
					} elseif ($daysDiff < 0) {
						$statusText = Text::sprintf('COM_ONECORE_LICENSE_EXPIRED_DAYS_AGO', abs($daysDiff));
						$statusHtml = '<div style="margin-top: 5px; color: #ef4444; font-size: 12px;">(' . htmlspecialchars($statusText) . ')</div>';
					} else {
						$statusText = Text::_('COM_ONECORE_LICENSE_EXPIRES_TODAY');
						$statusHtml = '<div style="margin-top: 5px; color: #f59e0b; font-size: 12px;">(' . htmlspecialchars($statusText) . ')</div>';
					}
				}
			} catch (\Exception $e) {
				// If there's an error parsing the date, show raw value
				$displayValue = htmlspecialchars($value);
			}
		}

		// Return readonly input with formatted value
		$html = '<div>';
		$html .= '<input type="text" readonly="readonly" class="readonly form-control" value="' . htmlspecialchars($displayValue, ENT_COMPAT, 'UTF-8') . '" />';
		$html .= $statusHtml;
		$html .= '</div>';
		
		// Add hidden field to store the actual value
		$html .= '<input type="hidden" name="' . $this->name . '" value="' . htmlspecialchars($this->value, ENT_COMPAT, 'UTF-8') . '" />';

		return $html;
	}
}
