<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Extension;

use Comdev\Component\Onecore\Administrator\Service\HTML\AdministratorService;

use Joomla\CMS\Categories\CategoryServiceInterface;
use Joomla\CMS\Categories\CategoryServiceTrait;
use Joomla\CMS\Component\Router\RouterServiceInterface;
use Joomla\CMS\Component\Router\RouterServiceTrait;
use Joomla\CMS\Extension\BootableExtensionInterface;
use Joomla\CMS\Extension\MVCComponent;
use Joomla\CMS\HTML\HTMLRegistryAwareTrait;
use Psr\Container\ContainerInterface;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Component class for com_onecore
 *
 * @since  1.0.0
 */
class OneCoreComponent extends MVCComponent implements
	BootableExtensionInterface,
	CategoryServiceInterface,
	RouterServiceInterface
{
	use HTMLRegistryAwareTrait;
	use RouterServiceTrait;

	/**
	 * The categories factory
	 *
	 * @var  \Joomla\CMS\Categories\CategoryFactoryInterface
	 *
	 * @since  1.0.0
	 */
	private $categoryFactory;

	/**
	 * Booting the extension. This is the function to set up the environment of the extension like
	 * registering new class loaders, etc.
	 *
	 * @param   ContainerInterface  $container  The container
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function boot(ContainerInterface $container)
	{
		// Register HTML services
		$this->getRegistry()->register('onecoreadministrator', new AdministratorService());
	}

	/**
	 * Returns the category service.
	 *
	 * @param   array   $options  The options
	 * @param   string  $section  The section
	 *
	 * @return  \Joomla\CMS\Categories\CategoryInterface
	 *
	 * @since   1.0.0
	 */
	public function getCategory(array $options = [], $section = ''): \Joomla\CMS\Categories\CategoryInterface
	{
		return $this->categoryFactory->createCategory($options, $section);
	}

	/**
	 * Sets the internal category factory.
	 *
	 * @param   \Joomla\CMS\Categories\CategoryFactoryInterface  $categoryFactory  The categories factory
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function setCategoryFactory(\Joomla\CMS\Categories\CategoryFactoryInterface $categoryFactory)
	{
		$this->categoryFactory = $categoryFactory;
	}

	/**
	 * Adds Count Items for Category Manager.
	 * Override to use junction table #__one_content_categories instead of direct catid column.
	 *
	 * @param   \stdClass[]  $items    The category objects
	 * @param   string       $section  The section
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function countItems(array $items, string $section)
	{
		$db = \Joomla\CMS\Factory::getDbo();

		// Index category objects by their ID
		$records = [];
		foreach ($items as $item) {
			$records[(int) $item->id] = $item;
			// Initialize counters
			$item->count_trashed = 0;
			$item->count_unpublished = 0;
			$item->count_published = 0;
			$item->count_archived = 0;
		}

		if (empty($records)) {
			return;
		}

		// Build query using junction table
		$query = $db->createQuery()
			->select(
				[
					$db->quoteName('cc.category_id', 'catid'),
					$db->quoteName('c.published', 'state'),
					'COUNT(*) AS ' . $db->quoteName('count'),
				]
			)
			->from($db->quoteName('#__one_content_categories', 'cc'))
			->innerJoin(
				$db->quoteName('#__one_content', 'c'),
				$db->quoteName('c.id') . ' = ' . $db->quoteName('cc.content_id')
			)
			->whereIn($db->quoteName('cc.category_id'), array_keys($records))
			->whereIn($db->quoteName('c.published'), [-2, 0, 1, 2])
			->group($db->quoteName(['cc.category_id', 'c.published']));

		$relationsAll = $db->setQuery($query)->loadObjectList();

		// Map state values to counter names
		$counterNames = [
			'-2' => 'count_trashed',
			'0'  => 'count_unpublished',
			'1'  => 'count_published',
			'2'  => 'count_archived',
		];

		// Loop through the DB data and update counters
		foreach ($relationsAll as $relation) {
			$id = (int) $relation->catid;
			$state = (string) $relation->state;
			
			if (isset($records[$id]) && isset($counterNames[$state])) {
				$records[$id]->{$counterNames[$state]} = (int) $relation->count;
			}
		}
	}

	/**
	 * Returns the table name for the count items functions for the given section.
	 *
	 * @param   ?string  $section  The section
	 *
	 * @return  string|null
	 *
	 * @since   1.0.0
	 */
	protected function getTableNameForSection(?string $section = null)
	{
		// Return table name without #__ prefix, as ContentHelper::countRelations() adds it
		return 'one_content';
	}

	/**
	 * Returns the state column for the count items functions for the given section.
	 *
	 * @param   ?string  $section  The section
	 *
	 * @return  string|null
	 *
	 * @since   1.0.0
	 */
	protected function getStateColumnForSection(?string $section = null)
	{
		return 'published';
	}

	/**
	 * Prepares the category form
	 *
	 * @param   \Joomla\CMS\Form\Form  $form  The form to change
	 * @param   array|object           $data  The form data
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function prepareForm(\Joomla\CMS\Form\Form $form, $data)
	{
		// No special form preparation needed for OneCore categories
	}

}

