<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Controller;

use Comdev\Component\Onecore\Administrator\Helper\LicenseHelper;
use Comdev\Component\Onecore\Administrator\Helper\EmailHelper;
use Joomla\CMS\MVC\Controller\FormController;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Reviews controller class. (OneCore Plus only)
 *
 * @since  1.0.16
 */
class ReviewsController extends FormController
{
	/**
	 * Cached Item model instance for edit operations.
	 *
	 * @var    \Joomla\CMS\MVC\Model\BaseDatabaseModel|null
	 * @since  1.0.16
	 */
	protected $itemModel = null;

	/**
	 * The prefix to use with controller messages.
	 *
	 * @var    string
	 * @since  1.0.16
	 */
	protected $text_prefix = 'COM_ONECORE_REVIEWS';

	/**
	 * The URL view item variable.
	 *
	 * @var    string
	 * @since  1.0.16
	 */
	protected $view_item = 'reviews';

	/**
	 * The URL view list variable.
	 *
	 * @var    string
	 * @since  1.0.16
	 */
	protected $view_list = 'reviews';

	/**
	 * Constructor.
	 *
	 * @param   array                 $config   An optional associative array of configuration settings.
	 * @param   ?MVCFactoryInterface  $factory  The factory.
	 * @param   ?CMSApplication       $app      The Application for the dispatcher
	 * @param   ?Input                $input    Input
	 *
	 * @since   1.0.16
	 */
	public function __construct($config = [], $factory = null, $app = null, $input = null)
	{
		parent::__construct($config, $factory, $app, $input);

		$this->registerTask('review.apply', 'save');
		$this->registerTask('review.save2new', 'save');
		$this->registerTask('review.delete', 'delete');
		$this->registerTask('reviews.add', 'add');
		$this->registerTask('reviews.edit', 'edit');
		$this->registerTask('unpublish', 'publish');
		$this->registerTask('unapprove', 'approve');
		$this->registerTask('reviews.publish', 'publish');
		$this->registerTask('reviews.unpublish', 'publish');
		$this->registerTask('reviews.approve', 'approve');
		$this->registerTask('reviews.unapprove', 'approve');
		$this->registerTask('reviews.delete', 'delete');
	}

	/**
	 * Prepare the view and ensure list model is set for list layout.
	 * The list model is in Model\Reviews\ReviewsModel (subfolder), so we set it explicitly.
	 *
	 * @param   \Joomla\CMS\MVC\View\ViewInterface  $view  The view object.
	 *
	 * @return  void
	 *
	 * @since   1.0.16
	 */
	protected function prepareViewModel($view)
	{
		parent::prepareViewModel($view);

		$layout = $this->input->getCmd('layout', 'default');
		if ($layout !== 'edit' && method_exists($view, 'setModel')) {
			$listModel = $this->getModel('Reviews', 'Administrator', ['ignore_request' => false]);
			if (!$listModel) {
				$className = \Comdev\Component\Onecore\Administrator\Model\Reviews\ReviewsModel::class;
				if (class_exists($className)) {
					$listModel = new $className(['ignore_request' => false], $this->factory);
					if ($listModel instanceof \Joomla\Database\DatabaseAwareInterface) {
						$listModel->setDatabase(\Joomla\CMS\Factory::getContainer()->get(\Joomla\Database\DatabaseInterface::class));
					}
					if ($listModel instanceof \Joomla\CMS\Form\FormFactoryAwareInterface) {
						$listModel->setFormFactory(\Joomla\CMS\Factory::getContainer()->get(\Joomla\CMS\Form\FormFactoryInterface::class));
					}
					if ($listModel instanceof \Joomla\Event\DispatcherAwareInterface) {
						$listModel->setDispatcher(\Joomla\CMS\Factory::getContainer()->get(\Joomla\Event\DispatcherInterface::class));
					}
				}
			}
			if ($listModel) {
				$view->setModel($listModel, true);
			}
		}
	}

	/**
	 * Check access and Plus license before running any task.
	 *
	 * @param   string  $task  The task to perform.
	 *
	 * @return  boolean  True if access is allowed.
	 *
	 * @since   1.0.16
	 */
	protected function checkAccess($task = '')
	{
		if (LicenseHelper::getLicenseType() !== 'plus') {
			$this->setMessage(\Joomla\CMS\Language\Text::_('COM_ONECORE_REVIEWS_PLUS_ONLY'), 'warning');
			$this->setRedirect(Route::_('index.php?option=com_onecore&view=dashboard', false));
			return false;
		}

		return true;
	}

	/**
	 * Method to run a task. Override to enforce Plus license.
	 *
	 * @return  mixed
	 *
	 * @since   1.0.16
	 */
	public function execute($task)
	{
		if (!$this->checkAccess($task)) {
			return false;
		}

		return parent::execute($task);
	}

	/**
	 * Method to get a model object, loading it if required.
	 *
	 * @param   string  $name    The model name. Optional.
	 * @param   string  $prefix  The class prefix. Optional.
	 * @param   array   $config  Configuration array for model. Optional.
	 *
	 * @return  \Joomla\CMS\MVC\Model\BaseDatabaseModel  The model.
	 *
	 * @since   1.0.16
	 */
	public function getModel($name = '', $prefix = '', $config = ['ignore_request' => true])
	{
		$input = $this->app->getInput();
		$task = $input->get('task', '');
		$view = $input->get('view', '');
		$layout = $input->get('layout', '');
		$id = $input->getInt('id', 0);

		$isEditOperation = false;
		if (
			$task === 'review.add' || $task === 'review.edit' || $task === 'review.save' || $task === 'review.apply'
			|| $task === 'review.save2new' || $task === 'review.delete' || $task === 'review.reload'
			|| $layout === 'edit' || ($view === 'review' && $id > 0)
			|| in_array($task, ['add', 'edit', 'save', 'apply', 'save2new', 'reload', 'delete'], true)
		) {
			$isEditOperation = true;
		}

		if (empty($name)) {
			$name = $isEditOperation ? 'Item' : 'Reviews';
		}

		if (($name === 'Reviews' || $name === 'reviews') && $isEditOperation) {
			$name = 'Item';
		}

		if ($name === 'Item' || $name === 'item') {
			if ($this->itemModel) {
				return $this->itemModel;
			}

			$className = 'Comdev\\Component\\Onecore\\Administrator\\Model\\Reviews\\ItemModel';
			if (class_exists($className)) {
				$component = \Joomla\CMS\Factory::getApplication()->bootComponent('com_onecore');
				$factory = $component->getMVCFactory();
				$config['ignore_request'] = $isEditOperation ? false : true;
				$model = new $className($config, $factory);
				if ($model instanceof \Joomla\Database\DatabaseAwareInterface) {
					$model->setDatabase(\Joomla\CMS\Factory::getContainer()->get(\Joomla\Database\DatabaseInterface::class));
				}
				if ($model instanceof \Joomla\CMS\Form\FormFactoryAwareInterface) {
					$model->setFormFactory(\Joomla\CMS\Factory::getContainer()->get(\Joomla\CMS\Form\FormFactoryInterface::class));
				}
				if ($model instanceof \Joomla\Event\DispatcherAwareInterface) {
					$model->setDispatcher(\Joomla\CMS\Factory::getContainer()->get(\Joomla\Event\DispatcherInterface::class));
				}
				$this->itemModel = $model;
				return $model;
			}
		}

		return parent::getModel($name, $prefix, $config);
	}

	/**
	 * Save method. Normalize task for FormController.
	 *
	 * @param   string  $key     The name of the primary key.
	 * @param   string  $urlVar  The name of the URL variable if different from the primary key.
	 *
	 * @return  boolean
	 *
	 * @since   1.0.16
	 */
	public function save($key = null, $urlVar = null)
	{
		$originalTask = $this->input->post->get('task', $this->input->get('task', '', 'cmd'), 'cmd');
		if ($originalTask === 'review.apply') {
			$this->task = 'apply';
			$this->input->set('task', 'apply');
		} elseif ($originalTask === 'review.save') {
			$this->task = 'save';
			$this->input->set('task', 'save');
		} elseif ($originalTask === 'review.save2new') {
			$this->task = 'save2new';
			$this->input->set('task', 'save2new');
		}

		return parent::save($key, $urlVar);
	}

	/**
	 * Method to publish/unpublish reviews.
	 *
	 * @return  void
	 *
	 * @since   1.0.16
	 */
	public function publish()
	{
		$this->checkToken();
		$cid = (array) $this->input->get('cid', [], 'int');
		$cid = array_filter($cid);
		$data = [
			'publish' => 1,
			'unpublish' => 0,
			'reviews.publish' => 1,
			'reviews.unpublish' => 0,
		];
		$task = $this->getTask();
		$value = \Joomla\Utilities\ArrayHelper::getValue($data, $task, 0, 'int');

		if (empty($cid)) {
			$this->setMessage(\Joomla\CMS\Language\Text::_($this->text_prefix . '_NO_ITEM_SELECTED'), 'warning');
		} else {
			$db = \Joomla\CMS\Factory::getDbo();
			$query = $db->getQuery(true)
				->update($db->quoteName('#__one_reviews'))
				->set($db->quoteName('published') . ' = ' . (int) $value)
				->whereIn($db->quoteName('id'), $cid);
			$db->setQuery($query)->execute();
			$ntext = $value == 1 ? $this->text_prefix . '_N_ITEMS_PUBLISHED' : $this->text_prefix . '_N_ITEMS_UNPUBLISHED';
			$this->setMessage(\Joomla\CMS\Language\Text::plural($ntext, \count($cid)));
		}

		$this->setRedirect(Route::_('index.php?option=com_onecore&view=reviews', false));
	}

	/**
	 * Approve/Unapprove reviews.
	 *
	 * @return  void
	 *
	 * @since   1.0.16
	 */
	public function approve()
	{
		$this->checkToken();
		$cid = (array) $this->input->get('cid', [], 'int');
		$cid = array_filter($cid);
		$task = $this->getTask();
		$value = ($task === 'approve' || $task === 'reviews.approve') ? 1 : 0;

		if (empty($cid)) {
			$this->setMessage(\Joomla\CMS\Language\Text::_($this->text_prefix . '_NO_ITEM_SELECTED'), 'warning');
		} else {
			$db = \Joomla\CMS\Factory::getDbo();
			
			// Get reviews data before updating (to check if they were already approved)
			$query = $db->getQuery(true)
				->select('id, name, email, title, body, content_id, event_id, approved')
				->from($db->quoteName('#__one_reviews'))
				->whereIn($db->quoteName('id'), $cid);
			$db->setQuery($query);
			$reviews = $db->loadObjectList();
			
			// Update approved status
			$model = $this->getModel('Reviews', 'Administrator', ['ignore_request' => false]);
			if (method_exists($model, 'approve')) {
				$model->approve($cid, $value);
			} else {
				$query = $db->getQuery(true)
					->update($db->quoteName('#__one_reviews'))
					->set($db->quoteName('approved') . ' = ' . (int) $value)
					->whereIn($db->quoteName('id'), $cid);
				$db->setQuery($query)->execute();
			}
			
			// Send email if approving (value = 1) and review was not already approved
			if ($value == 1) {
				$this->sendApprovalEmails($reviews);
			}
			
			$ntext = $value ? $this->text_prefix . '_N_ITEMS_APPROVED' : $this->text_prefix . '_N_ITEMS_UNAPPROVED';
			$this->setMessage(\Joomla\CMS\Language\Text::plural($ntext, \count($cid)));
		}

		$this->setRedirect(Route::_('index.php?option=com_onecore&view=reviews', false));
	}

	/**
	 * Send approval emails for reviews.
	 *
	 * @param   array  $reviews  Array of review objects
	 *
	 * @return  void
	 *
	 * @since   1.0.18
	 */
	private function sendApprovalEmails($reviews)
	{
		if (empty($reviews)) {
			return;
		}
		
		$db = \Joomla\CMS\Factory::getDbo();
		$app = \Joomla\CMS\Factory::getApplication();
		
		// Get "Review Approved" email template
		$query = $db->getQuery(true)
			->select('id, send_to_user, send_cc_to_admin')
			->from($db->quoteName('#__one_email_templ'))
			->where($db->quoteName('title') . ' = ' . $db->quote('Review Approved'))
			->where($db->quoteName('published') . ' = 1')
			->setLimit(1);
		$db->setQuery($query);
		$template = $db->loadObject();
		
		if (!$template || !$template->send_to_user) {
			return;
		}
		
		$siteName = $app->get('sitename');
		$siteUrl = Uri::root();
		
		foreach ($reviews as $review) {
			// Skip if review was already approved (don't send duplicate emails)
			if ($review->approved == 1) {
				continue;
			}
			
			// Get listing/event title
			$listingTitle = '';
			if ($review->content_id) {
				$query = $db->getQuery(true)
					->select('title')
					->from($db->quoteName('#__one_content'))
					->where($db->quoteName('id') . ' = ' . (int) $review->content_id)
					->setLimit(1);
				$db->setQuery($query);
				$listingTitle = $db->loadResult() ?: '';
			} elseif ($review->event_id) {
				$query = $db->getQuery(true)
					->select('title')
					->from($db->quoteName('#__one_events'))
					->where($db->quoteName('id') . ' = ' . (int) $review->event_id)
					->setLimit(1);
				$db->setQuery($query);
				$listingTitle = $db->loadResult() ?: '';
			}
			
			// Prepare variables for email template
			$variables = [
				'{user}' => $review->name,
				'{user_email}' => $review->email,
				'{listing_title}' => $listingTitle,
				'{event_title}' => $listingTitle, // Same for events
				'{review_title}' => $review->title,
				'{review_body}' => $review->body,
				'{site_name}' => $siteName,
				'{site_url}' => $siteUrl,
			];
			
			// Get admin email for CC
			$adminEmail = $template->send_cc_to_admin ? $app->get('mailfrom') : '';
			
			// Send email
			EmailHelper::sendEmail(
				$template->id,
				$variables,
				$review->email,
				$review->name,
				$adminEmail
			);
		}
	}
}
