<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Controller;

use Comdev\Component\Onecore\Administrator\Helper\LicenseHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Controller\BaseController;
use Joomla\CMS\Router\Route;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * License controller class.
 *
 * @since  1.0.0
 */
class LicenseController extends BaseController
{
	/**
	 * Activate license
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function activate(): void
	{
		// Check for request forgeries
		$this->checkToken();

		$app = Factory::getApplication();
		$input = $app->getInput();
		$licenseKey = trim($input->getString('license_key', ''));

		if (empty($licenseKey)) {
			$app->enqueueMessage(Text::_('COM_ONECORE_LICENSE_KEY_REQUIRED'), 'error');
			$this->setRedirect(Route::_('index.php?option=com_onecore&view=dashboard', false));
			return;
		}

		// Activate license
		$result = LicenseHelper::activateLicense($licenseKey);

		if ($result['success']) {
			$app->enqueueMessage(Text::_('COM_ONECORE_LICENSE_ACTIVATED_SUCCESS'), 'success');
		} else {
			$message = $result['message'] ?? Text::_('COM_ONECORE_LICENSE_ACTIVATION_FAILED');
			$app->enqueueMessage($message, 'error');
		}

		$this->setRedirect(Route::_('index.php?option=com_onecore&view=dashboard', false));
	}

	/**
	 * Deactivate license
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function deactivate(): void
	{
		// Check for request forgeries
		$this->checkToken();

		LicenseHelper::deactivateLicense();

		$app = Factory::getApplication();
		$app->enqueueMessage(Text::_('COM_ONECORE_LICENSE_DEACTIVATED_SUCCESS'), 'success');
		$this->setRedirect(Route::_('index.php?option=com_onecore&view=dashboard', false));
	}

	/**
	 * Check license status
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function check(): void
	{
		// Check for request forgeries
		$this->checkToken();

		$app = Factory::getApplication();
		$params = \Joomla\CMS\Component\ComponentHelper::getParams('com_onecore');
		$licenseKey = $params->get('license_key', '');

		if (empty($licenseKey)) {
			$app->enqueueMessage(Text::_('COM_ONECORE_LICENSE_KEY_NOT_SET'), 'warning');
			$this->setRedirect(Route::_('index.php?option=com_onecore&view=dashboard', false));
			return;
		}

		// Force fresh check by clearing cache
		$params->set('license_last_check', 0);
		$component = \Joomla\CMS\Component\ComponentHelper::getComponent('com_onecore');
		$table = \Joomla\CMS\Table\Table::getInstance('Extension');
		$table->load($component->id);
		$table->params = $params->toString();
		$table->store();

		// Check license - this will update expiration date internally
		$status = LicenseHelper::checkLicenseStatus($licenseKey);
		
		// Reload params to get the expiration date that was saved by checkLicenseStatus()
		$params = \Joomla\CMS\Component\ComponentHelper::getParams('com_onecore');
		$expires = $params->get('license_expires', null);
		
		// Update license status with the expiration date from API
		LicenseHelper::updateLicenseStatus($licenseKey, $status, $expires);
		
		// Get updated params to show expiration info in message
		$params = \Joomla\CMS\Component\ComponentHelper::getParams('com_onecore');
		$expiresFormatted = '';
		if ($expires > 0) {
			$expiresFormatted = Factory::getDate($expires)->format('Y-m-d');
		}

		if ($status === 'active') {
			$message = Text::_('COM_ONECORE_LICENSE_STATUS_ACTIVE');
			if (!empty($expiresFormatted)) {
				$message .= ' ' . Text::sprintf('COM_ONECORE_LICENSE_EXPIRES_ON', $expiresFormatted);
			}
			$app->enqueueMessage($message, 'success');
		} elseif ($status === 'expired') {
			$app->enqueueMessage(Text::_('COM_ONECORE_LICENSE_STATUS_EXPIRED'), 'warning');
		} else {
			$app->enqueueMessage(Text::_('COM_ONECORE_LICENSE_STATUS_INACTIVE'), 'error');
		}

		$this->setRedirect(Route::_('index.php?option=com_onecore&view=dashboard', false));
	}
}
