<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Controller;

use Comdev\Component\Onecore\Administrator\Helper\LicenseHelper;
use Joomla\CMS\MVC\Controller\BaseController;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Session\Session;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Import/Export controller class.
 *
 * @since  1.0.13
 */
class ImportexportController extends BaseController
{
	/**
	 * The default view.
	 *
	 * @var    string
	 * @since  1.0.13
	 */
	protected $default_view = 'importexport';

	/**
	 * Export data to CSV
	 *
	 * @return  void
	 *
	 * @since   1.0.13
	 */
	public function export()
	{
		// Check for request forgeries
		Session::checkToken() or jexit(Text::_('JINVALID_TOKEN'));

		if (LicenseHelper::getLicenseType() !== 'plus') {
			Factory::getApplication()->enqueueMessage(Text::_('COM_ONECORE_IMPORT_EXPORT_PLUS_ONLY'), 'warning');
			$this->setRedirect(Route::_('index.php?option=com_onecore&view=importexport', false));
			return;
		}

		$app = Factory::getApplication();
		$type = $app->input->getCmd('type', 'content'); // content or events

		// Get the model
		$model = $this->getModel('Importexport');

		if (!$model) {
			$app->enqueueMessage(Text::_('COM_ONECORE_ERROR_MODEL_NOT_FOUND'), 'error');
			$this->setRedirect(Route::_('index.php?option=com_onecore&view=importexport', false));
			return;
		}

		// Disable error display to prevent HTML output
		$errorReporting = error_reporting(0);
		$displayErrors = ini_get('display_errors');
		ini_set('display_errors', '0');
		
		try {
			$csvData = $model->exportToCsv($type);

			// Set headers for CSV download
			$filename = 'onecore_' . $type . '_export_' . date('Y-m-d_His') . '.csv';
			
			// Clear any previous output
			if (ob_get_level()) {
				ob_clean();
			}
			$app->clearHeaders();
			
			// Set headers
			$app->setHeader('Content-Type', 'text/csv; charset=utf-8', true);
			$app->setHeader('Content-Disposition', 'attachment; filename="' . $filename . '"', true);
			$app->setHeader('Pragma', 'no-cache', true);
			$app->setHeader('Expires', '0', true);
			$app->sendHeaders();

			// Output CSV
			$output = fopen('php://output', 'w');

			// Add BOM for UTF-8
			fprintf($output, chr(0xEF) . chr(0xBB) . chr(0xBF));

			foreach ($csvData as $row) {
				fputcsv($output, $row);
			}

			fclose($output);
			
			// Restore error reporting
			error_reporting($errorReporting);
			ini_set('display_errors', $displayErrors);
			
			$app->close();
		} catch (\Exception $e) {
			// Restore error reporting
			error_reporting($errorReporting);
			ini_set('display_errors', $displayErrors);
			
			$app->enqueueMessage(Text::sprintf('COM_ONECORE_EXPORT_ERROR', $e->getMessage()), 'error');
			$this->setRedirect(Route::_('index.php?option=com_onecore&view=importexport', false));
		}
	}

	/**
	 * Import data from CSV
	 *
	 * @return  void
	 *
	 * @since   1.0.13
	 */
	public function import()
	{
		// Check for request forgeries
		Session::checkToken() or jexit(Text::_('JINVALID_TOKEN'));

		if (LicenseHelper::getLicenseType() !== 'plus') {
			Factory::getApplication()->enqueueMessage(Text::_('COM_ONECORE_IMPORT_EXPORT_PLUS_ONLY'), 'warning');
			$this->setRedirect(Route::_('index.php?option=com_onecore&view=importexport', false));
			return;
		}

		$app = Factory::getApplication();
		$type = $app->input->getCmd('type', 'content'); // content or events
		$file = $app->input->files->get('csv_file', null, 'raw');

		if (!$file || $file['error'] !== UPLOAD_ERR_OK) {
			$app->enqueueMessage(Text::_('COM_ONECORE_IMPORT_ERROR_NO_FILE'), 'error');
			$this->setRedirect(Route::_('index.php?option=com_onecore&view=importexport', false));
			return;
		}

		// Get the model
		$model = $this->getModel('Importexport');

		if (!$model) {
			$app->enqueueMessage(Text::_('COM_ONECORE_ERROR_MODEL_NOT_FOUND'), 'error');
			$this->setRedirect(Route::_('index.php?option=com_onecore&view=importexport', false));
			return;
		}

		try {
			$result = $model->importFromCsv($file['tmp_name'], $type);

			if ($result['success']) {
				$app->enqueueMessage(
					Text::sprintf('COM_ONECORE_IMPORT_SUCCESS', $result['imported'], $result['total'], $result['errors']),
					$result['errors'] > 0 ? 'warning' : 'success'
				);
			} else {
				$app->enqueueMessage(Text::sprintf('COM_ONECORE_IMPORT_ERROR', $result['message']), 'error');
			}
		} catch (\Exception $e) {
			$app->enqueueMessage(Text::sprintf('COM_ONECORE_IMPORT_ERROR', $e->getMessage()), 'error');
		}

		$this->setRedirect(Route::_('index.php?option=com_onecore&view=importexport', false));
	}
}
