<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Controller;

use Comdev\Component\Onecore\Administrator\Helper\LicenseHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Controller\FormController;
use Joomla\CMS\Router\Route;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Events controller class.
 *
 * @since  1.0.0
 */
class EventsController extends FormController
{
	/**
	 * Cached Item model instance for edit operations.
	 *
	 * @var    \Joomla\CMS\MVC\Model\BaseDatabaseModel|null
	 * @since  1.0.0
	 */
	protected $itemModel = null;
	/**
	 * The prefix to use with controller messages.
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	protected $text_prefix = 'COM_ONECORE_EVENTS';

	/**
	 * The URL view item variable.
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	protected $view_item = 'events';

	/**
	 * The URL view list variable.
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	protected $view_list = 'events';

	/**
	 * Constructor.
	 *
	 * @param   array                 $config   An optional associative array of configuration settings.
	 * @param   ?MVCFactoryInterface  $factory  The factory.
	 * @param   ?CMSApplication       $app      The Application for the dispatcher
	 * @param   ?Input                $input    Input
	 *
	 * @since   1.0.0
	 */
	public function __construct($config = [], $factory = null, $app = null, $input = null)
	{
		parent::__construct($config, $factory, $app, $input);

		// Register publish/unpublish tasks
		$this->registerTask('unpublish', 'publish');
		
		// Register featured/unfeatured tasks
		$this->registerTask('unfeatured', 'featured');
		
		// 🔑 KLUCZ DO APPLY - map events.apply to save (FormController handles apply vs save logic)
		$this->registerTask('events.apply', 'save');
		$this->registerTask('events.save2new', 'save');
		$this->registerTask('events.save2copy', 'save');
		$this->registerTask('events.delete', 'delete');
	}

	public function save($key = null, $urlVar = null)
	{
		// Get form data before parent::save() processes it
		$data = $this->input->post->get('jform', [], 'array');
		
		// Determine the name of the primary key
		$model = $this->getModel();
		$table = $model->getTable();
		if (empty($key)) {
			$key = $table->getKeyName();
		}
		if (empty($urlVar)) {
			$urlVar = $key;
		}
		
		$recordId = (int) $this->input->getInt($urlVar, 0);
		
		// Check if this is a new record (not editing existing)
		$isNewRecord = ($recordId === 0);
		
		// Check if trying to enable recurring event (Plus feature)
		$recurrenceEnabled = (int) ($data['recurrence_enabled'] ?? 0);
		$isTryingToEnableRecurrence = ($recurrenceEnabled === 1);
		
		// Check if existing event already has recurrence enabled
		$hasExistingRecurrence = false;
		if (!$isNewRecord && $recordId > 0) {
			$db = \Joomla\CMS\Factory::getDbo();
			$query = $db->getQuery(true)
				->select('COUNT(*)')
				->from($db->quoteName('#__one_event_recurrences'))
				->where($db->quoteName('event_id') . ' = ' . $db->quote($recordId));
			$db->setQuery($query);
			$hasExistingRecurrence = (bool) $db->loadResult();
		}
		
		// If trying to enable recurrence for new event or existing event without recurrence, and license expired, block it
		if ($isTryingToEnableRecurrence && LicenseHelper::isLicenseExpired() && ($isNewRecord || !$hasExistingRecurrence)) {
			$this->app->enqueueMessage(
				Text::_('COM_ONECORE_LICENSE_EXPIRED_CANNOT_ADD_PLUS_FEATURES'),
				'error'
			);
			
			// Save form data in session to preserve user input
			$context = $this->option . '.edit.' . $this->context;
			$this->app->setUserState($context . '.data', $data);
			
			// Redirect back to edit form
			$this->setRedirect(
				Route::_(
					'index.php?option=' . $this->option . '&view=' . $this->view_item . '&layout=edit' . ($recordId > 0 ? '&id=' . $recordId : ''),
					false
				)
			);
			
			return false;
		}
		
		// Get original task from POST/GET (before registerTask normalization)
		$originalTask = $this->input->post->get('task', $this->input->get->get('task', '', 'cmd'), 'cmd');
		
		// Normalize task names - FormController expects 'apply' or 'save', not 'events.apply' or 'events.save'
		// registerTask already routes them to save(), but we need to set the correct task for FormController
		if ($originalTask === 'events.apply') {
			// 'apply' stays in edit view - FormController handles this
			$this->task = 'apply';
			$this->input->set('task', 'apply');
		} elseif ($originalTask === 'events.save') {
			// 'save' goes to list view - FormController handles this
			$this->task = 'save';
			$this->input->set('task', 'save');
		} elseif ($originalTask === 'events.save2new') {
			$this->task = 'save2new';
			$this->input->set('task', 'save2new');
		} elseif ($originalTask === 'events.save2copy') {
			$this->task = 'save2copy';
			$this->input->set('task', 'save2copy');
		}
		
		return parent::save($key, $urlVar);
	}

	/**
	 * Function that allows child controller access to model data
	 * after the data has been saved.
	 *
	 * @param   \Joomla\CMS\MVC\Model\BaseDatabaseModel  $model      The data model object.
	 * @param   array                                      $validData  The validated data.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	protected function postSaveHook(\Joomla\CMS\MVC\Model\BaseDatabaseModel $model, $validData = [])
	{
		// If apply task, ensure redirect has ID
		if ($this->getTask() === 'apply') {
			$recordId = (int) $model->getState($model->getName() . '.id');
			
			// If ID is 0, try to get it from table
			if ($recordId <= 0) {
				$table = $model->getTable();
				$recordId = (int) ($table->id ?? 0);
			}
			
			// Fallback: try insertid() for new records
			if ($recordId <= 0) {
				try {
					$recordId = (int) \Joomla\CMS\Factory::getDbo()->insertid();
				} catch (\Exception $e) {
					$recordId = 0;
				}
			}
			
			// Fallback: try from validData
			if ($recordId <= 0 && isset($validData['id'])) {
				$recordId = (int) $validData['id'];
			}
			
			// If we found ID, ensure redirect includes it
			if ($recordId > 0) {
				$context = $this->option . '.edit.' . $this->context;
				$this->holdEditId($context, $recordId);
				$this->app->setUserState($context . '.data', null);
				
				$this->setRedirect(
					\Joomla\CMS\Router\Route::_(
						'index.php?option=' . $this->option . '&view=' . $this->view_item
						. $this->getRedirectToItemAppend($recordId, 'id'),
						false
					)
				);
			}
		}
	}

	/**
	 * Method to get a model object, loading it if required.
	 *
	 * @param   string  $name    The model name. Optional.
	 * @param   string  $prefix  The class prefix. Optional.
	 * @param   array   $config  Configuration array for model. Optional.
	 *
	 * @return  \Joomla\CMS\MVC\Model\BaseDatabaseModel  The model.
	 *
	 * @since   1.0.0
	 */
	public function getModel($name = '', $prefix = '', $config = ['ignore_request' => true])
	{
		$input = $this->app->getInput();
		
		$task = $input->get('task', '');
		$view = $input->get('view', '');
		$layout = $input->get('layout', '');
		$id = $input->get('id', 0);
		
		$isEditOperation = false;
		if ($task === 'events.add' || 
			$task === 'events.edit' || 
			$task === 'events.save' || 
			$task === 'events.apply' || 
			$task === 'events.delete' ||
			$task === 'events.save2new' || 
			$task === 'events.save2copy' ||
			$task === 'events.reload' ||
			$layout === 'edit' ||
			($view === 'event' && $id > 0) ||
			in_array($task, ['add', 'edit', 'save', 'apply', 'save2new', 'save2copy', 'reload', 'delete'])) {
			$isEditOperation = true;
		}
		
		if (empty($name)) {
			if ($isEditOperation) {
				$name = 'Item';
			} else {
				$name = $this->context; // 'events' for list operations
			}
		}
		
		if (($name === 'events' || $name === 'Events') && $isEditOperation) {
			$name = 'Item';
		}
		
		if ($name === 'Item' || $name === 'item') {
			if ($this->itemModel) {
				return $this->itemModel;
			}

			$className = 'Comdev\\Component\\Onecore\\Administrator\\Model\\Event\\ItemModel';
			if (class_exists($className)) {
				$component = \Joomla\CMS\Factory::getApplication()->bootComponent('com_onecore');
				$factory = $component->getMVCFactory();
				
				if ($isEditOperation) {
					$config['ignore_request'] = false;
				}
				
				$model = new $className($config, $factory);
				
				if ($model instanceof \Joomla\Database\DatabaseAwareInterface) {
					$model->setDatabase(\Joomla\CMS\Factory::getContainer()->get(\Joomla\Database\DatabaseInterface::class));
				}
				if ($model instanceof \Joomla\CMS\Form\FormFactoryAwareInterface) {
					$model->setFormFactory(\Joomla\CMS\Factory::getContainer()->get(\Joomla\CMS\Form\FormFactoryInterface::class));
				}
				if ($model instanceof \Joomla\Event\DispatcherAwareInterface) {
					$model->setDispatcher(\Joomla\CMS\Factory::getContainer()->get(\Joomla\Event\DispatcherInterface::class));
				}
				
				$this->itemModel = $model;
				return $model;
			}
		}
		
		return parent::getModel($name, $prefix, $config);
	}

	/**
	 * Method to publish a list of items.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function publish()
	{
		// Check for request forgeries
		$this->checkToken();

		// Get items to publish from the request
		$cid = (array) $this->input->get('cid', [], 'int');
		$data = ['publish' => 1, 'unpublish' => 0, 'trash' => -2];
		$task = $this->getTask();
		$value = \Joomla\Utilities\ArrayHelper::getValue($data, $task, 0, 'int');

		// Remove zero values resulting from input filter
		$cid = array_filter($cid);

		if (empty($cid)) {
			\Joomla\CMS\Factory::getApplication()->enqueueMessage(
				\Joomla\CMS\Language\Text::_($this->text_prefix . '_NO_ITEM_SELECTED'),
				'warning'
			);
		} else {
			// Get the model
			$model = $this->getModel('Events', 'Administrator', ['ignore_request' => false]);

			// Publish the items
			try {
				if (!$model->publish($cid, $value)) {
					\Joomla\CMS\Factory::getApplication()->enqueueMessage(
						implode('<br>', $model->getErrors()),
						'error'
					);
				} else {
					$errors = $model->getErrors();
					$messageType = 'message';

					if ($value == 1) {
						if ($errors) {
							$messageType = 'error';
							$ntext = $this->text_prefix . '_N_ITEMS_FAILED_PUBLISHING';
						} else {
							$ntext = $this->text_prefix . '_N_ITEMS_PUBLISHED';
						}
					} elseif ($value == 0) {
						$ntext = $this->text_prefix . '_N_ITEMS_UNPUBLISHED';
					} else {
						$ntext = $this->text_prefix . '_N_ITEMS_TRASHED';
					}

					$this->setMessage(\Joomla\CMS\Language\Text::plural($ntext, \count($cid)), $messageType);
				}
			} catch (\Exception $e) {
				$this->setMessage($e->getMessage(), 'error');
			}
		}

		$this->setRedirect(
			\Joomla\CMS\Router\Route::_(
				'index.php?option=' . $this->option . '&view=' . $this->view_list,
				false
			)
		);
	}

	/**
	 * Removes an item.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function delete()
	{
		// Check for request forgeries
		$this->checkToken();

		// Get items to remove from the request.
		$cid = (array) $this->input->get('cid', [], 'int');

		// Remove zero values resulting from input filter
		$cid = array_filter($cid);

		if (empty($cid)) {
			$this->getLogger()->warning(\Joomla\CMS\Language\Text::_($this->text_prefix . '_NO_ITEM_SELECTED'), ['category' => 'jerror']);
		} else {
			// Get the model (Item model)
			$model = $this->getModel('Item', '', ['ignore_request' => true]);

			// Remove the items.
			if ($model->delete($cid)) {
				$this->setMessage(\Joomla\CMS\Language\Text::plural($this->text_prefix . '_N_ITEMS_DELETED', \count($cid)));
			} else {
				$this->setMessage($model->getError(), 'error');
			}
		}

		$this->setRedirect(
			\Joomla\CMS\Router\Route::_(
				'index.php?option=' . $this->option . '&view=' . $this->view_list,
				false
			)
		);
	}

	/**
	 * Method to toggle the featured setting of a list of items.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function featured()
	{
		// Check for request forgeries
		$this->checkToken();

		// Get items to toggle from the request
		$cid = (array) $this->input->get('cid', [], 'int');
		$data = ['featured' => 1, 'unfeatured' => 0];
		$task = $this->getTask();
		$value = \Joomla\Utilities\ArrayHelper::getValue($data, $task, 0, 'int');

		// Remove zero values resulting from input filter
		$cid = array_filter($cid);

		if (empty($cid)) {
			\Joomla\CMS\Factory::getApplication()->enqueueMessage(
				\Joomla\CMS\Language\Text::_($this->text_prefix . '_NO_ITEM_SELECTED'),
				'warning'
			);
		} else {
			// Get the model
			$model = $this->getModel('Events', 'Administrator', ['ignore_request' => false]);

			// Toggle featured status
			try {
				if (!$model->featured($cid, $value)) {
					\Joomla\CMS\Factory::getApplication()->enqueueMessage(
						implode('<br>', $model->getErrors()),
						'error'
					);
				} else {
					$errors = $model->getErrors();
					$messageType = 'message';

					if ($value == 1) {
						if ($errors) {
							$messageType = 'error';
							$ntext = $this->text_prefix . '_N_ITEMS_FAILED_FEATURING';
						} else {
							$ntext = $this->text_prefix . '_N_ITEMS_FEATURED';
						}
					} else {
						$ntext = $this->text_prefix . '_N_ITEMS_UNFEATURED';
					}

					$this->setMessage(\Joomla\CMS\Language\Text::plural($ntext, \count($cid)), $messageType);
				}
			} catch (\Exception $e) {
				$this->setMessage($e->getMessage(), 'error');
			}
		}

		$this->setRedirect(
			\Joomla\CMS\Router\Route::_(
				'index.php?option=' . $this->option . '&view=' . $this->view_list,
				false
			)
		);
	}
}
