<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Controller;

use Joomla\CMS\MVC\Controller\FormController;
use Joomla\CMS\Router\Route;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Email Templates controller class.
 *
 * @since  1.0.17
 */
class EmailtemplatesController extends FormController
{
	/**
	 * Cached Item model instance for edit operations.
	 *
	 * @var    \Joomla\CMS\MVC\Model\BaseDatabaseModel|null
	 * @since  1.0.17
	 */
	protected $itemModel = null;

	/**
	 * The prefix to use with controller messages.
	 *
	 * @var    string
	 * @since  1.0.17
	 */
	protected $text_prefix = 'COM_ONECORE_EMAIL_TEMPLATES';

	/**
	 * The URL view item variable.
	 *
	 * @var    string
	 * @since  1.0.17
	 */
	protected $view_item = 'emailtemplates';

	/**
	 * The URL view list variable.
	 *
	 * @var    string
	 * @since  1.0.17
	 */
	protected $view_list = 'emailtemplates';

	/**
	 * Constructor.
	 *
	 * @param   array                 $config   An optional associative array of configuration settings.
	 * @param   ?MVCFactoryInterface  $factory  The factory.
	 * @param   ?CMSApplication       $app      The Application for the dispatcher
	 * @param   ?Input                $input    Input
	 *
	 * @since   1.0.17
	 */
	public function __construct($config = [], $factory = null, $app = null, $input = null)
	{
		// Set default view before parent constructor
		if (empty($config['default_view'])) {
			$config['default_view'] = 'emailtemplates';
		}

		parent::__construct($config, $factory, $app, $input);

		$this->registerTask('emailtemplate.apply', 'save');
		$this->registerTask('emailtemplates.edit', 'edit');
		$this->registerTask('emailtemplates.preview', 'preview');
		$this->registerTask('emailtemplate.preview', 'preview');
		$this->registerTask('unpublish', 'publish');
		$this->registerTask('emailtemplates.publish', 'publish');
		$this->registerTask('emailtemplates.unpublish', 'publish');
		$this->registerTask('emailtemplate.cancel', 'cancel');
		$this->registerTask('emailtemplates.cancel', 'cancel');
	}

	/**
	 * Prepare the view and ensure list model is set for list layout.
	 *
	 * @param   \Joomla\CMS\MVC\View\ViewInterface  $view  The view object.
	 *
	 * @return  void
	 *
	 * @since   1.0.17
	 */
	protected function prepareViewModel($view)
	{
		parent::prepareViewModel($view);

		$layout = $this->input->getCmd('layout', 'default');
		if ($layout !== 'edit' && method_exists($view, 'setModel')) {
			$listModel = $this->getModel('EmailTemplates', 'Administrator', ['ignore_request' => false]);
			if (!$listModel) {
				$className = \Comdev\Component\Onecore\Administrator\Model\EmailTemplates\EmailTemplatesModel::class;
				if (class_exists($className)) {
					$listModel = new $className(['ignore_request' => false], $this->factory);
					if ($listModel instanceof \Joomla\Database\DatabaseAwareInterface) {
						$listModel->setDatabase(\Joomla\CMS\Factory::getContainer()->get(\Joomla\Database\DatabaseInterface::class));
					}
					if ($listModel instanceof \Joomla\CMS\Form\FormFactoryAwareInterface) {
						$listModel->setFormFactory(\Joomla\CMS\Factory::getContainer()->get(\Joomla\CMS\Form\FormFactoryInterface::class));
					}
					if ($listModel instanceof \Joomla\Event\DispatcherAwareInterface) {
						$listModel->setDispatcher(\Joomla\CMS\Factory::getContainer()->get(\Joomla\Event\DispatcherInterface::class));
					}
				}
			}
			if ($listModel) {
				$view->setModel($listModel, true);
			}
		}
	}

	/**
	 * Method to get a model object, loading it if required.
	 *
	 * @param   string  $name    The model name. Optional.
	 * @param   string  $prefix  The class prefix. Optional.
	 * @param   array   $config  Configuration array for model. Optional.
	 *
	 * @return  \Joomla\CMS\MVC\Model\BaseDatabaseModel  The model.
	 *
	 * @since   1.0.17
	 */
	public function getModel($name = '', $prefix = '', $config = ['ignore_request' => true])
	{
		$input = $this->app->getInput();
		$task = $input->get('task', '');
		$view = $input->get('view', '');
		$layout = $input->get('layout', '');
		$id = $input->getInt('id', 0);

		$isEditOperation = false;
		if (
			$task === 'emailtemplate.edit' || $task === 'emailtemplate.save' || $task === 'emailtemplate.apply'
			|| $task === 'emailtemplate.delete' || $task === 'emailtemplate.reload'
			|| $layout === 'edit' || ($view === 'emailtemplates' && $id > 0)
			|| in_array($task, ['edit', 'save', 'apply', 'reload', 'delete'], true)
		) {
			$isEditOperation = true;
		}

		if (empty($name)) {
			$name = $isEditOperation ? 'Item' : 'EmailTemplates';
		}

		if (($name === 'EmailTemplates' || $name === 'emailtemplates') && $isEditOperation) {
			$name = 'Item';
		}

		if ($name === 'Item' || $name === 'item') {
			if ($this->itemModel) {
				return $this->itemModel;
			}

			$className = 'Comdev\\Component\\Onecore\\Administrator\\Model\\EmailTemplates\\ItemModel';
			if (class_exists($className)) {
				$component = \Joomla\CMS\Factory::getApplication()->bootComponent('com_onecore');
				$factory = $component->getMVCFactory();
				$config['ignore_request'] = $isEditOperation ? false : true;
				$model = new $className($config, $factory);
				if ($model instanceof \Joomla\Database\DatabaseAwareInterface) {
					$model->setDatabase(\Joomla\CMS\Factory::getContainer()->get(\Joomla\Database\DatabaseInterface::class));
				}
				if ($model instanceof \Joomla\CMS\Form\FormFactoryAwareInterface) {
					$model->setFormFactory(\Joomla\CMS\Factory::getContainer()->get(\Joomla\CMS\Form\FormFactoryInterface::class));
				}
				if ($model instanceof \Joomla\Event\DispatcherAwareInterface) {
					$model->setDispatcher(\Joomla\CMS\Factory::getContainer()->get(\Joomla\Event\DispatcherInterface::class));
				}
				$this->itemModel = $model;
				return $model;
			}
		}

		return parent::getModel($name, $prefix, $config);
	}

	/**
	 * Save method. Normalize task for FormController.
	 *
	 * @param   string  $key     The name of the primary key.
	 * @param   string  $urlVar  The name of the URL variable if different from the primary key.
	 *
	 * @return  boolean
	 *
	 * @since   1.0.17
	 */
	public function save($key = null, $urlVar = null)
	{
		$originalTask = $this->input->post->get('task', $this->input->get('task', '', 'cmd'), 'cmd');
		if ($originalTask === 'emailtemplate.apply') {
			$this->task = 'apply';
			$this->input->set('task', 'apply');
		} elseif ($originalTask === 'emailtemplate.save') {
			$this->task = 'save';
			$this->input->set('task', 'save');
		}

		return parent::save($key, $urlVar);
	}

	/**
	 * Method to display the view.
	 *
	 * @param   boolean  $cachable   If true, the view output will be cached
	 * @param   array    $urlparams  An array of safe URL parameters and their variable types
	 *
	 * @return  static  A \JControllerLegacy object to support chaining.
	 *
	 * @since   1.0.17
	 */
	public function display($cachable = false, $urlparams = [])
	{
		// Ensure view parameter is set to 'emailtemplates' if not set or if it's 'emailtemplate' (singular)
		$view = $this->input->get('view', '');
		if (empty($view) || $view === 'emailtemplate') {
			$this->input->set('view', 'emailtemplates');
		}

		return parent::display($cachable, $urlparams);
	}

	/**
	 * Method to publish/unpublish email templates.
	 *
	 * @return  void
	 *
	 * @since   1.0.17
	 */
	public function publish()
	{
		$this->checkToken();
		$cid = (array) $this->input->get('cid', [], 'int');
		$cid = array_filter($cid);
		$data = [
			'publish' => 1,
			'unpublish' => 0,
			'emailtemplates.publish' => 1,
			'emailtemplates.unpublish' => 0,
		];
		$task = $this->getTask();
		$value = \Joomla\Utilities\ArrayHelper::getValue($data, $task, 0, 'int');

		if (empty($cid)) {
			$this->setMessage(\Joomla\CMS\Language\Text::_($this->text_prefix . '_NO_ITEM_SELECTED'), 'warning');
		} else {
			$db = \Joomla\CMS\Factory::getDbo();
			$query = $db->getQuery(true)
				->update($db->quoteName('#__one_email_templ'))
				->set($db->quoteName('published') . ' = ' . (int) $value)
				->whereIn($db->quoteName('id'), $cid);
			$db->setQuery($query)->execute();
			$ntext = $value == 1 ? $this->text_prefix . '_N_ITEMS_PUBLISHED' : $this->text_prefix . '_N_ITEMS_UNPUBLISHED';
			$this->setMessage(\Joomla\CMS\Language\Text::plural($ntext, \count($cid)));
		}

		$this->setRedirect(Route::_('index.php?option=com_onecore&view=emailtemplates', false));
	}

	/**
	 * Override add method to prevent creating new templates.
	 *
	 * @return  void
	 *
	 * @since   1.0.17
	 */
	public function add()
	{
		$this->setMessage(\Joomla\CMS\Language\Text::_('COM_ONECORE_EMAIL_TEMPLATES_NO_NEW_TEMPLATES'), 'warning');
		$this->setRedirect(Route::_('index.php?option=com_onecore&view=emailtemplates', false));
	}

	/**
	 * Preview email template with sample variables.
	 *
	 * @return  void
	 *
	 * @since   1.0.17
	 */
	public function preview()
	{
		// Check token for POST requests, but allow GET for AJAX
		if ($this->input->getMethod() === 'POST') {
			$this->checkToken();
		}
		
		$id = $this->input->getInt('id', 0);

		if (!$id) {
			$app = \Joomla\CMS\Factory::getApplication();
			$app->setHeader('Content-Type', 'application/json', true);
			$app->sendHeaders();
			echo json_encode([
				'success' => false,
				'message' => \Joomla\CMS\Language\Text::_('COM_ONECORE_EMAIL_TEMPLATE_INVALID_ID'),
			]);
			$app->close();
			return;
		}

		try {
			$model = $this->getModel('Item', 'Administrator', ['ignore_request' => true]);
			$preview = $model->getPreview($id);

			if (!$preview) {
				$app = \Joomla\CMS\Factory::getApplication();
				$app->setHeader('Content-Type', 'application/json', true);
				$app->sendHeaders();
				echo json_encode([
					'success' => false,
					'message' => \Joomla\CMS\Language\Text::_('COM_ONECORE_EMAIL_TEMPLATE_NOT_FOUND'),
				]);
				$app->close();
				return;
			}

			// Output JSON for AJAX response
			$app = \Joomla\CMS\Factory::getApplication();
			$app->setHeader('Content-Type', 'application/json', true);
			$app->sendHeaders();
			echo json_encode([
				'success' => true,
				'title' => $preview->email_title ?? '',
				'content' => $preview->email_content ?? '',
			]);
			$app->close();
		} catch (\Exception $e) {
			$app = \Joomla\CMS\Factory::getApplication();
			$app->setHeader('Content-Type', 'application/json', true);
			$app->sendHeaders();
			echo json_encode([
				'success' => false,
				'message' => $e->getMessage(),
			]);
			$app->close();
		}
	}

	/**
	 * Cancel edit operation.
	 *
	 * @param   string  $key  The name of the primary key of the URL variable.
	 *
	 * @return  boolean  True if access level checks pass, false otherwise.
	 *
	 * @since   1.0.17
	 */
	public function cancel($key = null)
	{
		// Ensure view parameter is set correctly
		$this->input->set('view', 'emailtemplates');
		return parent::cancel($key);
	}
}
