<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Controller;

use Joomla\CMS\Factory;
use Joomla\CMS\Http\HttpFactory;
use Joomla\CMS\MVC\Controller\BaseController;
use Joomla\CMS\Response\JsonResponse;
use Joomla\Registry\Registry;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Component Controller
 *
 * @since  1.0.0
 */
class DisplayController extends BaseController
{
	/**
	 * The default view.
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	protected $default_view = 'dashboard';

	/**
	 * Get chart data via AJAX
	 *
	 * @return  void
	 *
	 * @since   1.0.10
	 */
	public function getChartData()
	{
		try {
			// Check token
			$this->checkToken('get');

			// Set format to json
			$this->app->setHeader('Content-Type', 'application/json; charset=utf-8');

			// Load model using MVCFactory
			$factory = $this->app->bootComponent('com_onecore')->getMVCFactory();
			$model = $factory->createModel('Dashboard', 'Administrator', ['ignore_request' => true]);
			
			if (!$model) {
				// Fallback: try direct loading
				$className = 'Comdev\\Component\\Onecore\\Administrator\\Model\\Dashboard\\DashboardModel';
				if (!class_exists($className)) {
					$modelPath = JPATH_ADMINISTRATOR . '/components/com_onecore/src/Model/Dashboard/DashboardModel.php';
					if (file_exists($modelPath)) {
						require_once $modelPath;
					}
				}
				
				if (class_exists($className)) {
					$db = Factory::getDbo();
					$model = new $className(['dbo' => $db]);
				}
			}
			
			if (!$model || !method_exists($model, 'getStatistics')) {
				throw new \Exception('Dashboard model could not be loaded. Model: ' . ($model ? get_class($model) : 'null'));
			}
			
			$stats = $model->getStatistics();

			// Prepare chart data
			$chartData = [
				'labels' => [],
				'data' => [],
				'max' => 0
			];

			if (!empty($stats['listings_per_day']) && is_array($stats['listings_per_day'])) {
				foreach ($stats['listings_per_day'] as $dayData) {
					if (isset($dayData['day']) && isset($dayData['count'])) {
						$chartData['labels'][] = $dayData['day'];
						$chartData['data'][] = (int) $dayData['count'];
						if ($dayData['count'] > $chartData['max']) {
							$chartData['max'] = (int) $dayData['count'];
						}
					}
				}
			}

			// Ensure max is at least 1 to avoid division by zero
			if ($chartData['max'] == 0) {
				$chartData['max'] = 1;
			}

			// Return JSON response - JsonResponse wraps data in 'data' property
			$response = new JsonResponse($chartData, null, false, true);
			echo $response;
			$this->app->close();

		} catch (\Exception $e) {
			// Log error
			Factory::getApplication()->getLogger()->error(
				'OneCore Chart Data Error: ' . $e->getMessage(),
				['category' => 'onecore', 'exception' => $e]
			);

			// Return error response
			$this->app->setHeader('Status', '500 Internal Server Error', true);
			echo new JsonResponse([
				'success' => false,
				'message' => 'Error loading chart data: ' . $e->getMessage(),
				'data' => [
					'labels' => [],
					'data' => [],
					'max' => 1
				]
			]);
			$this->app->close();
		}
	}

	/**
	 * Check for component updates by fetching version from external URL
	 *
	 * @return  void
	 *
	 * @since   1.0.13
	 */
	public function checkVersion()
	{
		try {
			// Check token
			$this->checkToken('get');

			// Set format to json
			$this->app->setHeader('Content-Type', 'application/json; charset=utf-8');

			// Get current installed version
			$xmlPath = JPATH_ADMINISTRATOR . '/components/com_onecore/onecore.xml';
			$currentVersion = '1.0.0';
			if (file_exists($xmlPath)) {
				$xml = simplexml_load_file($xmlPath);
				if ($xml && isset($xml->version)) {
					$currentVersion = (string) $xml->version;
				}
			}

			// Fetch version from external URL
			$updateUrl = 'https://comdev.eu/onecore-all-in-one-joomla-directory-extension/';
			$latestVersion = null;
			$error = null;

			// Use Joomla HTTP client
			$httpOptions = new Registry([
				'timeout' => 10,
				'userAgent' => 'Joomla/OneCore Update Checker'
			]);
			$http = (new HttpFactory)->getHttp($httpOptions);

			try {
				$response = $http->get($updateUrl);

				if ($response->getStatusCode() === 200) {
					$html = (string) $response->getBody();

					// Prefer dedicated ID/class so the page can expose version reliably (comdev.eu)
					// 1. id="onecore-current-version" – recommended on the product page
					if (preg_match('/id=["\']onecore-current-version["\'][^>]*>[\s]*(?:<[^>]+>[\s]*)?([\d]+\.[\d]+\.[\d]+)/iu', $html, $matches)) {
						$latestVersion = trim($matches[1]);
					}
					// 2. class="onecore-version" – alternative
					elseif (preg_match('/class=["\'][^"\']*onecore-version[^"\']*["\'][^>]*>[\s]*([\d]+\.[\d]+\.[\d]+)/iu', $html, $matches)) {
						$latestVersion = trim($matches[1]);
					}
					elseif (preg_match('/class=["\'][^"\']*onecore-version[^"\']*["\'][^>]*>[\s]*<[^>]+>[\s]*([\d]+\.[\d]+\.[\d]+)/iu', $html, $matches)) {
						$latestVersion = trim($matches[1]);
					}
					// 3. Fallback: "Current Version: X.X.X" in page text
					elseif (preg_match('/Current\s+Version\s*:\s*([\d]+\.[\d]+\.[\d]+)/iu', $html, $matches)) {
						$latestVersion = trim($matches[1]);
					}
					elseif (preg_match('/Current\s+Version\s*:\s*<[^>]+>([\d]+\.[\d]+\.[\d]+)/iu', $html, $matches)) {
						$latestVersion = trim($matches[1]);
					}

					if (!$latestVersion) {
						$error = 'Could not find version information on the page';
					}
				} else {
					$error = 'HTTP Error: ' . $response->getStatusCode();
				}
			} catch (\Exception $e) {
				$error = 'Failed to fetch update URL: ' . $e->getMessage();
				Factory::getApplication()->getLogger()->error(
					'OneCore Version Check Error: ' . $e->getMessage(),
					['category' => 'onecore', 'exception' => $e]
				);
			}

			// Compare versions
			$isUpToDate = false;
			if ($latestVersion) {
				// Compare version strings
				$isUpToDate = version_compare($currentVersion, $latestVersion, '>=');
			}

			// Return JSON response
			$responseData = [
				'success' => true,
				'current_version' => $currentVersion,
				'latest_version' => $latestVersion,
				'is_up_to_date' => $isUpToDate,
				'error' => $error,
				'message' => $latestVersion 
					? ($isUpToDate 
						? 'You are using the latest version (' . $currentVersion . ')' 
						: 'New version available: ' . $latestVersion . ' (Current: ' . $currentVersion . ')')
					: ($error ? 'Error checking for updates: ' . $error : 'Could not determine latest version')
			];

			echo new JsonResponse($responseData, null, false, true);
			$this->app->close();

		} catch (\Exception $e) {
			// Log error
			Factory::getApplication()->getLogger()->error(
				'OneCore Version Check Error: ' . $e->getMessage(),
				['category' => 'onecore', 'exception' => $e]
			);

			// Return error response
			$this->app->setHeader('Status', '500 Internal Server Error', true);
			echo new JsonResponse([
				'success' => false,
				'message' => 'Error checking for updates: ' . $e->getMessage(),
				'current_version' => null,
				'latest_version' => null,
				'is_up_to_date' => false
			]);
			$this->app->close();
		}
	}
}
