<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Controller;

use Comdev\Component\Onecore\Administrator\Helper\LicenseHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Controller\FormController;
use Joomla\CMS\Router\Route;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Customfields controller class.
 *
 * @since  1.0.4
 */
class CustomfieldsController extends FormController
{
	/**
	 * The prefix to use with controller messages.
	 *
	 * @var    string
	 * @since  1.0.4
	 */
	protected $text_prefix = 'COM_ONECORE_CUSTOMFIELDS';

	/**
	 * The URL view list variable.
	 *
	 * @var    string
	 * @since  1.0.4
	 */
	protected $view_list = 'customfields';

	/**
	 * Constructor.
	 *
	 * @param   array  $config   An optional associative array of configuration settings.
	 * @param   mixed  $factory  The factory.
	 * @param   mixed  $app      The Application for the dispatcher
	 * @param   mixed  $input    Input
	 *
	 * @since   1.0.4
	 */
	public function __construct($config = [], $factory = null, $app = null, $input = null)
	{
		parent::__construct($config, $factory, $app, $input);

		// Register publish/unpublish tasks
		$this->registerTask('unpublish', 'publish');
	}

	/**
	 * Method to get a model object, loading it if required.
	 *
	 * @param   string  $name    The model name. Optional.
	 * @param   string  $prefix  The class prefix. Optional.
	 * @param   array   $config  Configuration array for model. Optional.
	 *
	 * @return  \Joomla\CMS\MVC\Model\BaseDatabaseModel  The model.
	 *
	 * @since   1.0.4
	 */
	public function getModel($name = '', $prefix = '', $config = ['ignore_request' => true])
	{
		$input = $this->app->getInput();
		$task = $input->get('task', '');
		$view = $input->get('view', '');
		$layout = $input->get('layout', '');
		$id = $input->get('id', 0);

		$isEditOperation = false;
		if (
			$task === 'customfields.add'
			|| $task === 'customfields.edit'
			|| $task === 'customfields.save'
			|| $task === 'customfields.apply'
			|| $task === 'customfields.save2new'
			|| $task === 'customfields.save2copy'
			|| $layout === 'edit'
			|| ($view === 'customfields' && (int) $id > 0)
			|| \in_array($task, ['add', 'edit', 'save', 'apply', 'save2new', 'save2copy', 'reload'], true)
		) {
			$isEditOperation = true;
		}

		if (empty($name)) {
			$name = $isEditOperation ? 'Item' : 'Customfields';
		}

		if (($name === 'Customfields' || $name === 'customfields') && $isEditOperation) {
			$name = 'Item';
		}

		if ($name === 'Item' || $name === 'item') {
			$className = 'Comdev\\Component\\Onecore\\Administrator\\Model\\Customfields\\ItemModel';

			if (\class_exists($className)) {
				$component = \Joomla\CMS\Factory::getApplication()->bootComponent('com_onecore');
				$factory = $component->getMVCFactory();

				$config['ignore_request'] = false;

				$model = new $className($config, $factory);

				if ($model instanceof \Joomla\Database\DatabaseAwareInterface) {
					$model->setDatabase(\Joomla\CMS\Factory::getContainer()->get(\Joomla\Database\DatabaseInterface::class));
				}
				if ($model instanceof \Joomla\CMS\Form\FormFactoryAwareInterface) {
					$model->setFormFactory(\Joomla\CMS\Factory::getContainer()->get(\Joomla\CMS\Form\FormFactoryInterface::class));
				}
				if ($model instanceof \Joomla\Event\DispatcherAwareInterface) {
					$model->setDispatcher(\Joomla\CMS\Factory::getContainer()->get(\Joomla\Event\DispatcherInterface::class));
				}

				return $model;
			}
		}

		return parent::getModel($name, $prefix, $config);
	}

	/**
	 * Method to publish a list of custom fields.
	 *
	 * @return  void
	 *
	 * @since  1.0.4
	 */
	public function publish()
	{
		$this->checkToken();

		$cid = (array) $this->input->get('cid', [], 'int');
		$cid = array_filter($cid);

		$data = ['publish' => 1, 'unpublish' => 0];
		$task = $this->getTask();
		$value = \Joomla\Utilities\ArrayHelper::getValue($data, $task, 0, 'int');

		if (empty($cid)) {
			\Joomla\CMS\Factory::getApplication()->enqueueMessage(
				\Joomla\CMS\Language\Text::_('JGLOBAL_NO_ITEM_SELECTED'),
				'warning'
			);
		} else {
			$model = $this->getModel('Customfields', 'Administrator', ['ignore_request' => false]);
			try {
				$model->publish($cid, $value);
			} catch (\Exception $e) {
				$this->setMessage($e->getMessage(), 'error');
			}
		}

		$this->setRedirect(
			\Joomla\CMS\Router\Route::_(
				'index.php?option=' . $this->option . '&view=' . $this->view_list,
				false
			)
		);
	}

	/**
	 * Method to save a record.
	 * Override to check if trying to add new Plus features when license expired.
	 *
	 * @param   string  $key     The name of the primary key of the URL variable.
	 * @param   string  $urlVar  The name of the URL variable if different from the primary key.
	 *
	 * @return  boolean  True if successful, false otherwise.
	 *
	 * @since   1.0.4
	 */
	public function save($key = null, $urlVar = null)
	{
		// Get form data before parent::save() processes it
		$data = $this->input->post->get('jform', [], 'array');
		
		// Determine the name of the primary key
		$model = $this->getModel();
		$table = $model->getTable();
		if (empty($key)) {
			$key = $table->getKeyName();
		}
		if (empty($urlVar)) {
			$urlVar = $key;
		}
		
		$recordId = (int) $this->input->getInt($urlVar, 0);
		
		// Check if this is a new record (not editing existing)
		$isNewRecord = ($recordId === 0);
		
		// Check if trying to add Plus feature (select or multiselect type)
		$fieldType = $data['type'] ?? '';
		$isPlusFeature = in_array($fieldType, ['select', 'multiselect'], true);
		
		// If trying to add new Plus feature and license expired, block it
		if ($isNewRecord && $isPlusFeature && LicenseHelper::isLicenseExpired()) {
			$this->app->enqueueMessage(
				Text::_('COM_ONECORE_LICENSE_EXPIRED_CANNOT_ADD_PLUS_FEATURES'),
				'error'
			);
			
			// Save form data in session to preserve user input
			$context = $this->option . '.edit.' . $this->context;
			$this->app->setUserState($context . '.data', $data);
			
			// Redirect back to edit form
			$this->setRedirect(
				Route::_(
					'index.php?option=' . $this->option . '&view=' . $this->view_item . '&layout=edit',
					false
				)
			);
			
			return false;
		}
		
		// Continue with normal save process
		return parent::save($key, $urlVar);
	}

	/**
	 * Delete selected custom fields.
	 *
	 * @return  void
	 *
	 * @since  1.0.4
	 */
	public function delete()
	{
		$this->checkToken();

		$cid = (array) $this->input->get('cid', [], 'int');
		$cid = array_filter($cid);

		if (empty($cid)) {
			\Joomla\CMS\Factory::getApplication()->enqueueMessage(
				\Joomla\CMS\Language\Text::_('JGLOBAL_NO_ITEM_SELECTED'),
				'warning'
			);
		} else {
			$model = $this->getModel('Customfields', 'Administrator', ['ignore_request' => false]);
			try {
				if (!$model->delete($cid)) {
					$err = $model->getError();
					$this->setMessage($err ?: \Joomla\CMS\Language\Text::_('JLIB_APPLICATION_ERROR_DELETE_FAILED'), 'error');
				}
			} catch (\Exception $e) {
				$this->setMessage($e->getMessage(), 'error');
			}
		}

		$this->setRedirect(
			\Joomla\CMS\Router\Route::_(
				'index.php?option=' . $this->option . '&view=' . $this->view_list,
				false
			)
		);
	}
}

