<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Controller;

use Joomla\CMS\MVC\Controller\FormController;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Content controller class.
 *
 * @since  1.0.0
 */
class ContentController extends FormController
{
	/**
	 * Cached Item model instance for edit operations.
	 *
	 * @var    \Joomla\CMS\MVC\Model\BaseDatabaseModel|null
	 * @since  1.0.0
	 */
	protected $itemModel = null;
	/**
	 * The prefix to use with controller messages.
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	protected $text_prefix = 'COM_ONECORE_CONTENT';

	/**
	 * The URL view item variable.
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	protected $view_item = 'content';

	/**
	 * The URL view list variable.
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	protected $view_list = 'content';

	/**
	 * Constructor.
	 *
	 * @param   array                 $config   An optional associative array of configuration settings.
	 * @param   ?MVCFactoryInterface  $factory  The factory.
	 * @param   ?CMSApplication       $app      The Application for the dispatcher
	 * @param   ?Input                $input    Input
	 *
	 * @since   1.0.0
	 */
	public function __construct($config = [], $factory = null, $app = null, $input = null)
	{
		parent::__construct($config, $factory, $app, $input);

		// Register publish/unpublish tasks
		$this->registerTask('unpublish', 'publish');
		
		// Register featured/unfeatured tasks
		$this->registerTask('unfeatured', 'featured');
		
		// 🔑 KLUCZ DO APPLY - map content.apply to save (FormController handles apply vs save logic)
		$this->registerTask('content.apply', 'save');
		$this->registerTask('content.save2new', 'save');
		$this->registerTask('content.save2copy', 'save');
		$this->registerTask('content.delete', 'delete');
	}

	/**
	 * Method to save a record.
	 *
	 * @param   string  $key     The name of the primary key of the URL variable.
	 * @param   string  $urlVar  The name of the URL variable if different from the primary key (sometimes required to avoid router collisions).
	 *
	 * @return  boolean  True if successful, false otherwise.
	 *
	 * @since   1.0.0
	 */
	public function save($key = null, $urlVar = null)
	{
		// Get original task from POST/GET (before registerTask normalization)
		$originalTask = $this->input->post->get('task', $this->input->get->get('task', '', 'cmd'), 'cmd');
		
		// Normalize task names - FormController expects 'apply' or 'save', not 'content.apply' or 'content.save'
		// registerTask already routes them to save(), but we need to set the correct task for FormController
		if ($originalTask === 'content.apply') {
			// 'apply' stays in edit view - FormController handles this
			$this->task = 'apply';
			$this->input->set('task', 'apply');
		} elseif ($originalTask === 'content.save') {
			// 'save' goes to list view - FormController handles this
			$this->task = 'save';
			$this->input->set('task', 'save');
		} elseif ($originalTask === 'content.save2new') {
			$this->task = 'save2new';
			$this->input->set('task', 'save2new');
		} elseif ($originalTask === 'content.save2copy') {
			$this->task = 'save2copy';
			$this->input->set('task', 'save2copy');
		}
		
		return parent::save($key, $urlVar);
	}

	/**
	 * Function that allows child controller access to model data
	 * after the data has been saved.
	 *
	 * @param   \Joomla\CMS\MVC\Model\BaseDatabaseModel  $model      The data model object.
	 * @param   array                                      $validData  The validated data.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	protected function postSaveHook(\Joomla\CMS\MVC\Model\BaseDatabaseModel $model, $validData = [])
	{
		// If apply task, ensure redirect has ID
		if ($this->getTask() === 'apply') {
			$recordId = (int) $model->getState($model->getName() . '.id');
			
			// If ID is 0, try to get it from table
			if ($recordId <= 0) {
				$table = $model->getTable();
				$recordId = (int) ($table->id ?? 0);
			}
			
			// Fallback: try insertid() for new records
			if ($recordId <= 0) {
				try {
					$recordId = (int) \Joomla\CMS\Factory::getDbo()->insertid();
				} catch (\Exception $e) {
					$recordId = 0;
				}
			}
			
			// Fallback: try from validData
			if ($recordId <= 0 && isset($validData['id'])) {
				$recordId = (int) $validData['id'];
			}
			
			// If we found ID, ensure redirect includes it
			if ($recordId > 0) {
				$context = $this->option . '.edit.' . $this->context;
				$this->holdEditId($context, $recordId);
				$this->app->setUserState($context . '.data', null);
				
				$this->setRedirect(
					\Joomla\CMS\Router\Route::_(
						'index.php?option=' . $this->option . '&view=' . $this->view_item
						. $this->getRedirectToItemAppend($recordId, 'id'),
						false
					)
				);
			}
		}
	}

	/**
	 * Method to get a model object, loading it if required.
	 *
	 * @param   string  $name    The model name. Optional.
	 * @param   string  $prefix  The class prefix. Optional.
	 * @param   array   $config  Configuration array for model. Optional.
	 *
	 * @return  \Joomla\CMS\MVC\Model\BaseDatabaseModel  The model.
	 *
	 * @since   1.0.0
	 */
	public function getModel($name = '', $prefix = '', $config = ['ignore_request' => true])
	{
		$input = $this->app->getInput();
		
		// Check if we need Item model (AdminModel) for edit operations
		// This must be checked BEFORE determining the model name
		$task = $input->get('task', '');
		$view = $input->get('view', '');
		$layout = $input->get('layout', '');
		$id = $input->get('id', 0);
		
		// Determine if this is an edit operation
		$isEditOperation = false;
		if ($task === 'content.add' || 
			$task === 'content.edit' || 
			$task === 'content.save' || 
			$task === 'content.apply' || 
			$task === 'content.delete' ||
			$task === 'content.save2new' || 
			$task === 'content.save2copy' ||
			$task === 'content.reload' ||
			$layout === 'edit' ||
			($view === 'content' && $id > 0) ||
			// Also check for standard FormController tasks
			in_array($task, ['add', 'edit', 'save', 'apply', 'save2new', 'save2copy', 'reload', 'delete'])) {
			$isEditOperation = true;
		}
		
		// If name is empty, FormController uses $this->context which is 'content'
		// For edit operations, we MUST use Item model (AdminModel)
		if (empty($name)) {
			if ($isEditOperation) {
				$name = 'Item';
			} else {
				$name = $this->context; // This will be 'content' for list operations
			}
		}
		
		// Also check if name is 'content' but we need Item model
		if (($name === 'content' || $name === 'Content') && $isEditOperation) {
			$name = 'Item';
		}
		
		// Model is in Content subdirectory, so load it directly if needed
		if ($name === 'Item' || $name === 'item') {
			if ($this->itemModel) {
				return $this->itemModel;
			}

			$className = 'Comdev\\Component\\Onecore\\Administrator\\Model\\Content\\ItemModel';
			if (class_exists($className)) {
				// Get MVCFactory from component
				$component = \Joomla\CMS\Factory::getApplication()->bootComponent('com_onecore');
				$factory = $component->getMVCFactory();
				
				// For edit operations, don't ignore request
				if ($isEditOperation) {
					$config['ignore_request'] = false;
				}
				
				$model = new $className($config, $factory);
				
				// Set required dependencies
				if ($model instanceof \Joomla\Database\DatabaseAwareInterface) {
					$model->setDatabase(\Joomla\CMS\Factory::getContainer()->get(\Joomla\Database\DatabaseInterface::class));
				}
				if ($model instanceof \Joomla\CMS\Form\FormFactoryAwareInterface) {
					$model->setFormFactory(\Joomla\CMS\Factory::getContainer()->get(\Joomla\CMS\Form\FormFactoryInterface::class));
				}
				if ($model instanceof \Joomla\Event\DispatcherAwareInterface) {
					$model->setDispatcher(\Joomla\CMS\Factory::getContainer()->get(\Joomla\Event\DispatcherInterface::class));
				}
				
				$this->itemModel = $model;
				return $model;
			}
		}
		
		return parent::getModel($name, $prefix, $config);
	}

	/**
	 * Method to publish a list of items.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function publish()
	{
		// Check for request forgeries
		$this->checkToken();

		// Get items to publish from the request
		$cid = (array) $this->input->get('cid', [], 'int');
		$data = ['publish' => 1, 'unpublish' => 0, 'trash' => -2];
		$task = $this->getTask();
		$value = \Joomla\Utilities\ArrayHelper::getValue($data, $task, 0, 'int');

		// Remove zero values resulting from input filter
		$cid = array_filter($cid);

		if (empty($cid)) {
			\Joomla\CMS\Factory::getApplication()->enqueueMessage(
				\Joomla\CMS\Language\Text::_($this->text_prefix . '_NO_ITEM_SELECTED'),
				'warning'
			);
		} else {
			// Get the model (Content model for list operations)
			$model = $this->getModel('Content', 'Administrator', ['ignore_request' => false]);

			// Publish the items
			try {
				if (!$model->publish($cid, $value)) {
					\Joomla\CMS\Factory::getApplication()->enqueueMessage(
						implode('<br>', $model->getErrors()),
						'error'
					);
				} else {
					$errors = $model->getErrors();
					$messageType = 'message';

					if ($value == 1) {
						if ($errors) {
							$messageType = 'error';
							$ntext = $this->text_prefix . '_N_ITEMS_FAILED_PUBLISHING';
						} else {
							$ntext = $this->text_prefix . '_N_ITEMS_PUBLISHED';
						}
					} elseif ($value == 0) {
						$ntext = $this->text_prefix . '_N_ITEMS_UNPUBLISHED';
					} else {
						$ntext = $this->text_prefix . '_N_ITEMS_TRASHED';
					}

					$this->setMessage(\Joomla\CMS\Language\Text::plural($ntext, \count($cid)), $messageType);
				}
			} catch (\Exception $e) {
				$this->setMessage($e->getMessage(), 'error');
			}
		}

		$this->setRedirect(
			\Joomla\CMS\Router\Route::_(
				'index.php?option=' . $this->option . '&view=' . $this->view_list,
				false
			)
		);
	}

	/**
	 * Removes an item.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function delete()
	{
		// Check for request forgeries
		$this->checkToken();

		// Get items to remove from the request.
		$cid = (array) $this->input->get('cid', [], 'int');

		// Remove zero values resulting from input filter
		$cid = array_filter($cid);

		if (empty($cid)) {
			$this->getLogger()->warning(\Joomla\CMS\Language\Text::_($this->text_prefix . '_NO_ITEM_SELECTED'), ['category' => 'jerror']);
		} else {
			// Get the model (Item model)
			$model = $this->getModel('Item', '', ['ignore_request' => true]);

			// Remove the items.
			if ($model->delete($cid)) {
				$this->setMessage(\Joomla\CMS\Language\Text::plural($this->text_prefix . '_N_ITEMS_DELETED', \count($cid)));
			} else {
				$this->setMessage($model->getError(), 'error');
			}
		}

		$this->setRedirect(
			\Joomla\CMS\Router\Route::_(
				'index.php?option=' . $this->option . '&view=' . $this->view_list,
				false
			)
		);
	}

	/**
	 * Method to toggle the featured setting of a list of items.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function featured()
	{
		// Check for request forgeries
		$this->checkToken();

		// Get items to toggle from the request
		$cid = (array) $this->input->get('cid', [], 'int');
		$data = ['featured' => 1, 'unfeatured' => 0];
		$task = $this->getTask();
		$value = \Joomla\Utilities\ArrayHelper::getValue($data, $task, 0, 'int');

		// Remove zero values resulting from input filter
		$cid = array_filter($cid);

		if (empty($cid)) {
			\Joomla\CMS\Factory::getApplication()->enqueueMessage(
				\Joomla\CMS\Language\Text::_($this->text_prefix . '_NO_ITEM_SELECTED'),
				'warning'
			);
		} else {
			// Get the model
			$model = $this->getModel('Content', 'Administrator', ['ignore_request' => false]);

			// Toggle featured status
			try {
				if (!$model->featured($cid, $value)) {
					\Joomla\CMS\Factory::getApplication()->enqueueMessage(
						implode('<br>', $model->getErrors()),
						'error'
					);
				} else {
					$errors = $model->getErrors();
					$messageType = 'message';

					if ($value == 1) {
						if ($errors) {
							$messageType = 'error';
							$ntext = $this->text_prefix . '_N_ITEMS_FAILED_FEATURING';
						} else {
							$ntext = $this->text_prefix . '_N_ITEMS_FEATURED';
						}
					} else {
						$ntext = $this->text_prefix . '_N_ITEMS_UNFEATURED';
					}

					$this->setMessage(\Joomla\CMS\Language\Text::plural($ntext, \count($cid)), $messageType);
				}
			} catch (\Exception $e) {
				$this->setMessage($e->getMessage(), 'error');
			}
		}

		$this->setRedirect(
			\Joomla\CMS\Router\Route::_(
				'index.php?option=' . $this->option . '&view=' . $this->view_list,
				false
			)
		);
	}

	/**
	 * Proxy method for geocoding addresses via Nominatim.
	 * This avoids CORS issues by making the request server-side.
	 *
	 * @return  void
	 *
	 * @since   1.0.7
	 */
	public function geocode()
	{
		$app = \Joomla\CMS\Factory::getApplication();
		$input = $app->getInput();

		// Set JSON response
		$app->setHeader('Content-Type', 'application/json', true);

		// Check for request forgeries
		// Use checkToken with redirect=false to avoid redirects on AJAX requests
		try {
			if (!$this->checkToken('get', false)) {
				http_response_code(403);
				echo json_encode(['error' => 'Token validation failed']);
				$app->close();
				return;
			}
		} catch (\Exception $e) {
			// Log error for debugging
			\Joomla\CMS\Log\Log::add('Geocode token check error: ' . $e->getMessage(), \Joomla\CMS\Log\Log::ERROR, 'onecore');
			http_response_code(403);
			echo json_encode(['error' => 'Token validation error: ' . $e->getMessage()]);
			$app->close();
			return;
		}

		// Get parameters
		$query = $input->getString('q', '');
		$lat = $input->getFloat('lat', 0);
		$lon = $input->getFloat('lon', 0);
		$type = $input->getString('type', 'search'); // 'search' or 'reverse'

		if ($type === 'reverse' && $lat && $lon) {
			// Reverse geocoding
			$url = 'https://nominatim.openstreetmap.org/reverse?format=json&lat=' . urlencode($lat) . '&lon=' . urlencode($lon) . '&zoom=18&addressdetails=1';
		} elseif ($query) {
			// Forward geocoding
			$url = 'https://nominatim.openstreetmap.org/search?format=json&q=' . urlencode($query) . '&limit=1';
		} else {
			http_response_code(400);
			echo json_encode(['error' => 'Missing required parameters']);
			$app->close();
			return;
		}

		// Check if cURL is available
		if (!function_exists('curl_init')) {
			http_response_code(500);
			echo json_encode(['error' => 'cURL extension is not available']);
			$app->close();
			return;
		}

		// Get referer URL safely
		$referer = '';
		try {
			$uri = \Joomla\CMS\Uri\Uri::getInstance();
			if ($uri) {
				$referer = $uri->toString(['scheme', 'host', 'port']);
			}
		} catch (\Exception $e) {
			// Fallback to empty referer
			$referer = '';
		}

		// Initialize cURL
		$ch = curl_init();
		if ($ch === false) {
			http_response_code(500);
			echo json_encode(['error' => 'Failed to initialize cURL']);
			$app->close();
			return;
		}

		curl_setopt($ch, CURLOPT_URL, $url);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_USERAGENT, 'Joomla OneCore Component/1.0.7');
		if ($referer) {
			curl_setopt($ch, CURLOPT_REFERER, $referer);
		}
		curl_setopt($ch, CURLOPT_TIMEOUT, 10);
		curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);

		$response = curl_exec($ch);
		$httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
		$error = curl_error($ch);
		$curlErrno = curl_errno($ch);
		curl_close($ch);

		if ($error || $curlErrno) {
			\Joomla\CMS\Log\Log::add('Geocode cURL error: ' . $error . ' (errno: ' . $curlErrno . ')', \Joomla\CMS\Log\Log::ERROR, 'onecore');
			http_response_code(500);
			echo json_encode(['error' => 'Request failed: ' . ($error ?: 'cURL error ' . $curlErrno)]);
			$app->close();
			return;
		}

		if ($httpCode !== 200) {
			\Joomla\CMS\Log\Log::add('Geocode Nominatim HTTP error: ' . $httpCode . ' - Response: ' . substr($response, 0, 200), \Joomla\CMS\Log\Log::ERROR, 'onecore');
			http_response_code($httpCode);
			echo json_encode(['error' => 'Nominatim returned HTTP ' . $httpCode]);
			$app->close();
			return;
		}

		// Validate JSON response
		$decoded = json_decode($response, true);
		if (json_last_error() !== JSON_ERROR_NONE) {
			\Joomla\CMS\Log\Log::add('Geocode JSON error: ' . json_last_error_msg() . ' - Response: ' . substr($response, 0, 200), \Joomla\CMS\Log\Log::ERROR, 'onecore');
			http_response_code(500);
			echo json_encode(['error' => 'Invalid JSON response from Nominatim: ' . json_last_error_msg()]);
			$app->close();
			return;
		}

		// Return the response
		echo $response;
		$app->close();
	}
}

