-- Reorganize Email menu structure
-- This update removes old Email Templates menu and creates new structure:
-- Emails (parent)
--   - Email Templates (submenu)
--   - Email Logs (submenu)

-- Get component ID
SET @component_id = (SELECT extension_id FROM `#__extensions` WHERE element = 'com_onecore' AND type = 'component' LIMIT 1);

-- Step 1: Delete existing Email Templates and Email Logs menu items
-- Get IDs and nested set values before deletion
SET @old_emailtemplates_id = (SELECT id FROM `#__menu` WHERE menutype = 'main' AND client_id = 1 AND link = 'index.php?option=com_onecore&view=emailtemplates' LIMIT 1);
SET @old_emaillogs_id = (SELECT id FROM `#__menu` WHERE menutype = 'main' AND client_id = 1 AND link = 'index.php?option=com_onecore&view=emaillogs' LIMIT 1);

-- Get nested set values for old Email Templates menu (if exists)
SET @old_emailtemplates_lft = (SELECT lft FROM `#__menu` WHERE id = @old_emailtemplates_id LIMIT 1);
SET @old_emailtemplates_rgt = (SELECT rgt FROM `#__menu` WHERE id = @old_emailtemplates_id LIMIT 1);
SET @old_emailtemplates_width = COALESCE(@old_emailtemplates_rgt, 0) - COALESCE(@old_emailtemplates_lft, 0) + 1;

-- Delete Email Logs first (child)
DELETE FROM `#__menu` 
WHERE menutype = 'main' 
    AND client_id = 1
    AND link = 'index.php?option=com_onecore&view=emaillogs';

-- Delete Email Templates (parent)
DELETE FROM `#__menu` 
WHERE menutype = 'main' 
    AND client_id = 1
    AND link = 'index.php?option=com_onecore&view=emailtemplates';

-- Close the gap in nested set after deletion (if old menu existed)
UPDATE `#__menu`
SET lft = CASE 
        WHEN lft > @old_emailtemplates_rgt THEN lft - @old_emailtemplates_width
        ELSE lft
    END,
    rgt = CASE 
        WHEN rgt > @old_emailtemplates_rgt THEN rgt - @old_emailtemplates_width
        ELSE rgt
    END
WHERE menutype = 'main'
    AND client_id = 1
    AND @old_emailtemplates_rgt IS NOT NULL
    AND (lft > @old_emailtemplates_rgt OR rgt > @old_emailtemplates_rgt);

-- Step 2: Find OneCore parent menu item (the main component menu)
-- OneCore parent menu has component_id = @component_id and parent_id = 1 (or is the root menu for component)
SET @onecore_parent_id = (SELECT id FROM `#__menu` 
    WHERE menutype = 'main' 
        AND client_id = 1 
        AND component_id = @component_id
        AND parent_id = 1
        AND (title = 'COM_ONECORE' OR link LIKE 'index.php?option=com_onecore%')
    ORDER BY lft ASC
    LIMIT 1);

-- If not found, try to find by title
SET @onecore_parent_id = COALESCE(@onecore_parent_id, 
    (SELECT id FROM `#__menu` 
        WHERE menutype = 'main' 
            AND client_id = 1 
            AND title = 'COM_ONECORE'
        LIMIT 1));

-- Get OneCore parent's nested set values
SET @onecore_parent_lft = (SELECT lft FROM `#__menu` WHERE id = @onecore_parent_id LIMIT 1);
SET @onecore_parent_rgt = (SELECT rgt FROM `#__menu` WHERE id = @onecore_parent_id LIMIT 1);
SET @onecore_parent_level = (SELECT level FROM `#__menu` WHERE id = @onecore_parent_id LIMIT 1);

-- Step 3: Find the position where we should insert the new "Emails" menu item
-- We'll insert it after Reviews menu item (as a child of OneCore parent)
SET @reviews_id = (SELECT id FROM `#__menu` WHERE menutype = 'main' AND client_id = 1 AND link = 'index.php?option=com_onecore&view=reviews' LIMIT 1);
SET @reviews_rgt = (SELECT rgt FROM `#__menu` WHERE id = @reviews_id LIMIT 1);

-- If Reviews doesn't exist, find the last child menu item of OneCore
SET @last_child_rgt = COALESCE(@reviews_rgt, 
    (SELECT MAX(rgt) FROM `#__menu` 
        WHERE menutype = 'main' 
            AND client_id = 1 
            AND parent_id = @onecore_parent_id
        LIMIT 1));

-- Set insert position (after last child of OneCore, or after parent's lft if no children)
SET @insert_position = COALESCE(@last_child_rgt, @onecore_parent_lft, 1) + 1;

-- Step 4: Create "Emails" parent menu item (as child of OneCore)
-- First, make room in nested set (we need space for parent + 2 children = 4 positions)
UPDATE `#__menu`
SET lft = CASE 
        WHEN lft >= @insert_position THEN lft + 4
        ELSE lft
    END,
    rgt = CASE 
        WHEN rgt >= @insert_position THEN rgt + 4
        ELSE rgt
    END
WHERE menutype = 'main'
    AND client_id = 1
    AND (lft >= @insert_position OR rgt >= @insert_position);

-- Update OneCore parent's rgt to include new child
UPDATE `#__menu`
SET rgt = rgt + 4
WHERE menutype = 'main'
    AND client_id = 1
    AND id = @onecore_parent_id
    AND @onecore_parent_id IS NOT NULL;

-- Insert "Emails" parent menu item (as child of OneCore)
INSERT INTO `#__menu` (
    menutype,
    title,
    alias,
    note,
    link,
    type,
    published,
    parent_id,
    component_id,
    checked_out,
    checked_out_time,
    browserNav,
    access,
    img,
    template_style_id,
    params,
    lft,
    rgt,
    level,
    home,
    language,
    client_id,
    path
)
SELECT
    'main',
    'COM_ONECORE_EMAILS',
    'com-onecore-emails',
    '',
    'index.php?option=com_onecore&view=emailtemplates',
    'component',
    1,
    @onecore_parent_id,
    @component_id,
    0,
    NULL,
    0,
    1,
    'class:envelope',
    0,
    '{}',
    @insert_position,
    @insert_position + 3,
    COALESCE(@onecore_parent_level, 1) + 1,
    0,
    '*',
    1,
    CONCAT(COALESCE((SELECT path FROM `#__menu` WHERE id = @onecore_parent_id LIMIT 1), 'onecore'), '/emails')
FROM DUAL
WHERE @component_id IS NOT NULL
    AND @onecore_parent_id IS NOT NULL
    AND NOT EXISTS (
        SELECT 1 FROM `#__menu` 
        WHERE menutype = 'main' 
            AND client_id = 1 
            AND alias = 'com-onecore-emails'
    );

-- Get the ID of the newly created "Emails" menu item
SET @emails_id = (SELECT id FROM `#__menu` WHERE menutype = 'main' AND client_id = 1 AND alias = 'com-onecore-emails' LIMIT 1);

-- Step 5: Create "Email Templates" submenu (only if parent exists)
-- Get parent's rgt value before update
SET @emails_rgt_before = (SELECT rgt FROM `#__menu` WHERE id = @emails_id LIMIT 1);

-- Update parent's rgt to make room for child
UPDATE `#__menu`
SET rgt = rgt + 2
WHERE menutype = 'main'
    AND client_id = 1
    AND id = @emails_id
    AND @emails_id IS NOT NULL;

-- Update all items after parent to make room (using variable to avoid MySQL error 1093)
UPDATE `#__menu`
SET lft = lft + 2,
    rgt = rgt + 2
WHERE menutype = 'main'
    AND client_id = 1
    AND lft > @emails_rgt_before
    AND @emails_id IS NOT NULL
    AND @emails_rgt_before IS NOT NULL;

-- Insert "Email Templates" submenu
INSERT INTO `#__menu` (
    menutype,
    title,
    alias,
    note,
    link,
    type,
    published,
    parent_id,
    component_id,
    checked_out,
    checked_out_time,
    browserNav,
    access,
    img,
    template_style_id,
    params,
    lft,
    rgt,
    level,
    home,
    language,
    client_id,
    path
)
SELECT
    'main',
    'COM_ONECORE_EMAIL_TEMPLATES',
    'com-onecore-emailtemplates',
    '',
    'index.php?option=com_onecore&view=emailtemplates',
    'component',
    1,
    @emails_id,
    @component_id,
    0,
    NULL,
    0,
    1,
    'class:envelope',
    0,
    '{}',
    @emails_rgt_before,
    @emails_rgt_before + 1,
    2,
    0,
    '*',
    1,
    CONCAT(COALESCE((SELECT path FROM `#__menu` WHERE id = @emails_id LIMIT 1), 'onecore/emails'), '/emailtemplates')
FROM DUAL
WHERE @component_id IS NOT NULL
    AND @emails_id IS NOT NULL
    AND @emails_rgt_before IS NOT NULL
    AND NOT EXISTS (
        SELECT 1 FROM `#__menu` 
        WHERE menutype = 'main' 
            AND client_id = 1 
            AND parent_id = @emails_id
            AND link = 'index.php?option=com_onecore&view=emailtemplates'
    );

-- Step 6: Create "Email Logs" submenu (only if parent exists)
-- Get parent's rgt value before update (after first child was added)
SET @emails_rgt_before2 = (SELECT rgt FROM `#__menu` WHERE id = @emails_id LIMIT 1);

-- Update parent's rgt to make room for second child
UPDATE `#__menu`
SET rgt = rgt + 2
WHERE menutype = 'main'
    AND client_id = 1
    AND id = @emails_id
    AND @emails_id IS NOT NULL;

-- Update all items after parent to make room (using variable to avoid MySQL error 1093)
UPDATE `#__menu`
SET lft = lft + 2,
    rgt = rgt + 2
WHERE menutype = 'main'
    AND client_id = 1
    AND lft > @emails_rgt_before2
    AND @emails_id IS NOT NULL
    AND @emails_rgt_before2 IS NOT NULL;

-- Insert "Email Logs" submenu
INSERT INTO `#__menu` (
    menutype,
    title,
    alias,
    note,
    link,
    type,
    published,
    parent_id,
    component_id,
    checked_out,
    checked_out_time,
    browserNav,
    access,
    img,
    template_style_id,
    params,
    lft,
    rgt,
    level,
    home,
    language,
    client_id,
    path
)
SELECT
    'main',
    'COM_ONECORE_EMAIL_LOGS',
    'com-onecore-emaillogs',
    '',
    'index.php?option=com_onecore&view=emaillogs',
    'component',
    1,
    @emails_id,
    @component_id,
    0,
    NULL,
    0,
    1,
    'class:list',
    0,
    '{}',
    @emails_rgt_before2,
    @emails_rgt_before2 + 1,
    2,
    0,
    '*',
    1,
    CONCAT(COALESCE((SELECT path FROM `#__menu` WHERE id = @emails_id LIMIT 1), 'onecore/emails'), '/emaillogs')
FROM DUAL
WHERE @component_id IS NOT NULL
    AND @emails_id IS NOT NULL
    AND @emails_rgt_before2 IS NOT NULL
    AND NOT EXISTS (
        SELECT 1 FROM `#__menu` 
        WHERE menutype = 'main' 
            AND client_id = 1 
            AND parent_id = @emails_id
            AND link = 'index.php?option=com_onecore&view=emaillogs'
    );
