<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

\defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;

/** @var \Comdev\Component\Onecore\Site\View\Content\HtmlView $this */

// Get current layout from URL parameter, then view property, then params, finally localStorage/default
$app = Factory::getApplication();
$input = $app->getInput();
$currentLayout = $input->get('layout', null, 'string');
if (empty($currentLayout)) {
	$currentLayout = $this->currentLayout ?? null;
}
if (empty($currentLayout)) {
	$currentLayout = $this->params->get('layout', 'grid');
}
$baseSefUrl = Route::_('index.php?option=com_onecore&view=content');
?>
<div class="com-onecore-content-wrapper one_container" data-base-sef-url="<?php echo $this->escape($baseSefUrl); ?>">
	<?php if ($this->params->get('show_page_heading')) : ?>
		<div class="page-header one_header">
			<h1 class="one_page_heading"><?php echo $this->escape($this->params->get('page_heading')); ?></h1>
		</div>
	<?php endif; ?>

	<!-- Filters and Layout Switcher -->
	<div class="one_filters_layout_wrapper mb-3 d-flex justify-content-between align-items-center flex-wrap gap-3">
		<!-- Filters: Sort and Items per Page -->
		<div class="one_filters d-flex align-items-center gap-2 flex-wrap">
			<!-- Sort Filter -->
			<div class="one_filter_sort">
				<label for="onecore-sort" class="visually-hidden"><?php echo Text::_('COM_ONECORE_SORT_BY'); ?></label>
				<select id="onecore-sort" name="sort" class="form-select form-select-sm one_sort_select">
					<?php 
					$app = Factory::getApplication();
					$currentSort = $app->getInput()->get('sort', 'newest');
					$sortOptions = [
						'newest' => Text::_('COM_ONECORE_SORT_NEWEST'),
						'oldest' => Text::_('COM_ONECORE_SORT_OLDEST'),
						'a-z' => Text::_('COM_ONECORE_SORT_A_Z'),
						'z-a' => Text::_('COM_ONECORE_SORT_Z_A'),
					];
					foreach ($sortOptions as $value => $label) :
					?>
						<option value="<?php echo $this->escape($value); ?>" <?php echo $currentSort === $value ? 'selected' : ''; ?>>
							<?php echo $this->escape($label); ?>
						</option>
					<?php endforeach; ?>
				</select>
			</div>

			<!-- Items per Page -->
			<div class="one_filter_limit">
				<label for="onecore-limit" class="visually-hidden"><?php echo Text::_('COM_ONECORE_ITEMS_PER_PAGE'); ?></label>
				<select id="onecore-limit" name="limit" class="form-select form-select-sm one_limit_select">
					<?php 
					$currentLimit = $app->getInput()->getInt('limit', $app->get('list_limit', 20));
					$limitOptions = [10, 20, 30, 50, 100];
					foreach ($limitOptions as $limit) :
					?>
						<option value="<?php echo (int) $limit; ?>" <?php echo $currentLimit == $limit ? 'selected' : ''; ?>>
							<?php echo (int) $limit; ?>
						</option>
					<?php endforeach; ?>
				</select>
			</div>
			
			<!-- Grid Columns (only show when grid layout is active) -->
			<div class="one_filter_columns" id="onecore-columns-wrapper" style="<?php echo $currentLayout === 'grid' ? '' : 'display: none;'; ?>">
				<label for="onecore-columns" class="visually-hidden"><?php echo Text::_('COM_ONECORE_COLUMNS_PER_ROW'); ?></label>
				<select id="onecore-columns" name="columns" class="form-select form-select-sm one_columns_select">
					<?php 
					$currentColumns = $app->getInput()->getInt('columns', 3);
					$columnOptions = [
						2 => Text::_('COM_ONECORE_COLUMNS_2'),
						3 => Text::_('COM_ONECORE_COLUMNS_3'),
						4 => Text::_('COM_ONECORE_COLUMNS_4'),
					];
					foreach ($columnOptions as $cols => $label) :
					?>
						<option value="<?php echo (int) $cols; ?>" <?php echo $currentColumns == $cols ? 'selected' : ''; ?>>
							<?php echo $this->escape($label); ?>
						</option>
					<?php endforeach; ?>
				</select>
			</div>
		</div>

		<!-- Layout Switcher -->
		<div class="layout-switcher one_layout_switcher d-flex gap-2">
			<button type="button" class="btn btn-sm btn-outline-primary layout-btn one_layout_btn one_layout_btn_grid <?php echo $currentLayout === 'grid' ? 'active' : ''; ?>" data-layout="grid" title="<?php echo Text::_('COM_ONECORE_LAYOUT_GRID'); ?>">
				<i data-feather="grid" class="one_layout_icon_grid" aria-hidden="true"></i>
				<span class="visually-hidden"><?php echo Text::_('COM_ONECORE_LAYOUT_GRID'); ?></span>
			</button>
			<button type="button" class="btn btn-sm btn-outline-primary layout-btn one_layout_btn one_layout_btn_list <?php echo $currentLayout === 'list' ? 'active' : ''; ?>" data-layout="list" title="<?php echo Text::_('COM_ONECORE_LAYOUT_LIST'); ?>">
				<i data-feather="list" class="one_layout_icon_list" aria-hidden="true"></i>
				<span class="visually-hidden"><?php echo Text::_('COM_ONECORE_LAYOUT_LIST'); ?></span>
			</button>
		</div>
	</div>

	<!-- Content Container -->
	<div id="onecore-content-container" class="onecore-content-container one_content_container">
		<?php
		// Load the appropriate layout based on current layout setting
		// Use direct include instead of loadTemplate to avoid default_grid naming issue
		$layoutFile = __DIR__ . '/' . $currentLayout . '.php';
		if (file_exists($layoutFile)) {
			include $layoutFile;
		} else {
			// Fallback to grid if layout file doesn't exist
			include __DIR__ . '/grid.php';
		}
		?>
	</div>
</div>

<?php
// Add AJAX functionality
$wa = Factory::getApplication()->getDocument()->getWebAssetManager();
$wa->useScript('core');
// Leaflet for content list map (initial + after AJAX load)
$wa->registerAndUseStyle('leaflet', 'https://unpkg.com/leaflet@1.9.4/dist/leaflet.css', [], ['rel' => 'stylesheet']);
$wa->registerAndUseScript('leaflet', 'https://unpkg.com/leaflet@1.9.4/dist/leaflet.js', [], ['defer' => true]);

// Note: Layout initialization happens in DOMContentLoaded to avoid redirect loops

$script = "
document.addEventListener('DOMContentLoaded', function() {
	const container = document.getElementById('onecore-content-container');
	const layoutButtons = document.querySelectorAll('.layout-btn');
	const sortSelect = document.getElementById('onecore-sort');
	const limitSelect = document.getElementById('onecore-limit');
	const columnsSelect = document.getElementById('onecore-columns');
	const columnsWrapper = document.getElementById('onecore-columns-wrapper');
	const baseSefEl = document.querySelector('[data-base-sef-url]');
	const baseSefUrl = baseSefEl ? baseSefEl.getAttribute('data-base-sef-url') : '';
	
	if (!container) return;
	
	function getLayoutFromUrlString(urlString) {
		try {
			const url = new URL(urlString, window.location.origin);
			const path = url.pathname || '';
			const m = path.match(/\\/(grid|list)(?:\\/|\\?|$)/);
			if (m) return m[1];
			const L = url.searchParams.get('layout');
			return (L === 'grid' || L === 'list') ? L : null;
		} catch (e) {
			return null;
		}
	}
	
	function getLayoutFromUrl() {
		return getLayoutFromUrlString(window.location.href);
	}
	
	function buildSefUrl(layout, params) {
		params = params || {};
		if (!baseSefUrl) {
			const cur = new URL(window.location.href);
			['layout','sort','limit','columns'].forEach(function(k){ var v = params[k]; if (v !== null && v !== '') cur.searchParams.set(k, v); else cur.searchParams.delete(k); });
			return cur.toString();
		}
		const base = new URL(baseSefUrl, window.location.origin);
		let path = base.pathname.replace(/\\/$/, '');
		if (layout === 'grid' || layout === 'list') path = path + '/' + layout;
		const q = [];
		if (params.sort) q.push('sort=' + encodeURIComponent(params.sort));
		if (params.limit) q.push('limit=' + encodeURIComponent(params.limit));
		if (params.columns && layout === 'grid') q.push('columns=' + encodeURIComponent(params.columns));
		return base.origin + path + (q.length ? '?' + q.join('&') : '');
	}
	
	// Initialize layout from URL (path or query) first, then localStorage
	const urlParams = new URLSearchParams(window.location.search);
	let currentLayout = getLayoutFromUrl();
	
	if (!currentLayout) {
		const layoutFromStorage = localStorage.getItem('onecore_layout');
		if (layoutFromStorage === 'grid' || layoutFromStorage === 'list') {
			currentLayout = layoutFromStorage;
			if (document.readyState === 'complete' && !document.getElementById('onecore-content-container')) {
				window.location.href = buildSefUrl(currentLayout, { sort: '', limit: '', columns: '' });
				return;
			}
			window.history.replaceState({layout: currentLayout}, '', buildSefUrl(currentLayout, { sort: '', limit: '', columns: '' }));
			localStorage.setItem('onecore_layout', currentLayout);
		} else {
			currentLayout = 'grid';
			localStorage.setItem('onecore_layout', 'grid');
			if (document.readyState === 'loading') {
				window.location.href = buildSefUrl('grid', { sort: '', limit: '', columns: '' });
				return;
			}
			window.history.replaceState({layout: 'grid'}, '', buildSefUrl('grid', { sort: '', limit: '', columns: '' }));
		}
	} else {
		localStorage.setItem('onecore_layout', currentLayout);
	}
	
	// Ensure buttons reflect current layout immediately
	layoutButtons.forEach(function(btn) {
		btn.classList.remove('active');
		if (btn.getAttribute('data-layout') === currentLayout) {
			btn.classList.add('active');
		}
	});
	updateColumnsSelectorVisibility(currentLayout);
	
	// Initialize columns from URL or localStorage (default: 3)
	let currentColumns = urlParams.get('columns');
	if (!currentColumns) {
		currentColumns = localStorage.getItem('onecore_columns') || '3';
	}
	// Validate columns - only allow 2, 3, or 4
	if (currentColumns && !['2', '3', '4'].includes(currentColumns)) {
		currentColumns = '3';
	}
	if (currentColumns) {
		localStorage.setItem('onecore_columns', currentColumns);
		if (columnsSelect) {
			columnsSelect.value = currentColumns;
		}
	}
	
	// Show/hide columns selector based on layout
	function updateColumnsSelectorVisibility(layout) {
		if (columnsWrapper) {
			if (layout === 'grid') {
				columnsWrapper.style.display = '';
			} else {
				columnsWrapper.style.display = 'none';
			}
		}
	}
	
	// Initialize visibility
	updateColumnsSelectorVisibility(currentLayout);
	
	// Function to build SEF URL with all parameters
	function buildUrl(params) {
		params = params || {};
		const layout = params.layout || getLayoutFromUrl() || currentLayout || 'grid';
		return buildSefUrl(layout, {
			sort: params.sort || '',
			limit: params.limit || '',
			columns: layout === 'grid' ? (params.columns || '') : ''
		});
	}
	
	// Init content list map (#onecore-map-items-top). Use data-map-items, data-map-zoom.
	// Call on first load and after AJAX loadContent replace.
	function initContentMap() {
		var mapContainer = document.getElementById('onecore-map-items-top');
		if (!mapContainer || mapContainer.dataset.initialized === 'true') return;
		var raw = mapContainer.getAttribute('data-map-items');
		var zoom = parseInt(mapContainer.getAttribute('data-map-zoom'), 10) || 13;
		var mapItems = [];
		try {
			if (raw) mapItems = JSON.parse(raw);
		} catch (e) { return; }
		if (!Array.isArray(mapItems) || mapItems.length === 0) return;
		if (typeof L === 'undefined') {
			setTimeout(initContentMap, 100);
			return;
		}
		mapContainer.dataset.initialized = 'true';
		
		var map;
		if (mapItems.length === 1) {
			map = L.map(mapContainer).setView([mapItems[0].lat, mapItems[0].lng], zoom);
		} else {
			var bounds = L.latLngBounds(mapItems.map(function(item) { return [item.lat, item.lng]; }));
			map = L.map(mapContainer).fitBounds(bounds, { padding: [50, 50] });
		}
		L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
			attribution: '&copy; <a href=\"https://www.openstreetmap.org/copyright\">OpenStreetMap</a> contributors',
			maxZoom: 19
		}).addTo(map);
		mapItems.forEach(function(item) {
			var marker = L.marker([item.lat, item.lng]).addTo(map);
			var popupContent = item.title || '';
			if (item.url) popupContent = '<a href=\"' + item.url + '\" style=\"text-decoration:none;color:#007bff;\">' + popupContent + '</a>';
			if (popupContent) marker.bindPopup(popupContent);
		});
	}
	
	// Function to load content via AJAX
	function loadContent(url) {
		// Show loading state
		if (container) {
			container.style.opacity = '0.5';
			container.style.pointerEvents = 'none';
		}
		
		// AJAX request
		fetch(url, {
			method: 'GET',
			headers: {
				'X-Requested-With': 'XMLHttpRequest',
				'Accept': 'text/html'
			}
		})
		.then(function(response) {
			if (!response.ok) {
				throw new Error('Network response was not ok');
			}
			return response.text();
		})
		.then(function(html) {
			// Parse response - look for the container content
			const parser = new DOMParser();
			const doc = parser.parseFromString(html, 'text/html');
			
			// Try to find the content container
			let newContainer = doc.getElementById('onecore-content-container');
			
			// If not found, try to find grid or list content directly
			if (!newContainer) {
				newContainer = doc.getElementById('onecore-content-grid') || doc.getElementById('onecore-content-list');
			}
			
			// If still not found, try to find by class
			if (!newContainer) {
				const contentDivs = doc.querySelectorAll('.com-onecore-content');
				if (contentDivs.length > 0) {
					newContainer = contentDivs[0];
				}
			}
			
			if (newContainer && container) {
				// Replace container content
				container.innerHTML = newContainer.innerHTML;
				
				// Update layout buttons active state if needed
				const urlObj = new URL(url, window.location.origin);
				const newLayout = getLayoutFromUrlString(url) || currentLayout;
				if (newLayout !== currentLayout) {
					currentLayout = newLayout;
					updateColumnsSelectorVisibility(newLayout);
					layoutButtons.forEach(function(btn) {
						if (btn.getAttribute('data-layout') === newLayout) {
							layoutButtons.forEach(function(b) {
								b.classList.remove('active');
							});
							btn.classList.add('active');
						}
					});
				}
				
				// Update columns selector if needed
				const newColumns = urlObj.searchParams.get('columns');
				if (newColumns && columnsSelect) {
					columnsSelect.value = newColumns;
					localStorage.setItem('onecore_columns', newColumns);
					currentColumns = newColumns;
				}
				
				// Update URL without page reload
				const state = {
					layout: newLayout,
					sort: urlObj.searchParams.get('sort') || '',
					limit: urlObj.searchParams.get('limit') || '',
					columns: newColumns || ''
				};
				window.history.pushState(state, '', url);
				
				// Scroll to top smoothly
				window.scrollTo({top: 0, behavior: 'smooth'});
				
				// Re-init map after AJAX replace (map div has data-map-items / data-map-zoom)
				setTimeout(initContentMap, 50);
			} else {
				// Fallback: reload page
				window.location.href = url;
			}
		})
		.catch(function(error) {
			console.error('Error loading content:', error);
			// Fallback to page reload
			window.location.href = url;
		})
		.finally(function() {
			if (container) {
				container.style.opacity = '1';
				container.style.pointerEvents = 'auto';
			}
		});
	}
	
	// Layout switcher handlers
	if (layoutButtons.length) {
		layoutButtons.forEach(function(btn) {
			btn.addEventListener('click', function(e) {
				e.preventDefault();
				const layout = this.getAttribute('data-layout');
				
				// Save to localStorage immediately
				localStorage.setItem('onecore_layout', layout);
				currentLayout = layout;
				
				// Update columns selector visibility
				updateColumnsSelectorVisibility(layout);
				
				// Update active state
				layoutButtons.forEach(function(b) {
					b.classList.remove('active');
				});
				this.classList.add('active');
				
				// Build SEF URL: all-content/grid or all-content/list + sort, limit, columns
				const currentUrl = new URL(window.location.href);
				const params = {
					sort: currentUrl.searchParams.get('sort') || '',
					limit: currentUrl.searchParams.get('limit') || '',
					columns: layout === 'grid' ? (currentUrl.searchParams.get('columns') || currentColumns || '3') : ''
				};
				const url = buildSefUrl(layout, params);
				window.location.href = url;
			});
		});
	}
	
	// Columns select handler (only for grid layout)
	if (columnsSelect) {
		columnsSelect.addEventListener('change', function() {
			let columns = this.value;
			
			// Validate columns - only allow 2, 3, or 4
			if (!['2', '3', '4'].includes(columns)) {
				columns = '3';
				this.value = '3';
			}
			
			// Save to localStorage
			localStorage.setItem('onecore_columns', columns);
			currentColumns = columns;
			
			// Build URL with columns parameter
			const currentUrl = new URL(window.location.href);
			const params = {
				layout: getLayoutFromUrl() || currentLayout || 'grid',
				sort: currentUrl.searchParams.get('sort') || '',
				limit: currentUrl.searchParams.get('limit') || '',
				columns: columns
			};
			const url = buildUrl(params);
			loadContent(url);
		});
	}
	
	// Sort select handler
	if (sortSelect) {
		sortSelect.addEventListener('change', function() {
			const currentUrl = new URL(window.location.href);
			const params = {
				layout: getLayoutFromUrl() || currentLayout || 'grid',
				sort: this.value,
				limit: currentUrl.searchParams.get('limit') || '',
				columns: currentUrl.searchParams.get('columns') || currentColumns || ''
			};
			const url = buildUrl(params);
			loadContent(url);
		});
	}
	
	// Limit select handler
	if (limitSelect) {
		limitSelect.addEventListener('change', function() {
			const currentUrl = new URL(window.location.href);
			const params = {
				layout: getLayoutFromUrl() || currentLayout || 'grid',
				sort: currentUrl.searchParams.get('sort') || '',
				limit: this.value,
				columns: currentUrl.searchParams.get('columns') || currentColumns || ''
			};
			const url = buildUrl(params);
			loadContent(url);
		});
	}
	
	// Handle browser back/forward buttons
	window.addEventListener('popstate', function(event) {
		if (event.state) {
			const state = event.state;
			const params = {
				layout: state.layout || '',
				sort: state.sort || '',
				limit: state.limit || '',
				columns: state.columns || ''
			};
			const url = buildUrl(params);
			
			// Update UI elements
			if (state.layout) {
				const btn = document.querySelector('.layout-btn[data-layout=\"' + state.layout + '\"]');
				if (btn) {
					layoutButtons.forEach(function(b) {
						b.classList.remove('active');
					});
					btn.classList.add('active');
				}
			}
			if (state.sort && sortSelect) {
				sortSelect.value = state.sort;
			}
			if (state.limit && limitSelect) {
				limitSelect.value = state.limit;
			}
			
			loadContent(url);
		} else {
			window.location.reload();
		}
	});
	
	// Init map on first load (Leaflet may load with defer)
	setTimeout(initContentMap, 100);
});
";

$wa->addInlineScript($script, ['name' => 'onecore-layout-switcher'], ['type' => 'module']);

$css = "
.one_filters_layout_wrapper {
	margin-bottom: 1.5rem;
}
.one_filters {
	min-width: 200px;
}
.one_filter_sort,
.one_filter_limit {
	min-width: 150px;
}
.one_sort_select,
.one_limit_select {
	display: inline-block;
}
.one_layout_switcher,
.layout-switcher {
	margin-left: auto;
}
.one_layout_btn,
.layout-btn {
	min-width: 40px;
}
.one_layout_btn.active,
.layout-btn.active {
	background-color: var(--bs-primary);
	color: var(--bs-white);
	border-color: var(--bs-primary);
}
.one_content_container,
.onecore-content-container {
	transition: opacity 0.3s ease;
}
.one_layout_icon_grid,
.one_layout_icon_list {
	display: inline-block;
	width: 1.2em;
	height: 1.2em;
	vertical-align: middle;
}
.one_layout_icon_grid svg,
.one_layout_icon_list svg {
	width: 1.2em;
	height: 1.2em;
}
@media (max-width: 768px) {
	.one_filters_layout_wrapper {
		flex-direction: column;
		align-items: flex-start !important;
	}
	.one_layout_switcher {
		margin-left: 0;
		width: 100%;
	}
}
";

$wa->addInlineStyle($css, ['name' => 'onecore-layout-switcher-styles']);
