<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Site\View\Event;

use Comdev\Component\Onecore\Site\Helper\CustomFieldsHelper;
use Comdev\Component\Onecore\Site\Helper\RouteHelper;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\View\HtmlView as BaseHtmlView;
use Joomla\CMS\MVC\View\GenericDataException;
use Joomla\CMS\Router\Route;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * HTML Event View class for the OneCore component
 *
 * @since  1.0.0
 */
class HtmlView extends BaseHtmlView
{
	/**
	 * The event object
	 *
	 * @var  \stdClass
	 */
	protected $item;

	/**
	 * The page parameters
	 *
	 * @var    \Joomla\Registry\Registry|null
	 */
	protected $params = null;

	/**
	 * The model state
	 *
	 * @var   \Joomla\Registry\Registry
	 */
	protected $state;

	/**
	 * Execute and display a template script.
	 *
	 * @param   string  $tpl  The name of the template file to parse; automatically searches through the template paths.
	 *
	 * @return  void
	 */
	public function display($tpl = null)
	{
		$app = Factory::getApplication();

		/** @var \Comdev\Component\Onecore\Site\Model\EventModel $model */
		$model       = $this->getModel('Event');
		$this->item  = $model->getItem();
		$this->state = $model->getState();

		// Check for errors.
		if (\count($errors = $model->getErrors())) {
			throw new GenericDataException(implode("\n", $errors), 500);
		}

		// Check if event exists
		if (!$this->item) {
			throw new GenericDataException(Text::_('COM_ONECORE_ERROR_ITEM_NOT_FOUND'), 404);
		}

		// Attach custom fields for event context ("item") in all positions
		$eid = (int) ($this->item->id ?? 0);
		if ($eid > 0) {
			$underTitle = CustomFieldsHelper::getFieldsForEventIds([$eid], 'item', 'under_title');
			$underIntro = CustomFieldsHelper::getFieldsForEventIds([$eid], 'item', 'under_intro');
			$sidebar = CustomFieldsHelper::getFieldsForEventIds([$eid], 'item', 'sidebar');

			$this->item->customfields_under_title = $underTitle[$eid] ?? [];
			$this->item->customfields_under_intro = $underIntro[$eid] ?? [];
			$this->item->customfields_sidebar = $sidebar[$eid] ?? [];
		}

		// Create a shortcut for $item.
		$item = $this->item;

		// Add router helpers.
		$item->slug = $item->alias ? ($item->id . ':' . $item->alias) : $item->id;

		// Merge event params. If this is single-event view, menu params override event params
		$params          = $this->state->get('params');
		$event_params    = new \Joomla\Registry\Registry();
		$active          = $app->getMenu()->getActive();
		$temp            = clone $params;

		// Check to see which parameters should take priority
		if ($active) {
			$currentLink = $active->link;
			// If the current view is the single event view menu item, then the menu item params take priority
			if (strpos($currentLink, 'view=event') && (strpos($currentLink, 'id=' . (string) $item->id))) {
				// Load layout from active query (in case it is an alternative menu item)
				if (isset($active->query['layout'])) {
					$this->setLayout($active->query['layout']);
				}
				// Check for alternative layouts (since we are not in a single-event menu item)
				// Single-event menu item param takes priority over param menu item
			} else {
				// Merge event params
				$event_params->loadString($item->params ?? '{}');
				$temp->merge($event_params);
			}
		} else {
			$event_params->loadString($item->params ?? '{}');
			$temp->merge($event_params);
		}

		$this->params = $temp;

		// Escape strings for HTML output
		$this->pageclass_sfx = htmlspecialchars($this->params->get('pageclass_sfx', ''));

		$this->_prepareDocument();

		$wa = $this->getDocument()->getWebAssetManager();
		$wa->registerAndUseStyle('com_onecore.event', 'components/com_onecore/assets/css/event.css')
			->registerAndUseStyle('com_onecore.event-attend', 'components/com_onecore/assets/css/event-attend.css')
			->registerAndUseStyle('com_onecore.social-share', 'components/com_onecore/assets/css/social-share.css');
		$wa->registerScript('com_onecore.feather', 'https://cdn.jsdelivr.net/npm/feather-icons/dist/feather.min.js', [], []);
		$wa->registerScript('com_onecore.feather-init', 'components/com_onecore/assets/js/feather-init.js', ['dependencies' => ['com_onecore.feather']], []);
		$wa->useScript('com_onecore.feather')->useScript('com_onecore.feather-init');

		parent::display($tpl);
	}

	/**
	 * Prepares the document (title, meta, canonical) for event view.
	 *
	 * @return  void
	 */
	protected function _prepareDocument(): void
	{
		$app        = Factory::getApplication();
		$menus      = $app->getMenu();
		$menu       = $menus->getActive();
		$item       = $this->item;
		$compParams = ComponentHelper::getParams('com_onecore');

		if ($menu) {
			$this->params->def('page_heading', $this->params->get('page_title', $menu->title));
		} else {
			$this->params->def('page_heading', $this->params->get('page_title', $item->title ?? Text::_('COM_ONECORE_EVENT_DEFAULT_PAGE_TITLE')));
		}

		$title = $this->params->get('page_title', '');
		if (empty($title)) {
			$title = $item->title ?? $app->get('sitename');
		}
		if ($app->get('sitename_pagetitles', 0) == 1) {
			$title = Text::sprintf('JPAGETITLE', $app->get('sitename'), $title);
		} elseif ($app->get('sitename_pagetitles', 0) == 2) {
			$title = Text::sprintf('JPAGETITLE', $title, $app->get('sitename'));
		}
		$this->getDocument()->setTitle($title);

		$metaDesc = trim((string) ($item->metadesc ?? ''));
		if ($metaDesc === '' && $this->params->get('menu-meta_description')) {
			$metaDesc = $this->params->get('menu-meta_description');
		}
		if ($metaDesc === '' && (int) $compParams->get('meta_desc_fallback', 1) === 1) {
			$intro = trim(strip_tags((string) ($item->description ?? '')));
			if ($intro !== '') {
				$metaDesc = mb_strlen($intro) > 160 ? mb_substr($intro, 0, 157) . '...' : $intro;
			}
		}
		if ($metaDesc !== '') {
			$this->getDocument()->setDescription($metaDesc);
		}

		$metaKey = trim((string) ($item->metakey ?? ''));
		if ($metaKey === '' && $this->params->get('menu-meta_keywords')) {
			$metaKey = $this->params->get('menu-meta_keywords');
		}
		if ($metaKey !== '') {
			$this->getDocument()->setMetadata('keywords', $metaKey);
		}

		$robots = $this->params->get('robots', '');
		if ($robots === '' && !empty($item->metadata)) {
			$meta = is_string($item->metadata) ? json_decode($item->metadata, true) : $item->metadata;
			if (!empty($meta['robots'])) {
				$robots = $meta['robots'];
			}
		}
		if ($robots === '' && $compParams->get('robots_fallback') !== '') {
			$robots = $compParams->get('robots_fallback');
		}
		if ($robots !== '') {
			$this->getDocument()->setMetadata('robots', $robots);
		}

		if ((int) $compParams->get('sef_add_canonical', 1) === 1) {
			$canonical = Route::_(
				RouteHelper::getEventRoute($item->slug),
				true,
				Route::TLS_IGNORE,
				true
			);
			$this->getDocument()->addHeadLink($canonical, 'canonical');
		}
	}
}
