<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Site\Service;

\defined('_JEXEC') or die;

use Joomla\CMS\Application\SiteApplication;
use Joomla\CMS\Categories\CategoryFactoryInterface;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Component\Router\RouterView;
use Joomla\CMS\Component\Router\RouterViewConfiguration;
use Comdev\Component\Onecore\Site\Service\Router\ContentMenuRule;
use Joomla\CMS\Component\Router\Rules\MenuRules;
use Joomla\CMS\Component\Router\Rules\StandardRules;
use Joomla\CMS\Component\Router\Rules\NomenuRules;
use Joomla\CMS\Menu\AbstractMenu;
use Joomla\Database\DatabaseInterface;
use Joomla\Database\ParameterType;

class Router extends RouterView
{
	protected bool $noIDs = false;

	private DatabaseInterface $db;

	public function __construct(
		SiteApplication $app,
		AbstractMenu $menu,
		CategoryFactoryInterface $categoryFactory,
		DatabaseInterface $db
	) {
		$this->db = $db;

		$params = ComponentHelper::getParams('com_onecore');
		$this->noIDs = (int) $params->get('sef_ids_in_url', 1) === 0;

		/**
		 * CONTENT LIST VIEW (ROOT)
		 * menu: /all-content
		 */
		$content = new RouterViewConfiguration('content');
		$content->setKey(null);
		// Root-level view: no parent (don't call setParent)
		$content->addLayout('grid'); // Register grid layout
		$content->addLayout('list'); // Register list layout
		$this->registerView($content);

		/**
		 * CATEGORIES VIEW (if needed)
		 */
		$categories = new RouterViewConfiguration('categories');
		$categories->setKey(null);
		$this->registerView($categories);

		/**
		 * EVENTS VIEW (list of events)
		 */
		$events = new RouterViewConfiguration('events');
		$events->setKey(null);
		$events->addLayout('grid'); // Register grid layout for events
		$events->addLayout('list'); // Register list layout for events
		$this->registerView($events);

		/**
		 * EVENT VIEW (single event)
		 */
		$event = new RouterViewConfiguration('event');
		$event->setKey('id');
		$event->setParent($events); // Parent is events, not content
		$event->setNestable(true);
		$this->registerView($event);

		/**
		 * SINGLE ITEM
		 * /all-content/alias
		 */
		$item = new RouterViewConfiguration('item');
		$item->setKey('id');
		$item->setParent($content);
		$item->setNestable(true);
		$this->registerView($item);

		parent::__construct($app, $menu);

		$this->attachRule(new MenuRules($this));
		$this->attachRule(new ContentMenuRule($this));
		$this->attachRule(new StandardRules($this));
		$this->attachRule(new NomenuRules($this));
	}

	/**
	 * SEGMENT: alias zamiast id
	 */
	public function getItemSegment($id, array $query): array
	{
		$alias = null;

		if (str_contains((string) $id, ':')) {
			[, $alias] = explode(':', $id, 2);
		} else {
			$itemId = (int) $id;
			$q = $this->db->createQuery()
				->select('alias')
				->from('#__one_content')
				->where('id = :id')
				->where('published = 1')
				->bind(':id', $itemId, ParameterType::INTEGER);

			$this->db->setQuery($q);
			$alias = $this->db->loadResult();
		}

		if (!$alias) {
			return [];
		}

		$pk = (int) (is_numeric($id) ? $id : explode(':', $id)[0]);

		return [$pk => $alias];
	}

	/**
	 * PARSE: alias → id
	 */
	public function getItemId(string $segment, array $query)
	{
		$alias = $segment;
		$q = $this->db->createQuery()
			->select('id')
			->from('#__one_content')
			->where('alias = :alias')
			->where('published = 1')
			->bind(':alias', $alias);

		$this->db->setQuery($q);

		return (int) $this->db->loadResult() ?: false;
	}

	/**
	 * SEGMENT: alias for event instead of id
	 */
	public function getEventSegment($id, array $query): array
	{
		$alias = null;

		if (str_contains((string) $id, ':')) {
			[, $alias] = explode(':', $id, 2);
		} else {
			$eventId = (int) $id;
			$q = $this->db->createQuery()
				->select('alias')
				->from('#__one_events')
				->where('id = :id')
				->where('published = 1')
				->bind(':id', $eventId, ParameterType::INTEGER);

			$this->db->setQuery($q);
			$alias = $this->db->loadResult();
		}

		if (!$alias) {
			return [];
		}

		$pk = (int) (is_numeric($id) ? $id : explode(':', $id)[0]);

		return [$pk => $alias];
	}

	/**
	 * PARSE: alias → id for event
	 */
	public function getEventId(string $segment, array $query)
	{
		$alias = $segment;
		$q = $this->db->createQuery()
			->select('id')
			->from('#__one_events')
			->where('alias = :alias')
			->where('published = 1')
			->bind(':alias', $alias);

		$this->db->setQuery($q);

		return (int) $this->db->loadResult() ?: false;
	}
}
