<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Site\Model;

use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Model\ItemModel as BaseItemModel;
use Joomla\CMS\Table\Table;
use Joomla\Database\ParameterType;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Content Component Item Model
 *
 * @since  1.0.0
 */
class ItemModel extends BaseItemModel
{
	/**
	 * Model context string.
	 *
	 * @var        string
	 */
	protected $_context = 'com_onecore.item';

	/**
	 * Method to auto-populate the model state.
	 *
	 * @return void
	 */
	protected function populateState()
	{
		$app = Factory::getApplication();

		// Load state from the request.
		$pk = $app->getInput()->getInt('id');
		$this->setState('item.id', $pk);

		$offset = $app->getInput()->getUint('limitstart');
		$this->setState('list.offset', $offset);

		// Load the parameters.
		$params = $app->getParams();
		$this->setState('params', $params);
	}

	/**
	 * Method to get item data.
	 *
	 * @param   integer  $pk  The id of the item.
	 *
	 * @return  object|boolean  Menu item data object on success, boolean false
	 */
	public function getItem($pk = null)
	{
		$pk = (int) ($pk ?: $this->getState('item.id'));

		if ($this->_item === null) {
			$this->_item = [];
		}

		if (!isset($this->_item[$pk])) {
			try {
				$db    = $this->getDatabase();
				$query = $db->createQuery();

				$query->select(
					$db->quoteName(
						[
							'a.id',
							'a.title',
							'a.alias',
							'a.introtext',
							'a.fulltext',
							'a.published',
							'a.featured',
							'a.access',
							'a.created',
							'a.created_by',
							'a.modified',
							'a.modified_by',
							'a.publish_up',
							'a.publish_down',
							'a.language',
							'a.note',
							'a.images',
							'a.video_link',
							'a.metakey',
							'a.metadesc',
							'a.metadata',
						]
					)
				)
					->from($db->quoteName('#__one_content', 'a'))
					->where($db->quoteName('a.id') . ' = :pk')
					->bind(':pk', $pk, ParameterType::INTEGER);

				// Join on category table - get first category from content_categories table
				// Use #__one_categories instead of #__categories
				$query->select($db->quoteName('c.title', 'category_title'))
					->select($db->quoteName('c.alias', 'category_alias'))
					->select($db->quoteName('c.access', 'category_access'))
					->select($db->quoteName('cc.category_id', 'catid'))
					->leftJoin($db->quoteName('#__one_content_categories', 'cc'), $db->quoteName('cc.content_id') . ' = ' . $db->quoteName('a.id'))
					->leftJoin($db->quoteName('#__one_categories', 'c'), $db->quoteName('c.id') . ' = ' . $db->quoteName('cc.category_id'));

				// Join on user table.
				$query->select($db->quoteName('u.name', 'author'))
					->leftJoin($db->quoteName('#__users', 'u'), $db->quoteName('u.id') . ' = ' . $db->quoteName('a.created_by'));

				// Join on address table.
				$query->select($db->quoteName('addr.latitude', 'latitude'))
					->select($db->quoteName('addr.longitude', 'longitude'))
					->select($db->quoteName('addr.address', 'address'))
					->select($db->quoteName('addr.address_street', 'address_street'))
					->select($db->quoteName('addr.address_street_number', 'address_street_number'))
					->select($db->quoteName('addr.address_postal_code', 'address_postal_code'))
					->select($db->quoteName('addr.address_city', 'address_city'))
					->select($db->quoteName('addr.address_country', 'address_country'))
					->leftJoin($db->quoteName('#__one_content_addresses', 'addr'), $db->quoteName('addr.content_id') . ' = ' . $db->quoteName('a.id'));

				// Filter by access level.
				$user = Factory::getUser();
				if (!$user->authorise('core.admin')) {
					$groups = $user->getAuthorisedViewLevels();
					$query->whereIn($db->quoteName('a.access'), $groups);
				}

				// Filter by published state
				$query->where($db->quoteName('a.published') . ' = 1');

				$db->setQuery($query);
				$data = $db->loadObject();

				if (empty($data)) {
					throw new \Exception(\Joomla\CMS\Language\Text::_('COM_ONECORE_ERROR_ITEM_NOT_FOUND'), 404);
				}

				// Convert parameter fields to objects.
				$registry = new \Joomla\Registry\Registry();
				$registry->loadString($data->params ?? '{}');
				$data->params = $registry;

				// Load connected events from junction table
				$eventsQuery = $db->createQuery()
					->select([
						$db->quoteName('e.id'),
						$db->quoteName('e.title'),
						$db->quoteName('e.alias'),
						$db->quoteName('e.description'),
						$db->quoteName('e.event_start'),
						$db->quoteName('e.event_end'),
						$db->quoteName('e.images'),
						$db->quoteName('e.published'),
						$db->quoteName('e.access'),
					])
					->from($db->quoteName('#__one_content_events', 'ce'))
					->innerJoin($db->quoteName('#__one_events', 'e'), $db->quoteName('e.id') . ' = ' . $db->quoteName('ce.event_id'))
					->where($db->quoteName('ce.content_id') . ' = :contentId')
					->where($db->quoteName('e.published') . ' = 1')
					->bind(':contentId', $pk, ParameterType::INTEGER)
					->order($db->quoteName('e.event_start') . ' ASC');

				// Filter by access level for events
				if (!$user->authorise('core.admin')) {
					$groups = $user->getAuthorisedViewLevels();
					$eventsQuery->whereIn($db->quoteName('e.access'), $groups);
				}

				$db->setQuery($eventsQuery);
				$events = $db->loadObjectList();

				// Load addresses, images, and custom fields for each event
				foreach ($events as $event) {
					// Load address
					$addressQuery = $db->createQuery()
						->select([
							$db->quoteName('address'),
							$db->quoteName('address_street'),
							$db->quoteName('address_street_number'),
							$db->quoteName('address_postal_code'),
							$db->quoteName('address_city'),
							$db->quoteName('address_country'),
							$db->quoteName('latitude'),
							$db->quoteName('longitude'),
						])
						->from($db->quoteName('#__one_event_addresses'))
						->where($db->quoteName('event_id') . ' = :eventId')
						->bind(':eventId', $event->id, ParameterType::INTEGER);
					
					$db->setQuery($addressQuery);
					$addressData = $db->loadObject();
					
					if ($addressData) {
						$event->address = $addressData->address ?? '';
						$event->address_street = $addressData->address_street ?? '';
						$event->address_street_number = $addressData->address_street_number ?? '';
						$event->address_postal_code = $addressData->address_postal_code ?? '';
						$event->address_city = $addressData->address_city ?? '';
						$event->address_country = $addressData->address_country ?? '';
						$event->latitude = $addressData->latitude ?? '';
						$event->longitude = $addressData->longitude ?? '';
					} else {
						$event->address = '';
						$event->address_street = '';
						$event->address_street_number = '';
						$event->address_postal_code = '';
						$event->address_city = '';
						$event->address_country = '';
						$event->latitude = '';
						$event->longitude = '';
					}

					// Parse all images from images field using ContentHelper
					$event->all_images = \Comdev\Component\Onecore\Site\Helper\ContentHelper::getAllImages($event->images);

					// Load categories for event
					$categoriesQuery = $db->createQuery()
						->select([
							$db->quoteName('c.id'),
							$db->quoteName('c.title'),
							$db->quoteName('c.alias'),
						])
						->from($db->quoteName('#__one_event_categories', 'ec'))
						->innerJoin($db->quoteName('#__one_categories', 'c'), $db->quoteName('c.id') . ' = ' . $db->quoteName('ec.category_id'))
						->where($db->quoteName('ec.event_id') . ' = :eventId')
						->where($db->quoteName('c.published') . ' = 1')
						->order($db->quoteName('c.title') . ' ASC')
						->bind(':eventId', $event->id, ParameterType::INTEGER);
					
					$db->setQuery($categoriesQuery);
					$event->categories = $db->loadObjectList() ?: [];

					// Load custom fields for event
					$customFieldsQuery = $db->createQuery()
						->select([
							$db->quoteName('v.value', 'field_value'),
							$db->quoteName('f.id', 'field_id'),
							$db->quoteName('f.title', 'field_title'),
							$db->quoteName('f.type', 'field_type'),
							$db->quoteName('f.display_label'),
							$db->quoteName('f.label_position'),
							$db->quoteName('f.ordering', 'field_ordering'),
							$db->quoteName('g.id', 'group_id'),
							$db->quoteName('g.title', 'group_title'),
							$db->quoteName('g.description', 'group_description'),
							$db->quoteName('g.show_title', 'group_show_title'),
							$db->quoteName('g.show_description', 'group_show_description'),
							$db->quoteName('g.ordering', 'group_ordering'),
							$db->quoteName('g.pos_module', 'group_pos_module'),
							$db->quoteName('g.pos_item', 'group_pos_item'),
						])
						->from($db->quoteName('#__one_customfield_values', 'v'))
						->innerJoin($db->quoteName('#__one_customfields', 'f'), $db->quoteName('f.id') . ' = ' . $db->quoteName('v.field_id'))
						->leftJoin($db->quoteName('#__one_customfield_groups', 'g'), $db->quoteName('g.id') . ' = ' . $db->quoteName('f.group_id'))
						->where($db->quoteName('v.event_id') . ' = :eventId')
						->where($db->quoteName('f.published') . ' = 1')
						->where($db->quoteName('v.value') . " != ''")
						->order($db->quoteName('g.ordering') . ' ASC, ' . $db->quoteName('f.ordering') . ' ASC')
						->bind(':eventId', $event->id, ParameterType::INTEGER);
					
					$db->setQuery($customFieldsQuery);
					$customFieldsRows = $db->loadObjectList() ?: [];
					
					// Group custom fields by group
					$customFieldsGroupsCard = [];
					$customFieldsGroupsModal = [];
					foreach ($customFieldsRows as $row) {
						$gid = (int) ($row->group_id ?? 0);
						$posModule = (string) ($row->group_pos_module ?? 'none');
						$posItem = (string) ($row->group_pos_item ?? 'none');
						
						// Group for card display (pos_module - for module/card view)
						// Show fields with pos_module set (for cards in lists)
						if ($posModule !== 'none') {
							if (!isset($customFieldsGroupsCard[$gid])) {
								$customFieldsGroupsCard[$gid] = (object) [
									'id' => $gid,
									'title' => (string) ($row->group_title ?? ''),
									'description' => (string) ($row->group_description ?? ''),
									'show_title' => (int) ($row->group_show_title ?? 1),
									'show_description' => (int) ($row->group_show_description ?? 1),
									'pos_module' => $posModule,
									'pos_item' => $posItem,
									'fields' => [],
								];
							}
							$customFieldsGroupsCard[$gid]->fields[] = (object) [
								'id' => (int) $row->field_id,
								'title' => (string) $row->field_title,
								'type' => (string) $row->field_type,
								'value' => (string) $row->field_value,
								'display_label' => (int) ($row->display_label ?? 1),
								'label_position' => (string) ($row->label_position ?? 'next'),
							];
						}
						
						// Group for modal display:
						// - If pos_item is 'modal', show in modal
						// - If pos_item is 'sidebar' (for events, this should be replaced with modal in admin), show in modal
						$showInModal = ($posItem === 'modal' || $posItem === 'sidebar');
						
						if ($showInModal) {
							if (!isset($customFieldsGroupsModal[$gid])) {
								$customFieldsGroupsModal[$gid] = (object) [
									'id' => $gid,
									'title' => (string) ($row->group_title ?? ''),
									'description' => (string) ($row->group_description ?? ''),
									'show_title' => (int) ($row->group_show_title ?? 1),
									'show_description' => (int) ($row->group_show_description ?? 1),
									'pos_module' => $posModule,
									'pos_item' => $posItem,
									'fields' => [],
								];
							}
							$customFieldsGroupsModal[$gid]->fields[] = (object) [
								'id' => (int) $row->field_id,
								'title' => (string) $row->field_title,
								'type' => (string) $row->field_type,
								'value' => (string) $row->field_value,
								'display_label' => (int) ($row->display_label ?? 1),
								'label_position' => (string) ($row->label_position ?? 'next'),
							];
						}
					}
					$event->customfields_card = array_values($customFieldsGroupsCard);
					$event->customfields_modal = array_values($customFieldsGroupsModal);
				}

				// Add events to item data
				$data->events = $events ?: [];

				$this->_item[$pk] = $data;
			} catch (\Exception $e) {
				$this->setError($e);
				$this->_item[$pk] = false;
			}
		}

		return $this->_item[$pk];
	}
}
