<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Site\Model;

use Comdev\Component\Onecore\Site\Helper\CustomFieldsHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Model\ListModel;
use Joomla\Database\ParameterType;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Content Component List Model
 *
 * @since  1.0.0
 */
class ContentModel extends ListModel
{
	/**
	 * Model context string.
	 *
	 * @var  string
	 */
	protected $_context = 'com_onecore.content';

	/**
	 * Constructor.
	 *
	 * @param   array  $config  An optional associative array of configuration settings.
	 */
	public function __construct($config = [])
	{
		if (empty($config['filter_fields'])) {
			$config['filter_fields'] = [
				'id', 'a.id',
				'title', 'a.title',
				'alias', 'a.alias',
				'state', 'a.published',
				'featured', 'a.featured',
				'created', 'a.created',
				'created_by', 'a.created_by',
				'ordering', 'a.ordering',
				'language', 'a.language',
				'hits', 'a.hits',
			];
		}

		parent::__construct($config);
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * @param   string  $ordering   An optional ordering field.
	 * @param   string  $direction  An optional direction (asc|desc).
	 *
	 * @return  void
	 */
	protected function populateState($ordering = 'a.ordering', $direction = 'ASC')
	{
		$app = Factory::getApplication();

		// Load the parameters.
		$params = $app->getParams();
		$this->setState('params', $params);

		// Get category ID from menu item (request.id) or input
		// In Joomla menu items, request.id is stored in the menu item's query array
		// Priority: 1. Active menu item request.id (if available), 2. URL parameter id
		$active = $app->getMenu()->getActive();
		$categoryId = 0;
		
		// First, try to get category_id from active menu item's request.id
		if ($active && isset($active->query['view']) && $active->query['view'] === 'content') {
			if (isset($active->query['id']) && $active->query['id'] > 0) {
				// Get from menu item request parameters
				$categoryId = (int) $active->query['id'];
			}
		}
		
		// If no category_id from menu item, check URL parameter
		// But only use URL id if it's explicitly for category filtering
		// URL id might be content id, so we need to be careful
		if ($categoryId === 0) {
			$urlId = $app->getInput()->getInt('id', 0);
			// Only use URL id if it's positive (0 means no filter)
			// Note: This assumes id in URL is category_id when no menu item id is set
			if ($urlId > 0) {
				$categoryId = $urlId;
			}
		}
		
		if ($categoryId > 0) {
			$this->setState('filter.category_id', $categoryId);
		}

		// List state information
		$input = $app->getInput();

		// Featured filter
		$showFeatured = $params->get('show_featured', 0);
		if ($showFeatured == 1) {
			$this->setState('filter.show_featured', 1);
		}

		$featured = $input->getInt('featured', 0);
		if ($featured == 1) {
			$this->setState('filter.featured', 1);
		}
		$defaultLimit = $app->getUserStateFromRequest('global.list.limit', 'limit', $app->get('list_limit', 20), 'uint');
		$limit = $input->getInt('limit', $defaultLimit);
		$this->setState('list.limit', $limit);

		$limitstart = $input->get('limitstart', 0, 'uint');
		$this->setState('list.start', $limitstart);

		// Filter by published state
		$this->setState('filter.published', 1);

		// Search filter
		$search = $input->getString('search', '');
		if (!empty($search)) {
			$this->setState('filter.search', $search);
		}

		// Address filter
		$address = $input->getString('address', '');
		if (trim($address) !== '') {
			$this->setState('filter.address', trim($address));
		}

		// Category filter (multi-select)
		$categories = $input->get('category', [], 'array');
		if (!empty($categories) && is_array($categories)) {
			$categories = array_filter(array_map('intval', $categories));
			if (!empty($categories)) {
				$this->setState('filter.categories', $categories);
			}
		}

		// Date filters
		$dateFrom = $input->getString('date_from', '');
		$dateTo = $input->getString('date_to', '');
		if (!empty($dateFrom)) {
			$this->setState('filter.date_from', $dateFrom);
		}
		if (!empty($dateTo)) {
			$this->setState('filter.date_to', $dateTo);
		}

		// Author filter
		$author = $input->getInt('author', 0);
		if ($author > 0) {
			$this->setState('filter.author', $author);
		}

		// Custom fields filters (cf_* parameters)
		$customFieldsFilters = [];
		$allInput = $input->getArray();
		foreach ($allInput as $key => $value) {
			if (strpos($key, 'cf_') === 0) {
				$fieldId = (int) substr($key, 3);
				if ($fieldId > 0 && !empty($value)) {
					$customFieldsFilters[$fieldId] = is_array($value) ? $value : $value;
				}
			}
		}
		if (!empty($customFieldsFilters)) {
			$this->setState('filter.custom_fields', $customFieldsFilters);
		}

		// Ordering - handle sort parameter (newest, oldest, a-z, z-a)
		$sortParam = $input->getString('sort', '');
		if (!empty($sortParam)) {
			switch ($sortParam) {
				case 'newest':
					$orderCol = 'a.created';
					$orderDirn = 'DESC';
					break;
				case 'oldest':
					$orderCol = 'a.created';
					$orderDirn = 'ASC';
					break;
				case 'a-z':
					$orderCol = 'a.title';
					$orderDirn = 'ASC';
					break;
				case 'z-a':
					$orderCol = 'a.title';
					$orderDirn = 'DESC';
					break;
				default:
					$orderCol = $ordering;
					$orderDirn = $direction;
			}
		} else {
			$orderCol = $input->get('filter_order', $ordering);
			if (!in_array($orderCol, $this->filter_fields)) {
				$orderCol = $ordering;
			}
			$orderDirn = $input->get('filter_order_Dir', $direction);
			if (!in_array(strtoupper($orderDirn), ['ASC', 'DESC', ''])) {
				$orderDirn = $direction;
			}
		}

		$this->setState('list.ordering', $orderCol);
		$this->setState('list.direction', $orderDirn);
	}

	/**
	 * Method to get a store id based on model configuration state.
	 *
	 * @param   string  $id  A prefix for the store id.
	 *
	 * @return  string  A store id.
	 */
	protected function getStoreId($id = '')
	{
		$id .= ':' . $this->getState('list.start');
		$id .= ':' . $this->getState('list.limit');
		$id .= ':' . $this->getState('list.ordering');
		$id .= ':' . $this->getState('list.direction');
		$id .= ':' . $this->getState('filter.published');
		$id .= ':' . $this->getState('filter.category_id');
		$id .= ':' . $this->getState('filter.search');
		$id .= ':' . serialize($this->getState('filter.categories', []));
		$id .= ':' . $this->getState('filter.date_from');
		$id .= ':' . $this->getState('filter.date_to');
		$id .= ':' . $this->getState('filter.show_featured');
		$id .= ':' . $this->getState('filter.featured');
		$id .= ':' . $this->getState('filter.author');
		$id .= ':' . $this->getState('filter.address');
		$id .= ':' . serialize($this->getState('filter.custom_fields', []));
		
		// Add sort parameter to cache ID if exists
		$app = Factory::getApplication();
		$sortParam = $app->getInput()->getString('sort', '');
		if (!empty($sortParam)) {
			$id .= ':' . $sortParam;
		}

		return parent::getStoreId($id);
	}

	/**
	 * Build an SQL query to load the list data.
	 *
	 * @return  \Joomla\Database\QueryInterface
	 */
	protected function getListQuery()
	{
		$db    = $this->getDatabase();
		$query = $db->createQuery();

		// Ensure custom fields schema is present before building search EXISTS subquery
		CustomFieldsHelper::ensureReady();

		// Select the required fields from the table.
		$query->select(
			$db->quoteName(
				[
					'a.id',
					'a.title',
					'a.alias',
					'a.introtext',
					'a.fulltext',
					'a.published',
					'a.featured',
					'a.created',
					'a.created_by',
					'a.modified',
					'a.hits',
					'a.language',
					'a.images',
					'a.video_link',
				]
			)
		)
			->from($db->quoteName('#__one_content', 'a'));

		// Note: Categories are loaded separately via ContentHelper::getCategories() in the view
		// We don't join categories here to avoid duplicate rows when content has multiple categories

		// Join on user table.
		$query->select($db->quoteName('u.name', 'author'))
			->leftJoin($db->quoteName('#__users', 'u'), $db->quoteName('u.id') . ' = ' . $db->quoteName('a.created_by'));

		// Join on address table.
		$query->select($db->quoteName('addr.latitude', 'latitude'))
			->select($db->quoteName('addr.longitude', 'longitude'))
			->select($db->quoteName('addr.address', 'address'))
			->leftJoin($db->quoteName('#__one_content_addresses', 'addr'), $db->quoteName('addr.content_id') . ' = ' . $db->quoteName('a.id'));

		// Filter by category (single category from menu/URL)
		$categoryId = $this->getState('filter.category_id');
		if (is_numeric($categoryId)) {
			$query->where($db->quoteName('a.id') . ' IN (
				SELECT ' . $db->quoteName('content_id') . ' 
				FROM ' . $db->quoteName('#__one_content_categories') . ' 
				WHERE ' . $db->quoteName('category_id') . ' = :category_id
			)')
				->bind(':category_id', $categoryId, ParameterType::INTEGER);
		}

		// Filter by categories (multi-select from search form)
		$categories = $this->getState('filter.categories');
		if (!empty($categories) && is_array($categories)) {
			$categories = array_filter(array_map('intval', $categories));
			if (!empty($categories)) {
				$query->where($db->quoteName('a.id') . ' IN (
					SELECT ' . $db->quoteName('content_id') . ' 
					FROM ' . $db->quoteName('#__one_content_categories') . ' 
					WHERE ' . $db->quoteName('category_id') . ' IN (' . implode(',', $categories) . ')
				)');
			}
		}

		// Address filter (search in address)
		$address = $this->getState('filter.address');
		if (!empty($address)) {
			$addressTerm = '%' . $db->escape($address, true) . '%';
			$query->where('(' . $db->quoteName('addr.address') . ' LIKE :address1
				OR ' . $db->quoteName('addr.address_street') . ' LIKE :address2
				OR ' . $db->quoteName('addr.address_city') . ' LIKE :address3
				OR ' . $db->quoteName('addr.address_postal_code') . ' LIKE :address4
				OR ' . $db->quoteName('addr.address_country') . ' LIKE :address5)')
				->bind(':address1', $addressTerm)
				->bind(':address2', $addressTerm)
				->bind(':address3', $addressTerm)
				->bind(':address4', $addressTerm)
				->bind(':address5', $addressTerm);
		}

		// Search filter (text search in title, introtext, fulltext)
		$search = $this->getState('filter.search');
		if (!empty($search)) {
			$searchTerm = '%' . $db->escape($search, true) . '%';
			$query->where('(' . $db->quoteName('a.title') . ' LIKE :search1 
				OR ' . $db->quoteName('a.introtext') . ' LIKE :search2
				OR ' . $db->quoteName('a.fulltext') . ' LIKE :search3
				OR EXISTS (
					SELECT 1
					FROM ' . $db->quoteName('#__one_customfield_values', 'cfv') . '
					INNER JOIN ' . $db->quoteName('#__one_customfields', 'cff') . ' ON ' . $db->quoteName('cff.id') . ' = ' . $db->quoteName('cfv.field_id') . '
					INNER JOIN ' . $db->quoteName('#__one_categories', 'fcat') . ' ON ' . $db->quoteName('fcat.id') . ' = ' . $db->quoteName('cff.category_id') . '
					INNER JOIN ' . $db->quoteName('#__one_content_categories', 'ccc') . ' ON ' . $db->quoteName('ccc.content_id') . ' = ' . $db->quoteName('a.id') . '
					INNER JOIN ' . $db->quoteName('#__one_categories', 'ccat') . ' ON ' . $db->quoteName('ccat.id') . ' = ' . $db->quoteName('ccc.category_id') . '
					WHERE ' . $db->quoteName('cfv.content_id') . ' = ' . $db->quoteName('a.id') . '
					  AND ' . $db->quoteName('cff.published') . ' = 1
					  AND ' . $db->quoteName('cff.searchable') . ' = 1
					  AND ' . $db->quoteName('fcat.lft') . ' <= ' . $db->quoteName('ccat.lft') . '
					  AND ' . $db->quoteName('fcat.rgt') . ' >= ' . $db->quoteName('ccat.rgt') . '
					  AND ' . $db->quoteName('cfv.value') . ' LIKE :search4
				)
			)')
				->bind(':search1', $searchTerm)
				->bind(':search2', $searchTerm)
				->bind(':search3', $searchTerm)
				->bind(':search4', $searchTerm);
		}

		// Date filters
		$dateFrom = $this->getState('filter.date_from');
		$dateTo = $this->getState('filter.date_to');
		if (!empty($dateFrom)) {
			$query->where($db->quoteName('a.created') . ' >= :date_from')
				->bind(':date_from', $dateFrom);
		}
		if (!empty($dateTo)) {
			$query->where($db->quoteName('a.created') . ' <= :date_to')
				->bind(':date_to', $dateTo);
		}

		// Featured filter (only from URL parameter, not from menu params)
		// show_featured from menu params is for sorting only, not filtering
		$featured = $this->getState('filter.featured');
		if ($featured == 1) {
			$query->where($db->quoteName('a.featured') . ' = 1');
		}

		// Author filter
		$author = $this->getState('filter.author');
		if ($author > 0) {
			$query->where($db->quoteName('a.created_by') . ' = :author')
				->bind(':author', $author, ParameterType::INTEGER);
		}

		// Custom fields filters
		$customFieldsFilters = $this->getState('filter.custom_fields', []);
		if (!empty($customFieldsFilters) && is_array($customFieldsFilters)) {
			// Get field types for all filtered fields
			$fieldIds = array_keys($customFieldsFilters);
			$fieldIds = array_filter(array_map('intval', $fieldIds));
			$fieldTypes = [];
			if (!empty($fieldIds)) {
				$typeQuery = $db->createQuery()
					->select([$db->quoteName('id'), $db->quoteName('type')])
					->from($db->quoteName('#__one_customfields'))
					->whereIn($db->quoteName('id'), $fieldIds);
				$db->setQuery($typeQuery);
				try {
					$typeResults = $db->loadObjectList();
					foreach ($typeResults as $typeResult) {
						$fieldTypes[(int) $typeResult->id] = (string) $typeResult->type;
					}
				} catch (\Exception $e) {
					// Ignore errors
				}
			}

			foreach ($customFieldsFilters as $fieldId => $fieldValue) {
				$fieldId = (int) $fieldId;
				if ($fieldId <= 0) {
					continue;
				}

				$fieldType = $fieldTypes[$fieldId] ?? 'input';

				// Build subquery to check if content has this custom field value
				$subQuery = $db->createQuery()
					->select('1')
					->from($db->quoteName('#__one_customfield_values', 'cfv'))
					->innerJoin(
						$db->quoteName('#__one_customfields', 'cff'),
						$db->quoteName('cff.id') . ' = ' . $db->quoteName('cfv.field_id')
					)
					->where($db->quoteName('cfv.content_id') . ' = ' . $db->quoteName('a.id'))
					->where($db->quoteName('cfv.field_id') . ' = :cf_field_id_' . $fieldId)
					->where($db->quoteName('cff.published') . ' = 1')
					->where($db->quoteName('cff.searchable') . ' = 1');

				// Handle different value types and field types
				if (is_array($fieldValue)) {
					// Multiple values (for multi-select)
					$fieldValue = array_filter(array_map('trim', $fieldValue));
					if (!empty($fieldValue)) {
						if ($fieldType === 'select') {
							// For select fields, use exact match
							$subQuery->whereIn($db->quoteName('cfv.value'), $fieldValue);
						} else {
							// For text fields, use LIKE for each value
							$likeConditions = [];
							foreach ($fieldValue as $idx => $val) {
								$paramName = ':cf_value_' . $fieldId . '_' . $idx;
								$likeConditions[] = $db->quoteName('cfv.value') . ' LIKE ' . $paramName;
								// Prepare search value - bind() will escape it automatically
								$val = (string) trim($val);
								$searchValue = '%' . $val . '%';
								$query->bind($paramName, $searchValue, ParameterType::STRING);
							}
							if (!empty($likeConditions)) {
								$subQuery->where('(' . implode(' OR ', $likeConditions) . ')');
							}
						}
						$query->where('EXISTS (' . (string) $subQuery . ')');
						$fieldIdParam = ':cf_field_id_' . $fieldId;
						$query->bind($fieldIdParam, $fieldId, ParameterType::INTEGER);
					}
				} else {
					// Single value
					$fieldValue = trim($fieldValue);
					if (!empty($fieldValue)) {
						if ($fieldType === 'select') {
							// For select fields, use exact match
							$valueParam = ':cf_value_' . $fieldId;
							$subQuery->where($db->quoteName('cfv.value') . ' = ' . $valueParam);
							$query->bind($valueParam, $fieldValue);
						} else {
							// For text fields (input, textarea), use LIKE
							$valueParam = ':cf_value_' . $fieldId;
							$subQuery->where($db->quoteName('cfv.value') . ' LIKE ' . $valueParam);
							// Prepare search value - bind() will escape it automatically
							$fieldValue = (string) trim($fieldValue);
							$searchValue = '%' . $fieldValue . '%';
							$query->bind($valueParam, $searchValue, ParameterType::STRING);
						}
						$query->where('EXISTS (' . (string) $subQuery . ')');
						$fieldIdParam = ':cf_field_id_' . $fieldId;
						$query->bind($fieldIdParam, $fieldId, ParameterType::INTEGER);
					}
				}
			}
		}

		// Filter by published state
		$published = $this->getState('filter.published');
		if (is_numeric($published)) {
			$query->where($db->quoteName('a.published') . ' = :published')
				->bind(':published', $published, ParameterType::INTEGER);
		}

		// Filter by access level.
		$user = Factory::getUser();
		if (!$user->authorise('core.admin')) {
			$groups = $user->getAuthorisedViewLevels();
			$query->whereIn($db->quoteName('a.access'), $groups);
		}

		// Add the list ordering clause.
		$orderCol  = $this->state->get('list.ordering', 'a.ordering');
		$orderDirn = $this->state->get('list.direction', 'ASC');

		// If show_featured is enabled, sort featured items first
		$showFeatured = $this->getState('filter.show_featured');
		if ($showFeatured == 1) {
			// Add featured DESC first, then the original ordering
			if (strpos($orderCol, ',') !== false) {
				// If already complex ordering, prepend featured
				$query->order($db->quoteName('a.featured') . ' DESC, ' . $db->escape($orderCol) . ' ' . $db->escape($orderDirn));
			} else {
				// Single column ordering - add featured first
				if ($orderCol === 'ordering') {
					$orderCol = 'a.' . $orderCol;
				}
				$query->order($db->quoteName('a.featured') . ' DESC, ' . $db->escape($orderCol) . ' ' . $db->escape($orderDirn));
			}
		} else {
			// Handle complex ordering (e.g., "a.created DESC, a.ordering")
			if (strpos($orderCol, ',') !== false) {
			// Multiple columns - split and process each
			$orderParts = explode(',', $orderCol);
			$orderClauses = [];
			foreach ($orderParts as $part) {
				$part = trim($part);
				// Check if direction is in the part
				if (preg_match('/\s+(ASC|DESC)$/i', $part, $matches)) {
					$orderClauses[] = $db->escape($part);
				} else {
					// Ensure ordering column has table alias
					if ($part === 'ordering') {
						$part = 'a.' . $part;
					}
					$orderClauses[] = $db->escape($part) . ' ' . $db->escape($orderDirn);
				}
			}
			$query->order(implode(', ', $orderClauses));
			} else {
				// Single column ordering
				// Ensure ordering column has table alias to avoid ambiguity
				if ($orderCol === 'ordering') {
					$orderCol = 'a.' . $orderCol;
				}
				$query->order($db->escape($orderCol) . ' ' . $db->escape($orderDirn));
			}
		}

		return $query;
	}
}
