<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Site\Helper;

use Joomla\CMS\Factory;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Reviews helper (OneCore Plus). Get reviews and average rating for content or event.
 *
 * @since  1.0.16
 */
class ReviewsHelper
{
	/**
	 * Get published and approved reviews for a content item or event.
	 *
	 * @param   int  $contentId  Content ID (use 0 for event).
	 * @param   int  $eventId     Event ID (use 0 for content).
	 *
	 * @return  \stdClass[]
	 *
	 * @since   1.0.16
	 */
	public static function getReviews($contentId = 0, $eventId = 0)
	{
		if (($contentId <= 0 && $eventId <= 0) || ($contentId > 0 && $eventId > 0)) {
			return [];
		}

		$db = Factory::getDbo();
		$query = $db->getQuery(true)
			->select(['a.id', 'a.name', 'a.title', 'a.rating', 'a.body', 'a.created'])
			->from($db->quoteName('#__one_reviews', 'a'))
			->where($db->quoteName('a.published') . ' = 1')
			->where($db->quoteName('a.approved') . ' = 1')
			->order($db->quoteName('a.created') . ' DESC');

		if ($contentId > 0) {
			$query->where($db->quoteName('a.content_id') . ' = ' . (int) $contentId);
		} else {
			$query->where($db->quoteName('a.event_id') . ' = ' . (int) $eventId);
		}

		$db->setQuery($query);
		$list = $db->loadObjectList();

		return $list ?: [];
	}

	/**
	 * Get average rating and count for a content item or event.
	 *
	 * @param   int  $contentId  Content ID (use 0 for event).
	 * @param   int  $eventId    Event ID (use 0 for content).
	 *
	 * @return  object  { average: float, count: int }
	 *
	 * @since   1.0.16
	 */
	public static function getAverageRating($contentId = 0, $eventId = 0)
	{
		$result = (object) ['average' => 0.0, 'count' => 0];

		if (($contentId <= 0 && $eventId <= 0) || ($contentId > 0 && $eventId > 0)) {
			return $result;
		}

		$db = Factory::getDbo();
		$query = $db->getQuery(true)
			->select('AVG(' . $db->quoteName('a.rating') . ') AS average, COUNT(*) AS count')
			->from($db->quoteName('#__one_reviews', 'a'))
			->where($db->quoteName('a.published') . ' = 1')
			->where($db->quoteName('a.approved') . ' = 1')
			->where($db->quoteName('a.rating') . ' >= 1');

		if ($contentId > 0) {
			$query->where($db->quoteName('a.content_id') . ' = ' . (int) $contentId);
		} else {
			$query->where($db->quoteName('a.event_id') . ' = ' . (int) $eventId);
		}

		$db->setQuery($query);
		$row = $db->loadObject();

		if ($row && (int) $row->count > 0) {
			$result->average = round((float) $row->average, 1);
			$result->count = (int) $row->count;
		}

		return $result;
	}
}
