<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Site\Helper;

use Joomla\CMS\Factory;
use Joomla\Database\ParameterType;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Custom fields helper.
 *
 * @since  1.0.4
 */
class CustomFieldsHelper
{
	/**
	 * Ensure custom fields tables exist (runtime fallback for installs where SQL updates were not applied).
	 *
	 * @return void
	 *
	 * @since  1.0.5
	 */
	/**
	 * Public wrapper so other code (component model/module) can ensure schema before querying.
	 *
	 * @return void
	 *
	 * @since  1.0.6
	 */
	public static function ensureReady(): void
	{
		self::ensureTables();
		self::ensureColumns();
		self::migrateFromLegacyMapping();
	}

	private static function ensureTables(): void
	{
		$db = Factory::getDbo();
		$prefix = $db->getPrefix();

		try {
			$existing = $db->getTableList();
		} catch (\Exception $e) {
			return;
		}

		$need = [
			$prefix . 'one_customfield_groups',
			$prefix . 'one_customfields',
			$prefix . 'one_customfield_values',
		];

		$missing = array_diff($need, $existing);

		if (empty($missing)) {
			return;
		}

		// Best-effort create (MySQL); if it fails we will just behave as if no custom fields exist.
		$queries = [
			"CREATE TABLE IF NOT EXISTS `{$prefix}one_customfield_groups` (
				`id` int NOT NULL AUTO_INCREMENT,
				`category_id` int NOT NULL DEFAULT 0,
				`title` varchar(255) NOT NULL DEFAULT '',
				`description` text,
				`show_title` tinyint NOT NULL DEFAULT 1,
				`show_description` tinyint NOT NULL DEFAULT 1,
				`pos_items` varchar(20) NOT NULL DEFAULT 'none',
				`pos_item` varchar(20) NOT NULL DEFAULT 'none',
				`pos_module` varchar(20) NOT NULL DEFAULT 'none',
				`ordering` int NOT NULL DEFAULT 0,
				`published` tinyint NOT NULL DEFAULT 1,
				`created` datetime NOT NULL,
				`modified` datetime,
				PRIMARY KEY (`id`),
				KEY `idx_category_id` (`category_id`),
				KEY `idx_published` (`published`),
				KEY `idx_ordering` (`ordering`)
			) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 DEFAULT COLLATE=utf8mb4_unicode_ci",
			"CREATE TABLE IF NOT EXISTS `{$prefix}one_customfields` (
				`id` int NOT NULL AUTO_INCREMENT,
				`group_id` int NOT NULL DEFAULT 0,
				`category_id` int NOT NULL DEFAULT 0,
				`title` varchar(255) NOT NULL DEFAULT '',
				`type` varchar(20) NOT NULL DEFAULT 'input',
				`required` tinyint NOT NULL DEFAULT 0,
				`height` int NOT NULL DEFAULT 0,
				`ordering` int NOT NULL DEFAULT 0,
				`display_label` tinyint NOT NULL DEFAULT 1,
				`label_position` varchar(10) NOT NULL DEFAULT 'next',
				`searchable` tinyint NOT NULL DEFAULT 0,
				`published` tinyint NOT NULL DEFAULT 1,
				`created` datetime NOT NULL,
				`modified` datetime,
				PRIMARY KEY (`id`),
				KEY `idx_group_id` (`group_id`),
				KEY `idx_published` (`published`),
				KEY `idx_category_id` (`category_id`),
				KEY `idx_ordering` (`ordering`),
				KEY `idx_searchable` (`searchable`),
				KEY `idx_type` (`type`)
			) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 DEFAULT COLLATE=utf8mb4_unicode_ci",
			"CREATE TABLE IF NOT EXISTS `{$prefix}one_customfield_values` (
				`id` int NOT NULL AUTO_INCREMENT,
				`content_id` int NOT NULL,
				`field_id` int NOT NULL,
				`value` mediumtext NOT NULL,
				`created` datetime NOT NULL,
				`modified` datetime,
				PRIMARY KEY (`id`),
				UNIQUE KEY `idx_content_field` (`content_id`, `field_id`),
				KEY `idx_field_id` (`field_id`)
			) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 DEFAULT COLLATE=utf8mb4_unicode_ci",
		];

		foreach ($queries as $sql) {
			try {
				$db->setQuery($sql)->execute();
			} catch (\Exception $e) {
				// ignore
			}
		}
	}

	/**
	 * Ensure required columns exist on existing installs (tables might exist from earlier version).
	 *
	 * @return void
	 *
	 * @since  1.0.6
	 */
	private static function ensureColumns(): void
	{
		$db = Factory::getDbo();
		$prefix = $db->getPrefix();

		try {
			$cols = $db->getTableColumns($prefix . 'one_customfields', false);
			if (!isset($cols['category_id'])) {
				try {
					$db->setQuery("ALTER TABLE `{$prefix}one_customfields` ADD COLUMN `category_id` int NOT NULL DEFAULT 0 AFTER `id`")->execute();
				} catch (\Exception $e) {
					// ignore
				}
			}
			if (!isset($cols['group_id'])) {
				try {
					$db->setQuery("ALTER TABLE `{$prefix}one_customfields` ADD COLUMN `group_id` int NOT NULL DEFAULT 0 AFTER `id`")->execute();
				} catch (\Exception $e) {
					// ignore
				}
			}
		} catch (\Exception $e) {
			// ignore
		}

		try {
			$keys = $db->getTableKeys($prefix . 'one_customfields');
			$hasKey = false;
			foreach ($keys as $key) {
				if (isset($key->Key_name) && $key->Key_name === 'idx_category_id') {
					$hasKey = true;
					break;
				}
			}
			if (!$hasKey) {
				try {
					$db->setQuery("ALTER TABLE `{$prefix}one_customfields` ADD KEY `idx_category_id` (`category_id`)")->execute();
				} catch (\Exception $e) {
					// ignore
				}
			}
		} catch (\Exception $e) {
			// ignore
		}

		// Feather icon (for type input)
		try {
			$cols = $db->getTableColumns($prefix . 'one_customfields', false);
			if (!isset($cols['icon'])) {
				try {
					$db->setQuery("ALTER TABLE `{$prefix}one_customfields` ADD COLUMN `icon` varchar(64) NOT NULL DEFAULT '' AFTER `type`")->execute();
				} catch (\Exception $e) {
					// ignore
				}
			}
		} catch (\Exception $e) {
			// ignore
		}
	}

	/**
	 * Best-effort migration from legacy N:N mapping table if it exists.
	 * If fields have category_id = 0 and mapping exists, set category_id to MIN(category_id) per field.
	 *
	 * @return void
	 *
	 * @since  1.0.6
	 */
	private static function migrateFromLegacyMapping(): void
	{
		$db = Factory::getDbo();
		$prefix = $db->getPrefix();

		try {
			$tables = $db->getTableList();
			if (!in_array($prefix . 'one_customfield_categories', $tables, true)) {
				return;
			}

			$sql = "UPDATE `{$prefix}one_customfields` f
				INNER JOIN (
					SELECT field_id, MIN(category_id) AS category_id
					FROM `{$prefix}one_customfield_categories`
					GROUP BY field_id
				) x ON x.field_id = f.id
				SET f.category_id = x.category_id
				WHERE f.category_id = 0";

			$db->setQuery($sql)->execute();
		} catch (\Exception $e) {
			// ignore
		}
	}

	/**
	 * Load custom fields (with values) for a set of content ids, filtered by context+position.
	 *
	 * @param   array   $contentIds  Content ids
	 * @param   string  $context     items|item|module
	 * @param   string  $position    under_title|under_intro|sidebar
	 * @param   bool    $onlySearchable  If true, only return fields with searchable = 1
	 *
	 * @return  array  content_id => array(field objects)
	 *
	 * @since  1.0.4
	 */
	public static function getFieldsForContentIds(array $contentIds, string $context, string $position, bool $onlySearchable = false): array
	{
		self::ensureReady();

		$contentIds = array_values(array_filter(array_map('intval', $contentIds)));

		if (empty($contentIds)) {
			return [];
		}

		$posColumn = match ($context) {
			'items' => 'pos_items',
			'item' => 'pos_item',
			'module' => 'pos_module',
			default => null,
		};

		if ($posColumn === null) {
			return [];
		}

		// Validate position against context
		$allowed = ['under_title', 'under_intro'];
		if ($context === 'item') {
			$allowed[] = 'sidebar';
		}
		if (!in_array($position, $allowed, true)) {
			return [];
		}

		$db = Factory::getDbo();
		$query = $db->createQuery()
			->select([
				$db->quoteName('v.content_id'),
				$db->quoteName('f.id', 'field_id'),
				$db->quoteName('f.group_id', 'field_group_id'),
				$db->quoteName('f.title', 'field_title'),
				$db->quoteName('f.type', 'field_type'),
				$db->quoteName('f.icon', 'field_icon'),
				$db->quoteName('f.options', 'field_options'),
				$db->quoteName('f.display_label'),
				$db->quoteName('f.label_position'),
				$db->quoteName('f.ordering', 'field_ordering'),
				$db->quoteName('v.value', 'field_value'),
				$db->quoteName('g.id', 'group_id'),
				$db->quoteName('g.title', 'group_title'),
				$db->quoteName('g.description', 'group_description'),
				$db->quoteName('g.show_title', 'group_show_title'),
				$db->quoteName('g.show_description', 'group_show_description'),
				$db->quoteName('g.ordering', 'group_ordering'),
			])
			->from($db->quoteName('#__one_customfield_values', 'v'))
			->innerJoin(
				$db->quoteName('#__one_customfields', 'f'),
				$db->quoteName('f.id') . ' = ' . $db->quoteName('v.field_id')
			)
			->innerJoin(
				$db->quoteName('#__one_customfield_groups', 'g'),
				$db->quoteName('g.id') . ' = ' . $db->quoteName('f.group_id')
			)
			->leftJoin($db->quoteName('#__one_categories', 'gcat'), $db->quoteName('gcat.id') . ' = ' . $db->quoteName('g.category_id'))
			->whereIn($db->quoteName('v.content_id'), $contentIds)
			->where($db->quoteName('f.published') . ' = 1')
			->where($db->quoteName('g.published') . ' = 1')
			->where($db->quoteName('g.' . $posColumn) . ' = :position')
			->where($db->quoteName('v.value') . " != ''");
		
		// Filter by searchable if requested
		if ($onlySearchable) {
			$query->where($db->quoteName('f.searchable') . ' = 1');
		}
		
		$query
			// Inheritance: group category is ancestor of at least one content category (or group has no category)
			// If group is assigned to ROOT category (parent_id = 0, level = 0), it should be available for all content
			->where('(
				' . $db->quoteName('g.category_id') . ' = 0
				OR (
					' . $db->quoteName('gcat.parent_id') . ' = 0 
					AND ' . $db->quoteName('gcat.level') . ' = 0
				)
				OR EXISTS (
					SELECT 1
					FROM ' . $db->quoteName('#__one_content_categories', 'cc') . '
					INNER JOIN ' . $db->quoteName('#__one_categories', 'ccat') . ' ON ' . $db->quoteName('ccat.id') . ' = ' . $db->quoteName('cc.category_id') . '
					WHERE ' . $db->quoteName('cc.content_id') . ' = ' . $db->quoteName('v.content_id') . '
					  AND ' . $db->quoteName('gcat.lft') . ' <= ' . $db->quoteName('ccat.lft') . '
					  AND ' . $db->quoteName('gcat.rgt') . ' >= ' . $db->quoteName('ccat.rgt') . '
				)
			)')
			->order($db->quoteName('g.ordering') . ' ASC, ' . $db->quoteName('g.id') . ' ASC, ' . $db->quoteName('f.ordering') . ' ASC, ' . $db->quoteName('f.id') . ' ASC')
			->bind(':position', $position);

		$db->setQuery($query);

		try {
			$rows = $db->loadObjectList() ?: [];
		} catch (\Exception $e) {
			return [];
		}

		// Group by content_id, then by group_id
		$grouped = [];
		foreach ($rows as $row) {
			$cid = (int) $row->content_id;
			$gid = (int) ($row->group_id ?? 0);

			// Initialize content_id array if not exists
			if (!isset($grouped[$cid])) {
				$grouped[$cid] = [];
			}

			// Initialize group if not exists
			if (!isset($grouped[$cid][$gid])) {
				$grouped[$cid][$gid] = (object) [
					'id' => $gid,
					'title' => (string) ($row->group_title ?? ''),
					'description' => (string) ($row->group_description ?? ''),
					'show_title' => (int) ($row->group_show_title ?? 1),
					'show_description' => (int) ($row->group_show_description ?? 1),
					'ordering' => (int) ($row->group_ordering ?? 0),
					'fields' => [],
				];
			}

			// Add field to group
			$fieldValue = (string) $row->field_value;
			$fieldType = (string) $row->field_type;
			$fieldOptions = (string) ($row->field_options ?? '');
			
			// For multiselect, parse JSON array and format as comma-separated
			if ($fieldType === 'multiselect' && !empty($fieldValue)) {
				$decoded = json_decode($fieldValue, true);
				if (is_array($decoded) && !empty($decoded)) {
					// Convert values to labels if options are available
					if (!empty($fieldOptions)) {
						$options = explode("\n", $fieldOptions);
						$optionMap = [];
						foreach ($options as $option) {
							$option = trim($option);
							if (empty($option)) {
								continue;
							}
							// Support format: value|Label or just value
							if (strpos($option, '|') !== false) {
								list($optValue, $optLabel) = explode('|', $option, 2);
								$optionMap[trim($optValue)] = trim($optLabel);
							} else {
								$optionMap[$option] = $option;
							}
						}
						// Map decoded values to labels
						$labels = [];
						foreach ($decoded as $val) {
							$labels[] = $optionMap[$val] ?? $val;
						}
						$fieldValue = implode(', ', $labels);
					} else {
						// No options, just join values with comma
						$fieldValue = implode(', ', $decoded);
					}
				}
			}
			// For select and radio fields, convert value to label if options are available
			elseif (($fieldType === 'select' || $fieldType === 'radio') && !empty($fieldOptions) && !empty($fieldValue)) {
				$options = explode("\n", $fieldOptions);
				foreach ($options as $option) {
					$option = trim($option);
					if (empty($option)) {
						continue;
					}
					// Support format: value|Label or just value
					if (strpos($option, '|') !== false) {
						list($optValue, $optLabel) = explode('|', $option, 2);
						if (trim($optValue) === $fieldValue) {
							$fieldValue = trim($optLabel);
							break;
						}
					} elseif ($option === $fieldValue) {
						$fieldValue = $option;
						break;
					}
				}
			}
			
			$field = (object) [
				'id' => (int) $row->field_id,
				'group_id' => $gid,
				'title' => (string) $row->field_title,
				'type' => $fieldType,
				'icon' => (string) ($row->field_icon ?? ''),
				'value' => $fieldValue,
				'display_label' => (int) ($row->display_label ?? 1),
				'label_position' => (string) ($row->label_position ?? 'next'),
				'ordering' => (int) ($row->field_ordering ?? 0),
			];

			$grouped[$cid][$gid]->fields[] = $field;
		}

		// Convert to indexed array (remove group_id keys, keep order)
		$result = [];
		foreach ($grouped as $cid => $groups) {
			$result[$cid] = array_values($groups);
		}

		return $result;
	}

	/**
	 * Get custom fields for multiple event IDs, grouped by event and group
	 *
	 * @param   array   $eventIds       Array of event IDs
	 * @param   string  $context        Context: 'items', 'item', or 'module'
	 * @param   string  $position       Position: 'under_title', 'under_intro', 'sidebar', 'modal', 'card'
	 * @param   bool    $onlySearchable Only return searchable fields
	 *
	 * @return  array  Array indexed by event_id, containing arrays of group objects with fields
	 *
	 * @since   1.0.0
	 */
	public static function getFieldsForEventIds(array $eventIds, string $context, string $position, bool $onlySearchable = false): array
	{
		self::ensureReady();

		$eventIds = array_values(array_filter(array_map('intval', $eventIds)));

		if (empty($eventIds)) {
			return [];
		}

		$posColumn = match ($context) {
			'items' => 'pos_items',
			'item' => 'pos_item',
			'module' => 'pos_module',
			default => null,
		};

		if ($posColumn === null) {
			return [];
		}

		// Validate position against context
		$allowed = ['under_title', 'under_intro'];
		if ($context === 'item') {
			$allowed[] = 'sidebar';
			$allowed[] = 'modal';
		}
		if ($context === 'items' || $context === 'module') {
			$allowed[] = 'card';
		}
		if (!in_array($position, $allowed, true)) {
			return [];
		}

		$db = Factory::getDbo();
		$query = $db->createQuery()
			->select([
				$db->quoteName('v.event_id'),
				$db->quoteName('f.id', 'field_id'),
				$db->quoteName('f.group_id', 'field_group_id'),
				$db->quoteName('f.title', 'field_title'),
				$db->quoteName('f.type', 'field_type'),
				$db->quoteName('f.icon', 'field_icon'),
				$db->quoteName('f.options', 'field_options'),
				$db->quoteName('f.display_label'),
				$db->quoteName('f.label_position'),
				$db->quoteName('f.ordering', 'field_ordering'),
				$db->quoteName('v.value', 'field_value'),
				$db->quoteName('g.id', 'group_id'),
				$db->quoteName('g.title', 'group_title'),
				$db->quoteName('g.description', 'group_description'),
				$db->quoteName('g.show_title', 'group_show_title'),
				$db->quoteName('g.show_description', 'group_show_description'),
				$db->quoteName('g.ordering', 'group_ordering'),
			])
			->from($db->quoteName('#__one_customfield_values', 'v'))
			->innerJoin(
				$db->quoteName('#__one_customfields', 'f'),
				$db->quoteName('f.id') . ' = ' . $db->quoteName('v.field_id')
			)
			->innerJoin(
				$db->quoteName('#__one_customfield_groups', 'g'),
				$db->quoteName('g.id') . ' = ' . $db->quoteName('f.group_id')
			)
			->leftJoin($db->quoteName('#__one_categories', 'gcat'), $db->quoteName('gcat.id') . ' = ' . $db->quoteName('g.category_id'))
			->whereIn($db->quoteName('v.event_id'), $eventIds)
			->where($db->quoteName('f.published') . ' = 1')
			->where($db->quoteName('g.published') . ' = 1')
			->where($db->quoteName('g.entity_type') . ' = ' . $db->quote('events'))
			->where($db->quoteName('g.' . $posColumn) . ' = :position')
			->where($db->quoteName('v.value') . " != ''");
		
		// Filter by searchable if requested
		if ($onlySearchable) {
			$query->where($db->quoteName('f.searchable') . ' = 1');
		}
		
		$query
			// Inheritance: group category is ancestor of at least one event category (or group has no category)
			->where('(
				' . $db->quoteName('g.category_id') . ' = 0
				OR (
					' . $db->quoteName('gcat.parent_id') . ' = 0 
					AND ' . $db->quoteName('gcat.level') . ' = 0
				)
				OR EXISTS (
					SELECT 1
					FROM ' . $db->quoteName('#__one_event_categories', 'ec') . '
					INNER JOIN ' . $db->quoteName('#__one_categories', 'ecat') . ' ON ' . $db->quoteName('ecat.id') . ' = ' . $db->quoteName('ec.category_id') . '
					WHERE ' . $db->quoteName('ec.event_id') . ' = ' . $db->quoteName('v.event_id') . '
					  AND ' . $db->quoteName('gcat.lft') . ' <= ' . $db->quoteName('ecat.lft') . '
					  AND ' . $db->quoteName('gcat.rgt') . ' >= ' . $db->quoteName('ecat.rgt') . '
				)
			)')
			->order($db->quoteName('g.ordering') . ' ASC, ' . $db->quoteName('g.id') . ' ASC, ' . $db->quoteName('f.ordering') . ' ASC, ' . $db->quoteName('f.id') . ' ASC')
			->bind(':position', $position);

		$db->setQuery($query);

		try {
			$rows = $db->loadObjectList() ?: [];
		} catch (\Exception $e) {
			return [];
		}

		// Group by event_id, then by group_id
		$grouped = [];
		foreach ($rows as $row) {
			$eid = (int) $row->event_id;
			$gid = (int) ($row->group_id ?? 0);

			// Initialize event_id array if not exists
			if (!isset($grouped[$eid])) {
				$grouped[$eid] = [];
			}

			// Initialize group if not exists
			if (!isset($grouped[$eid][$gid])) {
				$grouped[$eid][$gid] = (object) [
					'id' => $gid,
					'title' => (string) ($row->group_title ?? ''),
					'description' => (string) ($row->group_description ?? ''),
					'show_title' => (int) ($row->group_show_title ?? 1),
					'show_description' => (int) ($row->group_show_description ?? 1),
					'ordering' => (int) ($row->group_ordering ?? 0),
					'fields' => [],
				];
			}

			// Add field to group
			$fieldValue = (string) $row->field_value;
			$fieldType = (string) $row->field_type;
			$fieldOptions = (string) ($row->field_options ?? '');
			
			// For multiselect, parse JSON array and format as comma-separated
			if ($fieldType === 'multiselect' && !empty($fieldValue)) {
				$decoded = json_decode($fieldValue, true);
				if (is_array($decoded) && !empty($decoded)) {
					// Convert values to labels if options are available
					if (!empty($fieldOptions)) {
						$options = explode("\n", $fieldOptions);
						$optionMap = [];
						foreach ($options as $option) {
							$option = trim($option);
							if (empty($option)) {
								continue;
							}
							// Support format: value|Label or just value
							if (strpos($option, '|') !== false) {
								list($optValue, $optLabel) = explode('|', $option, 2);
								$optionMap[trim($optValue)] = trim($optLabel);
							} else {
								$optionMap[$option] = $option;
							}
						}
						// Map decoded values to labels
						$labels = [];
						foreach ($decoded as $val) {
							$labels[] = $optionMap[$val] ?? $val;
						}
						$fieldValue = implode(', ', $labels);
					} else {
						// No options, just join values with comma
						$fieldValue = implode(', ', $decoded);
					}
				}
			}
			// For select and radio fields, convert value to label if options are available
			elseif (($fieldType === 'select' || $fieldType === 'radio') && !empty($fieldOptions) && !empty($fieldValue)) {
				$options = explode("\n", $fieldOptions);
				foreach ($options as $option) {
					$option = trim($option);
					if (empty($option)) {
						continue;
					}
					// Support format: value|Label or just value
					if (strpos($option, '|') !== false) {
						list($optValue, $optLabel) = explode('|', $option, 2);
						if (trim($optValue) === $fieldValue) {
							$fieldValue = trim($optLabel);
							break;
						}
					} elseif ($option === $fieldValue) {
						$fieldValue = $option;
						break;
					}
				}
			}
			
			$field = (object) [
				'id' => (int) $row->field_id,
				'group_id' => $gid,
				'title' => (string) $row->field_title,
				'type' => $fieldType,
				'icon' => (string) ($row->field_icon ?? ''),
				'value' => $fieldValue,
				'display_label' => (int) ($row->display_label ?? 1),
				'label_position' => (string) ($row->label_position ?? 'next'),
				'ordering' => (int) ($row->field_ordering ?? 0),
			];

			$grouped[$eid][$gid]->fields[] = $field;
		}

		// Convert to indexed array (remove group_id keys, keep order)
		$result = [];
		foreach ($grouped as $eid => $groups) {
			$result[$eid] = array_values($groups);
		}

		return $result;
	}
}

