<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Site\Helper;

use Joomla\CMS\Factory;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Content Helper class
 *
 * @since  1.0.0
 */
class ContentHelper
{
	/**
	 * Extract first image from images JSON field.
	 *
	 * @param   string|array  $imagesJson  JSON string or array with images data
	 *
	 * @return  string|null  First image path or null
	 */
	public static function getFirstImage($imagesJson)
	{
		if (empty($imagesJson)) {
			return null;
		}

		// If it's already an array
		if (is_array($imagesJson)) {
			if (isset($imagesJson[0]) && is_array($imagesJson[0])) {
				// Subform array format: [{"image":"path"}, ...]
				foreach ($imagesJson as $imageData) {
					if (isset($imageData['image']) && !empty($imageData['image'])) {
						$imagePath = $imageData['image'];
						// Handle new Joomla 4+ format: path#joomlaImage://...
						if (strpos($imagePath, '#') !== false) {
							$imagePath = explode('#', $imagePath)[0];
						}
						return $imagePath;
					}
				}
			} elseif (isset($imagesJson['image']) && !empty($imagesJson['image'])) {
				$imagePath = $imagesJson['image'];
				// Handle new Joomla 4+ format: path#joomlaImage://...
				if (strpos($imagePath, '#') !== false) {
					$imagePath = explode('#', $imagePath)[0];
				}
				return $imagePath;
			}
			return null;
		}

		// If it's not a string, return null
		if (!is_string($imagesJson)) {
			return null;
		}

		$trimmed = trim($imagesJson);
		if (empty($trimmed)) {
			return null;
		}

		// Handle new Joomla 4+ format: path#joomlaImage://...
		if (strpos($trimmed, '#') !== false) {
			$trimmed = explode('#', $trimmed)[0];
		}

		// First, try to decode as JSON array (subform format: [{"image":"path"}, ...])
		$images = json_decode($imagesJson, true);

		if (is_array($images) && json_last_error() === JSON_ERROR_NONE) {
			if (isset($images[0]) && is_array($images[0])) {
				foreach ($images as $imageData) {
					if (isset($imageData['image']) && !empty($imageData['image'])) {
						$imagePath = $imageData['image'];
						// Handle new Joomla 4+ format: path#joomlaImage://...
						if (strpos($imagePath, '#') !== false) {
							$imagePath = explode('#', $imagePath)[0];
						}
						return $imagePath;
					}
				}
			} elseif (isset($images['image']) && !empty($images['image'])) {
				$imagePath = $images['image'];
				// Handle new Joomla 4+ format: path#joomlaImage://...
				if (strpos($imagePath, '#') !== false) {
					$imagePath = explode('#', $imagePath)[0];
				}
				return $imagePath;
			}
		}

		// Try Registry format
		try {
			$registry = new \Joomla\Registry\Registry($imagesJson);
			$images = $registry->toArray();

			if (is_array($images) && !empty($images)) {
				// Look for image fields in Registry format
				foreach ($images as $key => $value) {
					if (is_string($key) && stripos($key, 'image') !== false && !empty($value)) {
						// Check if it's an array with 'image' key
						if (is_array($value) && isset($value['image'])) {
							$imagePath = $value['image'];
							// Handle new Joomla 4+ format: path#joomlaImage://...
							if (strpos($imagePath, '#') !== false) {
								$imagePath = explode('#', $imagePath)[0];
							}
							return $imagePath;
						} elseif (is_string($value)) {
							$imagePath = $value;
							// Handle new Joomla 4+ format: path#joomlaImage://...
							if (strpos($imagePath, '#') !== false) {
								$imagePath = explode('#', $imagePath)[0];
							}
							return $imagePath;
						}
					}
				}
			}
		} catch (\Exception $e) {
			// Registry parsing failed, continue
		}

		// If it's a direct path, return it
		if (file_exists(JPATH_ROOT . '/' . $trimmed)) {
			return $trimmed;
		}

		return null;
	}

	/**
	 * Extract all images from images JSON field.
	 *
	 * @param   string|array  $imagesJson  JSON string or array with images data
	 *
	 * @return  array  Array of image paths
	 */
	public static function getAllImages($imagesJson)
	{
		if (empty($imagesJson)) {
			return [];
		}

		$images = [];

		// If it's already an array
		if (is_array($imagesJson)) {
			if (isset($imagesJson[0]) && is_array($imagesJson[0])) {
				// Subform array format: [{"image":"path"}, ...]
				foreach ($imagesJson as $imageData) {
					if (isset($imageData['image']) && !empty($imageData['image'])) {
						$imagePath = $imageData['image'];
						// Handle new Joomla 4+ format: path#joomlaImage://...
						if (strpos($imagePath, '#') !== false) {
							$imagePath = explode('#', $imagePath)[0];
						}
						$images[] = $imagePath;
					}
				}
			} elseif (isset($imagesJson['image']) && !empty($imagesJson['image'])) {
				$imagePath = $imagesJson['image'];
				if (strpos($imagePath, '#') !== false) {
					$imagePath = explode('#', $imagePath)[0];
				}
				$images[] = $imagePath;
			}
			return $images;
		}

		// If it's not a string, return empty array
		if (!is_string($imagesJson)) {
			return [];
		}

		$trimmed = trim($imagesJson);
		if (empty($trimmed)) {
			return [];
		}

		// Handle new Joomla 4+ format: path#joomlaImage://...
		if (strpos($trimmed, '#') !== false) {
			$trimmed = explode('#', $trimmed)[0];
		}

		// First, try to decode as JSON array (subform format: [{"image":"path"}, ...])
		$decoded = json_decode($imagesJson, true);

		if (is_array($decoded) && json_last_error() === JSON_ERROR_NONE) {
			if (isset($decoded[0]) && is_array($decoded[0])) {
				foreach ($decoded as $imageData) {
					if (isset($imageData['image']) && !empty($imageData['image'])) {
						$imagePath = $imageData['image'];
						if (strpos($imagePath, '#') !== false) {
							$imagePath = explode('#', $imagePath)[0];
						}
						$images[] = $imagePath;
					}
				}
			} elseif (isset($decoded['image']) && !empty($decoded['image'])) {
				$imagePath = $decoded['image'];
				if (strpos($imagePath, '#') !== false) {
					$imagePath = explode('#', $imagePath)[0];
				}
				$images[] = $imagePath;
			}
		}

		// Try Registry format
		if (empty($images)) {
			try {
				$registry = new \Joomla\Registry\Registry($imagesJson);
				$regData = $registry->toArray();

				if (is_array($regData) && !empty($regData)) {
					foreach ($regData as $key => $value) {
						if (is_string($key) && stripos($key, 'image') !== false && !empty($value)) {
							if (is_array($value) && isset($value['image'])) {
								$imagePath = $value['image'];
								if (strpos($imagePath, '#') !== false) {
									$imagePath = explode('#', $imagePath)[0];
								}
								$images[] = $imagePath;
							} elseif (is_string($value)) {
								$imagePath = $value;
								if (strpos($imagePath, '#') !== false) {
									$imagePath = explode('#', $imagePath)[0];
								}
								$images[] = $imagePath;
							}
						}
					}
				}
			} catch (\Exception $e) {
				// Registry parsing failed, continue
			}
		}

		// If no images found and it's a direct path, try it
		if (empty($images) && file_exists(JPATH_ROOT . '/' . $trimmed)) {
			$images[] = $trimmed;
		}

		return array_unique($images);
	}

	/**
	 * Get all categories for a content item
	 *
	 * @param   int  $contentId  Content item ID
	 *
	 * @return  array  Array of category objects
	 */
	public static function getCategories($contentId)
	{
		$db = \Joomla\CMS\Factory::getDbo();
		$query = $db->createQuery();

		// Use #__one_categories instead of #__categories
		$query->select($db->quoteName(['c.id', 'c.title', 'c.alias']))
			->from($db->quoteName('#__one_categories', 'c'))
			->innerJoin($db->quoteName('#__one_content_categories', 'cc'), $db->quoteName('cc.category_id') . ' = ' . $db->quoteName('c.id'))
			->where($db->quoteName('cc.content_id') . ' = :content_id')
			->where($db->quoteName('c.published') . ' = 1')
			->order($db->quoteName('c.lft') . ' ASC')
			->bind(':content_id', $contentId, \Joomla\Database\ParameterType::INTEGER);

		$db->setQuery($query);
		return $db->loadObjectList() ?: [];
	}

	/**
	 * Get all categories for an event
	 *
	 * @param   int  $eventId  Event ID
	 *
	 * @return  array  Array of category objects
	 */
	public static function getEventCategories($eventId)
	{
		$db = \Joomla\CMS\Factory::getDbo();
		$query = $db->createQuery();

		$query->select($db->quoteName(['c.id', 'c.title', 'c.alias']))
			->from($db->quoteName('#__one_categories', 'c'))
			->innerJoin($db->quoteName('#__one_event_categories', 'ec'), $db->quoteName('ec.category_id') . ' = ' . $db->quoteName('c.id'))
			->where($db->quoteName('ec.event_id') . ' = :event_id')
			->where($db->quoteName('c.published') . ' = 1')
			->order($db->quoteName('c.lft') . ' ASC')
			->bind(':event_id', $eventId, \Joomla\Database\ParameterType::INTEGER);

		$db->setQuery($query);
		return $db->loadObjectList() ?: [];
	}

}
