<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Site\Controller;

use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Controller\BaseController;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Session\Session;
use Joomla\CMS\Response\JsonResponse;
use Joomla\CMS\Language\Text;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Event controller class.
 *
 * @since  1.0.13
 */
class EventController extends BaseController
{
	/**
	 * Handle event attend/unattend action
	 *
	 * @return  void
	 *
	 * @since   1.0.13
	 */
	public function attend()
	{
		$app = Factory::getApplication();
		$input = $app->getInput();
		$isAjax = $input->server->get('HTTP_X_REQUESTED_WITH') === 'XMLHttpRequest';
		
		// Check for request forgeries
		if (!Session::checkToken('get')) {
			if ($isAjax) {
				$app->setHeader('Content-Type', 'application/json', true);
				echo new JsonResponse(['success' => false, 'message' => Text::_('JINVALID_TOKEN')], null, false, true);
				$app->close();
				return;
			}
			jexit(Text::_('JINVALID_TOKEN'));
		}

		$eventId = $input->getInt('id', 0);
		$user = Factory::getUser();
		$db = Factory::getDbo();
		
		// Log for debugging
		$app->getLogger()->info('Event attend called - Event ID: ' . $eventId . ' | User ID: ' . $user->id . ' | Guest: ' . ($user->guest ? 'yes' : 'no'));

		// Validate event ID
		if ($eventId <= 0) {
			if ($isAjax) {
				$app->setHeader('Content-Type', 'application/json', true);
				echo new JsonResponse(['success' => false, 'message' => Text::_('COM_ONECORE_EVENT_ATTEND_ERROR_INVALID')]);
				$app->close();
				return;
			}
			$app->enqueueMessage(Text::_('COM_ONECORE_EVENT_ATTEND_ERROR_INVALID'), 'error');
			$app->redirect(Route::_('index.php?option=com_onecore&view=events', false));
			return;
		}

		// Check if attendees_count column exists
		$columns = $db->getTableColumns('#__one_events', false);
		$hasAttendeesCount = isset($columns['attendees_count']);

		// Check if event exists and is published
		$selectFields = ['id', 'title', 'published'];
		if ($hasAttendeesCount) {
			$selectFields[] = 'attendees_count';
		}

		$query = $db->getQuery(true)
			->select($db->quoteName($selectFields))
			->from($db->quoteName('#__one_events'))
			->where($db->quoteName('id') . ' = :id')
			->bind(':id', $eventId, \Joomla\Database\ParameterType::INTEGER);

		$db->setQuery($query);
		$event = $db->loadObject();

		if (!$event || !$event->published) {
			if ($isAjax) {
				$app->setHeader('Content-Type', 'application/json', true);
				echo new JsonResponse(['success' => false, 'message' => Text::_('COM_ONECORE_EVENT_ATTEND_ERROR_NOT_FOUND')]);
				$app->close();
				return;
			}
			$app->enqueueMessage(Text::_('COM_ONECORE_EVENT_ATTEND_ERROR_NOT_FOUND'), 'error');
			$app->redirect(Route::_('index.php?option=com_onecore&view=events', false));
			return;
		}

		$isAttending = false;
		$newCount = $hasAttendeesCount ? (int) ($event->attendees_count ?? 0) : 0;

		if (!$user->guest) {
			// For logged-in users: use database table
			// Check if table exists, create if not
			$tables = $db->getTableList();
			$prefix = $db->getPrefix();
			$tableName = $prefix . 'one_event_attendees';

			if (!in_array($tableName, $tables, true)) {
				// Create table
				try {
					$db->setQuery(
						"CREATE TABLE IF NOT EXISTS `{$tableName}` (
							`id` int NOT NULL AUTO_INCREMENT,
							`event_id` int NOT NULL,
							`user_id` int unsigned NOT NULL,
							`created` datetime NOT NULL,
							PRIMARY KEY (`id`),
							UNIQUE KEY `idx_event_user` (`event_id`, `user_id`),
							KEY `idx_event_id` (`event_id`),
							KEY `idx_user_id` (`user_id`),
							CONSTRAINT `fk_one_event_attendees_event` FOREIGN KEY (`event_id`) REFERENCES `{$prefix}one_events` (`id`) ON DELETE CASCADE
						) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 DEFAULT COLLATE=utf8mb4_unicode_ci"
					)->execute();
				} catch (\Exception $e) {
					$app->getLogger()->error('Error creating attendees table: ' . $e->getMessage());
				}
			}

			// Check if user is already attending
			$query = $db->getQuery(true)
				->select('id')
				->from($db->quoteName('#__one_event_attendees'))
				->where($db->quoteName('event_id') . ' = :event_id')
				->where($db->quoteName('user_id') . ' = :user_id')
				->bind(':event_id', $eventId, \Joomla\Database\ParameterType::INTEGER)
				->bind(':user_id', $user->id, \Joomla\Database\ParameterType::INTEGER);

			$db->setQuery($query);
			$existing = $db->loadResult();

			if ($existing) {
				// Unattend: remove from database
				$query = $db->getQuery(true)
					->delete($db->quoteName('#__one_event_attendees'))
					->where($db->quoteName('id') . ' = :id')
					->bind(':id', $existing, \Joomla\Database\ParameterType::INTEGER);
				$db->setQuery($query)->execute();
				$isAttending = false;
				$newCount = max(0, $newCount - 1);
			} else {
				// Attend: add to database
				$obj = new \stdClass();
				$obj->event_id = $eventId;
				$obj->user_id = $user->id;
				$obj->created = Factory::getDate()->toSql();
				
				try {
					// Use insert query instead of insertObject for better error handling
					$query = $db->getQuery(true)
						->insert($db->quoteName('#__one_event_attendees'))
						->columns([$db->quoteName('event_id'), $db->quoteName('user_id'), $db->quoteName('created')])
						->values(':event_id, :user_id, :created')
						->bind(':event_id', $eventId, \Joomla\Database\ParameterType::INTEGER)
						->bind(':user_id', $user->id, \Joomla\Database\ParameterType::INTEGER)
						->bind(':created', $obj->created, \Joomla\Database\ParameterType::STRING);
					
					$db->setQuery($query);
					$db->execute();
					
					$isAttending = true;
					$newCount = $newCount + 1;
					
					$app->getLogger()->info('Attendee added successfully - Event ID: ' . $eventId . ' | User ID: ' . $user->id);
				} catch (\Exception $e) {
					// Log error but continue
					$app->getLogger()->error('Error inserting attendee: ' . $e->getMessage() . ' | Event ID: ' . $eventId . ' | User ID: ' . $user->id . ' | SQL: ' . $db->getQuery());
					if ($isAjax) {
						$app->setHeader('Content-Type', 'application/json', true);
						echo new JsonResponse(['success' => false, 'message' => 'Database error: ' . $e->getMessage()], null, false, true);
						$app->close();
						return;
					}
					throw $e;
				}
			}
		} else {
			// For guests: use session
			$session = $app->getSession();
			$attendedEvents = $session->get('onecore_attended_events', []);

			if (in_array($eventId, $attendedEvents, true)) {
				// Unattend: remove from session
				$attendedEvents = array_diff($attendedEvents, [$eventId]);
				$session->set('onecore_attended_events', $attendedEvents);
				$isAttending = false;
				$newCount = max(0, $newCount - 1);
			} else {
				// Attend: add to session
				$attendedEvents[] = $eventId;
				$session->set('onecore_attended_events', $attendedEvents);
				$isAttending = true;
				$newCount = $newCount + 1;
			}
		}

		// Update attendees_count in events table (only if column exists)
		if ($hasAttendeesCount) {
			$query = $db->getQuery(true)
				->update($db->quoteName('#__one_events'))
				->set($db->quoteName('attendees_count') . ' = :count')
				->where($db->quoteName('id') . ' = :id')
				->bind(':count', $newCount, \Joomla\Database\ParameterType::INTEGER)
				->bind(':id', $eventId, \Joomla\Database\ParameterType::INTEGER);

			$db->setQuery($query)->execute();
		}

		// Return JSON response for AJAX requests
		if ($isAjax) {
			// Clear any output buffers
			while (ob_get_level()) {
				ob_end_clean();
			}
			
			$app->clearHeaders();
			$app->setHeader('Content-Type', 'application/json; charset=utf-8', true);
			$app->setHeader('Cache-Control', 'no-cache, must-revalidate', true);
			
			// JsonResponse wraps data in 'data' property, so we pass our data as first parameter
			$responseData = [
				'success' => true,
				'attending' => $isAttending,
				'count' => $newCount,
				'message' => $isAttending ? Text::_('COM_ONECORE_EVENT_ATTEND_SUCCESS') : Text::_('COM_ONECORE_EVENT_UNATTEND_SUCCESS')
			];
			
			echo new JsonResponse($responseData, null, false, true);
			$app->close();
			return;
		}

		// Redirect for non-AJAX requests
		$app->enqueueMessage(
			$isAttending ? Text::_('COM_ONECORE_EVENT_ATTEND_SUCCESS') : Text::_('COM_ONECORE_EVENT_UNATTEND_SUCCESS'),
			'success'
		);
		$app->redirect(Route::_('index.php?option=com_onecore&view=event&id=' . $eventId, false));
	}
}
