<?php
/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

\defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Session\Session;
use Joomla\CMS\Uri\Uri;

/** @var array $displayData */
$item = $displayData['item'] ?? null;

if (!$item || empty($item->id)) {
	return;
}

$app = Factory::getApplication();
$user = Factory::getUser();
$eventId = (int) $item->id;
$attendeesCount = (int) ($item->attendees_count ?? 0);

// Check if current user is attending (stored in session/cookie for guests, or database for logged-in users)
$isAttending = false;
if (!$user->guest) {
	// For logged-in users, check database
	$db = Factory::getDbo();
	$query = $db->getQuery(true)
		->select('COUNT(*)')
		->from($db->quoteName('#__one_event_attendees'))
		->where($db->quoteName('event_id') . ' = ' . $eventId)
		->where($db->quoteName('user_id') . ' = ' . (int) $user->id);
	$db->setQuery($query);
	$isAttending = (bool) $db->loadResult();
} else {
	// For guests, check session
	$session = $app->getSession();
	$attendedEvents = $session->get('onecore_attended_events', []);
	$isAttending = in_array($eventId, $attendedEvents, true);
}

$token = Session::getFormToken();
$attendUrl = Route::_('index.php?option=com_onecore&task=event.attend&id=' . $eventId . '&' . $token . '=1', false);
?>

<div class="onecore-event-attend mb-4">
	<div class="card">
		<div class="card-body">
			<h4 class="card-title mb-3"><?php echo Text::_('COM_ONECORE_EVENT_ATTEND_TITLE'); ?></h4>
			
			<div class="d-flex align-items-center gap-3 mb-3">
				<button type="button" 
						class="btn <?php echo $isAttending ? 'btn-success' : 'btn-outline-primary'; ?> onecore-attend-btn" 
						data-event-id="<?php echo $eventId; ?>"
						data-attending="<?php echo $isAttending ? '1' : '0'; ?>"
						data-url="<?php echo htmlspecialchars($attendUrl, ENT_QUOTES, 'UTF-8'); ?>">
					<span class="icon-<?php echo $isAttending ? 'check' : 'plus'; ?>" aria-hidden="true"></span>
					<span class="ms-1"><?php echo $isAttending ? Text::_('COM_ONECORE_EVENT_ATTENDING') : Text::_('COM_ONECORE_EVENT_ATTEND'); ?></span>
				</button>
				
				<div class="onecore-attendees-count">
					<strong><?php echo Text::_('COM_ONECORE_EVENT_ATTENDEES_COUNT'); ?>:</strong>
					<span class="badge bg-primary ms-2" id="onecore-attendees-count-<?php echo $eventId; ?>">
						<?php echo $attendeesCount; ?>
					</span>
				</div>
			</div>
			
			<?php if ($isAttending) : ?>
				<div class="alert alert-success alert-sm mb-0">
					<span class="icon-check" aria-hidden="true"></span>
					<?php echo Text::_('COM_ONECORE_EVENT_ATTENDING_MESSAGE'); ?>
				</div>
			<?php endif; ?>
		</div>
	</div>
</div>

<script>
(function() {
	'use strict';
	
	document.addEventListener('DOMContentLoaded', function() {
		var attendButtons = document.querySelectorAll('.onecore-attend-btn');
		
		attendButtons.forEach(function(btn) {
			btn.addEventListener('click', function(e) {
				e.preventDefault();
				
				var eventId = this.getAttribute('data-event-id');
				var isAttending = this.getAttribute('data-attending') === '1';
				var url = this.getAttribute('data-url');
				var countElement = document.getElementById('onecore-attendees-count-' + eventId);
				
				// Get cardBody reference early - try multiple selectors
				var cardBody = null;
				try {
					cardBody = this.closest('.card-body');
					if (!cardBody) {
						cardBody = this.closest('.onecore-event-attend');
					}
					if (!cardBody) {
						cardBody = this.parentElement;
					}
					// Final check - ensure cardBody is a valid DOM element
					if (cardBody && (!cardBody.querySelector || !cardBody.appendChild)) {
						cardBody = null;
					}
				} catch (err) {
					console.warn('Could not find cardBody:', err);
					cardBody = null;
				}
				
				// Store reference to button for use in callbacks
				var buttonElement = this;
				
				// Disable button during request
				this.disabled = true;
				var originalText = this.innerHTML;
				this.innerHTML = '<span class="spinner-border spinner-border-sm me-1" role="status" aria-hidden="true"></span><?php echo Text::_('JLOADING'); ?>';
				
				// Make AJAX request
				console.log('Sending request to:', url);
				fetch(url, {
					method: 'GET',
					headers: {
						'X-Requested-With': 'XMLHttpRequest',
						'Accept': 'application/json'
					},
					credentials: 'same-origin'
				})
				.then(function(response) {
					console.log('Response status:', response.status);
					console.log('Response headers:', response.headers);
					if (!response.ok) {
						// Try to get error message from response
						return response.text().then(function(text) {
							console.error('Error response:', text);
							throw new Error('Network response was not ok: ' + response.status + ' - ' + text);
						});
					}
					return response.json();
				})
				.then(function(responseData) {
					console.log('Response data:', responseData);
					
					// Joomla JsonResponse wraps data in data property
					// Structure: {success: true, message: null, messages: null, data: {...}}
					var data = null;
					if (responseData && responseData.data) {
						// Data is wrapped in data property
						data = responseData.data;
						console.log('Using data from responseData.data:', data);
					} else if (responseData && responseData.success !== undefined) {
						// Data is at root level
						data = responseData;
						console.log('Using data from root level:', data);
					} else {
						console.error('Unexpected response structure:', responseData);
						data = responseData;
					}
					
					// Helper function to safely get cardBody
					function getCardBody() {
						if (cardBody && typeof cardBody.querySelector === 'function' && typeof cardBody.appendChild === 'function') {
							return cardBody;
						}
						// Try to find it again from buttonElement
						try {
							var found = buttonElement.closest('.card-body');
							if (found && typeof found.querySelector === 'function' && typeof found.appendChild === 'function') {
								return found;
							}
							found = buttonElement.closest('.onecore-event-attend');
							if (found && typeof found.querySelector === 'function' && typeof found.appendChild === 'function') {
								return found;
							}
						} catch (err) {
							console.warn('Could not find cardBody:', err);
						}
						return null;
					}
					
					if (data && data.success) {
						// Update button state
						if (data.attending) {
							buttonElement.classList.remove('btn-outline-primary');
							buttonElement.classList.add('btn-success');
							buttonElement.setAttribute('data-attending', '1');
							buttonElement.innerHTML = '<span class="icon-check" aria-hidden="true"></span><span class="ms-1"><?php echo Text::_('COM_ONECORE_EVENT_ATTENDING'); ?></span>';
							
							// Show success message - safely get cardBody
							try {
								var safeCardBody = getCardBody();
								if (safeCardBody) {
									var alert = document.createElement('div');
									alert.className = 'alert alert-success alert-sm mb-0 mt-2';
									alert.innerHTML = '<span class="icon-check" aria-hidden="true"></span> <?php echo Text::_('COM_ONECORE_EVENT_ATTENDING_MESSAGE'); ?>';
									var existingAlert = safeCardBody.querySelector('.alert');
									if (existingAlert) {
										existingAlert.remove();
									}
									safeCardBody.appendChild(alert);
								}
							} catch (err) {
								console.warn('Could not update alert message:', err);
							}
						} else {
							buttonElement.classList.remove('btn-success');
							buttonElement.classList.add('btn-outline-primary');
							buttonElement.setAttribute('data-attending', '0');
							buttonElement.innerHTML = '<span class="icon-plus" aria-hidden="true"></span><span class="ms-1"><?php echo Text::_('COM_ONECORE_EVENT_ATTEND'); ?></span>';
							
							// Remove success message - safely get cardBody
							try {
								var safeCardBody = getCardBody();
								if (safeCardBody) {
									var existingAlert = safeCardBody.querySelector('.alert');
									if (existingAlert) {
										existingAlert.remove();
									}
								}
							} catch (err) {
								console.warn('Could not remove alert message:', err);
							}
						}
						
						// Update count
						if (countElement && data.count !== undefined) {
							countElement.textContent = data.count;
						}
					} else {
						alert(data.message || '<?php echo Text::_('COM_ONECORE_EVENT_ATTEND_ERROR'); ?>');
						buttonElement.innerHTML = originalText;
					}
				})
				.catch(function(error) {
					console.error('Error:', error);
					console.error('URL:', url);
					console.error('Response status:', error.status || 'unknown');
					alert('<?php echo Text::_('COM_ONECORE_EVENT_ATTEND_ERROR'); ?>: ' + (error.message || error));
					buttonElement.innerHTML = originalText;
				})
				.finally(function() {
					buttonElement.disabled = false;
				});
			});
		});
	});
})();
</script>
