/**
 * @package     Comdev.Module
 * @subpackage  mod_onecore_search
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

(function() {
	'use strict';

	document.addEventListener('DOMContentLoaded', function() {
		// Initialize autocomplete if enabled
		const searchInputs = document.querySelectorAll('.mod-onecore-search__input');
		
		searchInputs.forEach(function(input) {
			if (input.type === 'text' && input.hasAttribute('autocomplete')) {
				initAutocomplete(input);
			}
		});

		// Handle form reset
		const resetButtons = document.querySelectorAll('.mod-onecore-search__reset');
		resetButtons.forEach(function(button) {
			button.addEventListener('click', function(e) {
				e.preventDefault();
				const form = this.closest('.mod-onecore-search__form');
				if (form) {
					// Reset form fields
					form.reset();
					
					// Clear suggestions
					const suggestions = form.querySelector('.mod-onecore-search__suggestions');
					if (suggestions) {
						suggestions.innerHTML = '';
						suggestions.classList.remove('is-visible');
					}
					
					// Build reset URL (base URL without search parameters)
					const formAction = form.getAttribute('action');
					if (formAction) {
						// Get base URL from form action
						const url = new URL(formAction, window.location.origin);
						
						// Remove all search parameters except Itemid if present
						const itemId = url.searchParams.get('Itemid');
						url.search = '';
						
						if (itemId) {
							url.searchParams.set('Itemid', itemId);
						}
						
						// Redirect to clean URL
						window.location.href = url.toString();
					} else {
						// Fallback: redirect to current page without query parameters
						const currentUrl = new URL(window.location.href);
						const itemId = currentUrl.searchParams.get('Itemid');
						currentUrl.search = '';
						
						if (itemId) {
							currentUrl.searchParams.set('Itemid', itemId);
						}
						
						window.location.href = currentUrl.toString();
					}
				}
			});
		});
	});

	/**
	 * Initialize autocomplete for search input
	 */
	function initAutocomplete(input) {
		const moduleId = input.id.replace('mod-onecore-search-input-', '');
		const suggestionsContainer = document.getElementById('mod-onecore-search-suggestions-' + moduleId);
		
		if (!suggestionsContainer) {
			return;
		}

		let debounceTimer;
		let currentRequest = null;

		input.addEventListener('input', function() {
			const query = this.value.trim();
			
			clearTimeout(debounceTimer);
			
			if (query.length < 2) {
				suggestionsContainer.innerHTML = '';
				suggestionsContainer.classList.remove('is-visible');
				return;
			}
			
			debounceTimer = setTimeout(function() {
				if (currentRequest) {
					currentRequest.abort();
				}
				
				suggestionsContainer.classList.add('is-loading');
				
				const url = new URL(Joomla.getOptions('system.paths').baseFull + 'index.php', window.location.origin);
				url.searchParams.set('option', 'com_onecore');
				url.searchParams.set('task', 'suggestions.suggest');
				url.searchParams.set('format', 'json');
				url.searchParams.set('q', query);
				
				currentRequest = fetch(url.toString(), {
					method: 'GET',
					headers: {
						'X-Requested-With': 'XMLHttpRequest',
						'Accept': 'application/json'
					}
				})
				.then(function(response) {
					if (!response.ok) {
						throw new Error('Network response was not ok');
					}
					return response.json();
				})
				.then(function(data) {
					suggestionsContainer.classList.remove('is-loading');
					
					if (data && data.length > 0) {
						let html = '';
						data.forEach(function(item) {
							html += '<div class="mod-onecore-search__suggestion" role="option" tabindex="0" data-title="' + 
								escapeHtml(item.title) + '">' + 
								escapeHtml(item.title) + 
								'</div>';
						});
						suggestionsContainer.innerHTML = html;
						suggestionsContainer.classList.add('is-visible');
						
						// Add click handlers
						suggestionsContainer.querySelectorAll('.mod-onecore-search__suggestion').forEach(function(suggestion) {
							suggestion.addEventListener('click', function() {
								input.value = this.getAttribute('data-title');
								suggestionsContainer.innerHTML = '';
								suggestionsContainer.classList.remove('is-visible');
								input.focus();
							});
							
							suggestion.addEventListener('keydown', function(e) {
								if (e.key === 'Enter' || e.key === ' ') {
									e.preventDefault();
									input.value = this.getAttribute('data-title');
									suggestionsContainer.innerHTML = '';
									suggestionsContainer.classList.remove('is-visible');
									input.focus();
								}
							});
						});
					} else {
						suggestionsContainer.innerHTML = '';
						suggestionsContainer.classList.remove('is-visible');
					}
				})
				.catch(function(error) {
					suggestionsContainer.classList.remove('is-loading');
					suggestionsContainer.classList.remove('is-visible');
					console.error('Autocomplete error:', error);
				});
			}, 300);
		});
		
		// Hide suggestions when clicking outside
		document.addEventListener('click', function(e) {
			if (!input.contains(e.target) && !suggestionsContainer.contains(e.target)) {
				suggestionsContainer.classList.remove('is-visible');
			}
		});
		
		// Handle keyboard navigation
		input.addEventListener('keydown', function(e) {
			if (e.key === 'Escape') {
				suggestionsContainer.classList.remove('is-visible');
			}
		});
	}

	/**
	 * Escape HTML to prevent XSS
	 */
	function escapeHtml(text) {
		const map = {
			'&': '&amp;',
			'<': '&lt;',
			'>': '&gt;',
			'"': '&quot;',
			"'": '&#039;'
		};
		return text.replace(/[&<>"']/g, function(m) { return map[m]; });
	}
})();
