<?php

/**
 * @package     Comdev.Module
 * @subpackage  mod_onecore_search
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Module\OnecoreSearch\Site\Helper;

use Comdev\Component\Onecore\Site\Helper\CustomFieldsHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Router\Route;
use Joomla\Database\DatabaseAwareInterface;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Registry\Registry;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Helper class for mod_onecore_search
 *
 * @since  1.0.0
 */
class SearchHelper implements DatabaseAwareInterface
{
	use DatabaseAwareTrait;

	/**
	 * Get list of categories for filter dropdown
	 *
	 * @return  array  Array of category objects
	 *
	 * @since   1.0.0
	 */
	public function getCategories(): array
	{
		$db = $this->getDatabase();
		$query = $db->getQuery(true);

		$query->select([
			$db->quoteName('c.id'),
			$db->quoteName('c.title'),
			$db->quoteName('c.alias'),
			$db->quoteName('c.parent_id'),
		])
			->from($db->quoteName('#__one_categories', 'c'))
			->where($db->quoteName('c.published') . ' = 1')
			->order($db->quoteName('c.title') . ' ASC');

		$db->setQuery($query);

		try {
			return $db->loadObjectList() ?: [];
		} catch (\Exception $e) {
			return [];
		}
	}

	/**
	 * Get list of authors for filter dropdown
	 *
	 * @return  array  Array of author objects
	 *
	 * @since   1.0.0
	 */
	public function getAuthors(): array
	{
		$db = $this->getDatabase();
		$query = $db->getQuery(true);

		$query->select([
			$db->quoteName('u.id'),
			$db->quoteName('u.name'),
		])
			->from($db->quoteName('#__users', 'u'))
			->innerJoin(
				$db->quoteName('#__one_content', 'c'),
				$db->quoteName('c.created_by') . ' = ' . $db->quoteName('u.id')
			)
			->where($db->quoteName('c.published') . ' = 1')
			->group($db->quoteName('u.id'))
			->order($db->quoteName('u.name') . ' ASC');

		$db->setQuery($query);

		try {
			return $db->loadObjectList() ?: [];
		} catch (\Exception $e) {
			return [];
		}
	}

	/**
	 * Get list of searchable custom fields for filter dropdowns
	 *
	 * @param   string  $entity  Entity type: 'content' or 'events'
	 *
	 * @return  array  Array of custom field objects with unique values
	 *
	 * @since   1.0.0
	 */
	public function getSearchableCustomFields(string $entity = 'content'): array
	{
		// Ensure custom fields schema exists
		CustomFieldsHelper::ensureReady();

		$db = $this->getDatabase();
		$query = $db->getQuery(true);

		// Get all published custom fields that are searchable
		$query->select([
			$db->quoteName('f.id'),
			$db->quoteName('f.title'),
			$db->quoteName('f.type'),
			$db->quoteName('f.options'),
		])
			->from($db->quoteName('#__one_customfields', 'f'))
			->innerJoin($db->quoteName('#__one_customfield_groups', 'g'), $db->quoteName('g.id') . ' = ' . $db->quoteName('f.group_id'))
			->where($db->quoteName('f.published') . ' = 1')
			->where($db->quoteName('f.searchable') . ' = 1')
			->where($db->quoteName('g.entity_type') . ' = ' . $db->quote($entity))
			->order($db->quoteName('f.ordering') . ' ASC, ' . $db->quoteName('f.id') . ' ASC');

		$db->setQuery($query);

		try {
			$fields = $db->loadObjectList() ?: [];
		} catch (\Exception $e) {
			return [];
		}

		// For each field, get unique values if it's a select/multiselect type
		foreach ($fields as $field) {
			$field->values = [];

			if (($field->type === 'select' || $field->type === 'multiselect') && !empty($field->options)) {
				// Parse options (format: value1|label1\nvalue2|label2 or just value1\nvalue2)
				// Options are stored with newlines, not commas
				$options = explode("\n", $field->options);
				foreach ($options as $option) {
					$option = trim($option);
					if (empty($option)) {
						continue;
					}
					if (strpos($option, '|') !== false) {
						list($value, $label) = explode('|', $option, 2);
						$field->values[] = (object) [
							'value' => trim($value),
							'label' => trim($label),
						];
					} else {
						$field->values[] = (object) [
							'value' => $option,
							'label' => $option,
						];
					}
				}
			} elseif ($field->type === 'input' || $field->type === 'textarea') {
				// For text fields, get unique values from customfield_values
				$valueQuery = $db->getQuery(true);
				$valueQuery->select('DISTINCT ' . $db->quoteName('value'))
					->from($db->quoteName('#__one_customfield_values'));
				
				// Filter by entity type
				if ($entity === 'events') {
					$valueQuery->where($db->quoteName('event_id') . ' IS NOT NULL');
				} else {
					$valueQuery->where($db->quoteName('content_id') . ' IS NOT NULL');
				}
				
				$valueQuery->where($db->quoteName('field_id') . ' = ' . (int) $field->id)
					->where($db->quoteName('value') . ' != ' . $db->quote(''))
					->where($db->quoteName('value') . ' IS NOT NULL')
					->order($db->quoteName('value') . ' ASC')
					->setLimit(100); // Limit to prevent too many options

				$db->setQuery($valueQuery);

				try {
					$values = $db->loadColumn() ?: [];
					foreach ($values as $value) {
						$field->values[] = (object) [
							'value' => $value,
							'label' => $value,
						];
					}
				} catch (\Exception $e) {
					// Ignore errors
				}
			}
		}

		return $fields;
	}

	/**
	 * Build search URL with parameters
	 *
	 * @param   Registry  $params  Module parameters
	 * @param   array      $filters  Filter parameters
	 * @param   string     $view     Target view: 'content' or 'events'
	 *
	 * @return  string  URL string
	 *
	 * @since   1.0.0
	 */
	public function buildSearchUrl(Registry $params, array $filters = [], string $view = 'content'): string
	{
		$targetMenu = $params->get('target_menu', 0);
		
		// Use provided view or default to content
		if (empty($view) || !in_array($view, ['content', 'events'])) {
			$view = 'content';
		}
		
		if ($targetMenu > 0) {
			$menu = Factory::getApplication()->getMenu();
			$item = $menu->getItem($targetMenu);
			
			if ($item) {
				$route = Route::_('index.php?Itemid=' . $targetMenu, false);
			} else {
				$route = Route::_('index.php?option=com_onecore&view=' . $view, false);
			}
		} else {
			$route = Route::_('index.php?option=com_onecore&view=' . $view, false);
		}

		$uri = new \Joomla\CMS\Uri\Uri($route);

		// Always add view parameter to ensure it's preserved
		$uri->setVar('view', $view);

		// Add show_custom_fields parameter from module params
		$showCustomFields = $params->get('show_custom_fields', 1);
		$uri->setVar('show_custom_fields', (int) $showCustomFields);

		// Add filter parameters
		if (!empty($filters['search'])) {
			$uri->setVar('search', $filters['search']);
		}

		if (!empty($filters['category']) && is_array($filters['category'])) {
			$uri->setVar('category', $filters['category']);
		}

		if (!empty($filters['sort'])) {
			$uri->setVar('sort', $filters['sort']);
		}

		if (!empty($filters['date_from'])) {
			$uri->setVar('date_from', $filters['date_from']);
		}

		if (!empty($filters['date_to'])) {
			$uri->setVar('date_to', $filters['date_to']);
		}

		if (!empty($filters['featured'])) {
			$uri->setVar('featured', 1);
		}

		// Add custom field filters
		if (!empty($filters['custom_fields']) && is_array($filters['custom_fields'])) {
			foreach ($filters['custom_fields'] as $fieldId => $value) {
				if (!empty($value)) {
					$fieldKey = 'cf_' . (int) $fieldId;
					if (is_array($value)) {
						// For arrays, join with comma or set as array parameter
						$uri->setVar($fieldKey, $value);
					} else {
						$uri->setVar($fieldKey, $value);
					}
				}
			}
		}

		return (string) $uri;
	}

	/**
	 * Get search suggestions for autocomplete
	 *
	 * @param   string  $query  Search query
	 * @param   int     $limit  Maximum number of suggestions
	 *
	 * @return  array  Array of suggestion objects
	 *
	 * @since   1.0.0
	 */
	public function getSuggestions(string $query, int $limit = 10): array
	{
		if (strlen($query) < 2) {
			return [];
		}

		// Ensure custom fields schema exists (site + module autocomplete)
		CustomFieldsHelper::ensureReady();

		$db = $this->getDatabase();
		$searchTerm = '%' . $db->escape($query, true) . '%';

		$query = $db->getQuery(true);
		$query->select([
			$db->quoteName('a.id'),
			$db->quoteName('a.title'),
			$db->quoteName('a.alias'),
		])
			->from($db->quoteName('#__one_content', 'a'))
			->where($db->quoteName('a.published') . ' = 1')
			->where('(' . $db->quoteName('a.title') . ' LIKE :search1 OR ' . $db->quoteName('a.introtext') . ' LIKE :search2
				OR EXISTS (
					SELECT 1
					FROM ' . $db->quoteName('#__one_customfield_values', 'cfv') . '
					INNER JOIN ' . $db->quoteName('#__one_customfields', 'cff') . ' ON ' . $db->quoteName('cff.id') . ' = ' . $db->quoteName('cfv.field_id') . '
					INNER JOIN ' . $db->quoteName('#__one_categories', 'fcat') . ' ON ' . $db->quoteName('fcat.id') . ' = ' . $db->quoteName('cff.category_id') . '
					INNER JOIN ' . $db->quoteName('#__one_content_categories', 'ccc') . ' ON ' . $db->quoteName('ccc.content_id') . ' = ' . $db->quoteName('a.id') . '
					INNER JOIN ' . $db->quoteName('#__one_categories', 'ccat') . ' ON ' . $db->quoteName('ccat.id') . ' = ' . $db->quoteName('ccc.category_id') . '
					WHERE ' . $db->quoteName('cfv.content_id') . ' = ' . $db->quoteName('a.id') . '
					  AND ' . $db->quoteName('cff.published') . ' = 1
					  AND ' . $db->quoteName('cff.searchable') . ' = 1
					  AND ' . $db->quoteName('fcat.lft') . ' <= ' . $db->quoteName('ccat.lft') . '
					  AND ' . $db->quoteName('fcat.rgt') . ' >= ' . $db->quoteName('ccat.rgt') . '
					  AND ' . $db->quoteName('cfv.value') . ' LIKE :search3
				)
			)')
			->bind(':search1', $searchTerm)
			->bind(':search2', $searchTerm)
			->bind(':search3', $searchTerm)
			->order($db->quoteName('a.title') . ' ASC')
			->setLimit($limit);

		$db->setQuery($query);

		try {
			return $db->loadObjectList() ?: [];
		} catch (\Exception $e) {
			return [];
		}
	}

	/**
	 * Validate search parameters
	 *
	 * @param   array  $params  Parameters to validate
	 *
	 * @return  array  Validated parameters
	 *
	 * @since   1.0.0
	 */
	public function validateSearchParams(array $params): array
	{
		$validated = [];

		if (isset($params['search'])) {
			$validated['search'] = trim($params['search']);
		}

		if (isset($params['category']) && is_array($params['category'])) {
			$validated['category'] = array_filter(array_map('intval', $params['category']));
		}

		if (isset($params['sort']) && in_array($params['sort'], ['newest', 'oldest', 'a-z', 'z-a'])) {
			$validated['sort'] = $params['sort'];
		}

		if (isset($params['date_from'])) {
			$validated['date_from'] = $params['date_from'];
		}

		if (isset($params['date_to'])) {
			$validated['date_to'] = $params['date_to'];
		}

		if (isset($params['featured'])) {
			$validated['featured'] = (int) $params['featured'];
		}

		// Validate custom field filters
		$validated['custom_fields'] = [];
		foreach ($params as $key => $value) {
			if (strpos($key, 'cf_') === 0) {
				$fieldId = (int) substr($key, 3);
				if ($fieldId > 0 && !empty($value)) {
					$validated['custom_fields'][$fieldId] = is_array($value) 
						? array_map('trim', $value) 
						: trim($value);
				}
			}
		}

		return $validated;
	}
}
