<?php

/**
 * @package     Comdev.Module
 * @subpackage  mod_onecore_search
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Module\OnecoreSearch\Site\Dispatcher;

use Joomla\CMS\Dispatcher\AbstractModuleDispatcher;
use Joomla\CMS\Factory;
use Joomla\CMS\Helper\HelperFactoryAwareInterface;
use Joomla\CMS\Helper\HelperFactoryAwareTrait;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Dispatcher class for mod_onecore_search
 *
 * @since  1.0.0
 */
class Dispatcher extends AbstractModuleDispatcher implements HelperFactoryAwareInterface
{
	use HelperFactoryAwareTrait;

	/**
	 * Returns the layout data.
	 *
	 * @return  array
	 *
	 * @since   1.0.0
	 */
	protected function getLayoutData()
	{
		$data = parent::getLayoutData();

		// Load component language file for translations (COM_ONECORE_*)
		Factory::getLanguage()->load('com_onecore', JPATH_SITE . '/components/com_onecore');

		// Initialize defaults
		$data['categories'] = [];
		$data['searchableCustomFields'] = [];
		$data['currentCustomFields'] = [];
		
		// Detect current view from input or menu item
		$input = $data['input'];
		$currentView = $input->getString('view', '');
		$app = Factory::getApplication();
		$menu = $app->getMenu();
		$active = $menu->getActive();
		
		// If no view in URL, check menu item
		if (empty($currentView) && $active) {
			$currentView = $active->query['view'] ?? '';
		}
		
		// Default to content if not events
		$targetView = ($currentView === 'events') ? 'events' : 'content';
		$data['formAction'] = '/index.php?option=com_onecore&view=' . $targetView;
		$data['currentSearch'] = '';
		$data['currentCategories'] = [];
		$data['currentSort'] = 'newest';
		$data['currentDateFrom'] = '';
		$data['currentDateTo'] = '';
		$data['currentFeatured'] = 0;

		try {
			// Get helper
			$helperFactory = $this->getHelperFactory();
			if ($helperFactory) {
				$helper = $helperFactory->getHelper('SearchHelper');
				if ($helper) {
					// Get current filter values from request
					$input = $data['input'];
					// Get entity from custom_fields_entity_type parameter (for custom fields filtering)
					$customFieldsEntityType = $data['params']->get('custom_fields_entity_type', 'content');
					$data['customFieldsEntityType'] = $customFieldsEntityType;
					$data['currentSearch'] = $input->getString('search', '');
					$data['currentCategories'] = $input->get('category', [], 'array');
					$data['currentSort'] = $input->getString('sort', 'newest');
					$data['currentDateFrom'] = $input->getString('date_from', '');
					$data['currentDateTo'] = $input->getString('date_to', '');
					$data['currentFeatured'] = $input->getInt('featured', 0);

					// Get custom field filter values
					$data['currentCustomFields'] = [];
					$allInput = $input->getArray();
					foreach ($allInput as $key => $value) {
						if (strpos($key, 'cf_') === 0) {
							$fieldId = (int) substr($key, 3);
							if ($fieldId > 0) {
								$data['currentCustomFields'][$fieldId] = is_array($value) ? $value : $value;
							}
						}
					}

					// Get data for dropdowns
					$data['categories'] = $helper->getCategories();
					
					// Determine entity type based on current view
					$entityType = ($targetView === 'events') ? 'events' : 'content';
					
					// Only load custom fields if show_custom_fields is enabled
					if ($data['params']->get('show_custom_fields', 1)) {
						$data['searchableCustomFields'] = $helper->getSearchableCustomFields($entityType);
					} else {
						$data['searchableCustomFields'] = [];
					}

					// Build form action URL
					$filters = [
						'search' => $data['currentSearch'],
						'category' => $data['currentCategories'],
						'sort' => $data['currentSort'],
						'date_from' => $data['currentDateFrom'],
						'date_to' => $data['currentDateTo'],
						'featured' => $data['currentFeatured'],
						'custom_fields' => $data['currentCustomFields'],
					];
					$data['formAction'] = $helper->buildSearchUrl($data['params'], $filters, $targetView);
					$data['helper'] = $helper;
				}
			}
		} catch (\Exception $e) {
			// Silently fail - module will render with defaults
		}

		$data['moduleClassSfx'] = $data['params']->get('moduleclass_sfx', '');
		$data['showSearch'] = $data['params']->get('show_search', 1);
		$data['showAutosuggest'] = $data['params']->get('show_autosuggest', 1);
		$data['showCategory'] = $data['params']->get('show_category', 1);
		$data['showSort'] = $data['params']->get('show_sort', 1);
		$data['showDate'] = $data['params']->get('show_date', 0);
		$data['showFeatured'] = $data['params']->get('show_featured', 0);
		$data['showAdvanced'] = $data['params']->get('show_advanced', 0);
		$data['showCustomFields'] = $data['params']->get('show_custom_fields', 1);
		
		// Set entity type based on detected view (use $targetView from above)
		$data['customFieldsEntityType'] = ($targetView === 'events') ? 'events' : 'content';

		return $data;
	}
}
