<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\View\Dashboard;

use Comdev\Component\Onecore\Administrator\Helper\OneCoreHelper;
use Joomla\CMS\MVC\View\HtmlView as BaseHtmlView;
use Joomla\CMS\Toolbar\ToolbarHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Dashboard view class.
 *
 * @since  1.0.0
 */
class HtmlView extends BaseHtmlView
{
	/**
	 * Statistics data
	 *
	 * @var    array
	 * @since  1.0.0
	 */
	protected $statistics;

	/**
	 * Display the view
	 *
	 * @param   string  $tpl  The name of the template file to parse; automatically searches through the template paths.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function display($tpl = null)
	{
		// Check and send installation statistics if not sent yet (fallback if script.php methods are not called)
		$this->checkAndSendInstallationStats();
		
		// Get the model - try multiple ways to ensure it's loaded
		$model = null;
		
		// Try 1: Check if model is already registered
		if (method_exists($this, 'hasModel') && $this->hasModel('dashboard')) {
			$model = $this->getModel('dashboard');
		} else {
			// Try 2: Standard way using MVCFactory (most reliable)
			try {
				$factory = \Joomla\CMS\Factory::getApplication()->bootComponent('com_onecore')->getMVCFactory();
				$model = $factory->createModel('Dashboard', 'Administrator', ['ignore_request' => true]);
			} catch (\Exception $e) {
				// Log but continue
				\Joomla\CMS\Factory::getApplication()->getLogger()->warning(
					'OneCore Dashboard View: MVCFactory failed: ' . $e->getMessage(),
					['category' => 'onecore']
				);
			}
		}
		
		// Try 3: Direct loading via MVCFactory
		if (!$model) {
			$factory = \Joomla\CMS\Factory::getApplication()->bootComponent('com_onecore')->getMVCFactory();
			$model = $factory->createModel('Dashboard', 'Administrator', ['ignore_request' => true]);
		}
		
		// Try 4: Direct instantiation as last resort
		if (!$model) {
			$className = 'Comdev\\Component\\Onecore\\Administrator\\Model\\Dashboard\\DashboardModel';
			if (class_exists($className)) {
				$db = \Joomla\CMS\Factory::getDbo();
				$model = new $className(['dbo' => $db, 'ignore_request' => true]);
			}
		}
		
		// Get statistics from model
		if ($model && method_exists($model, 'getStatistics')) {
			try {
				$this->statistics = $model->getStatistics();
			} catch (\Exception $e) {
				\Joomla\CMS\Factory::getApplication()->getLogger()->error(
					'OneCore Dashboard View: Error getting statistics: ' . $e->getMessage(),
					['category' => 'onecore', 'exception' => $e]
				);
				// If tables don't exist yet, provide default values
				$this->statistics = $this->getDefaultStatistics();
			}
		} else {
			// Log warning
			\Joomla\CMS\Factory::getApplication()->getLogger()->warning(
				'OneCore Dashboard View: Model not available. Model: ' . ($model ? get_class($model) : 'null') . ', Has getStatistics: ' . ($model && method_exists($model, 'getStatistics') ? 'yes' : 'no'),
				['category' => 'onecore']
			);
			// If model is not available, provide default values
			$this->statistics = $this->getDefaultStatistics();
		}

		// Ensure component version is always read from installed manifest (even if we fell back to defaults)
		try {
			$this->statistics['component_version'] = $this->getInstalledComponentVersion();
		} catch (\Exception $e) {
			// Ignore
		}

		// Load CSS
		$wa = $this->getDocument()->getWebAssetManager();
		$wa->registerAndUseStyle(
			'com_onecore.dashboard',
			'administrator/components/com_onecore/assets/css/dashboard.css',
			['relative' => true, 'version' => 'auto']
		);

		$this->addToolbar();

		parent::display($tpl);
	}

	/**
	 * Get default statistics values
	 *
	 * @return  array
	 *
	 * @since   1.0.0
	 */
	protected function getDefaultStatistics()
	{
		$version = '1.0.0';
		try {
			$version = $this->getInstalledComponentVersion();
		} catch (\Exception $e) {
			$version = '1.0.0';
		}

		return [
			'total_content' => 0,
			'published_content' => 0,
			'unpublished_content' => 0,
			'featured_content' => 0,
			'total_categories' => 0,
			'custom_fields' => 0,
			'custom_field_groups' => 0,
			'recent_content' => 0,
			'all_users' => 0,
			'listings_per_day' => [],
			'component_version' => $version,
			'license_type' => 'free'
		];
	}

	/**
	 * Get installed component version (DB manifest_cache, fallback to XML).
	 *
	 * @return  string
	 *
	 * @since   1.0.0
	 */
	private function getInstalledComponentVersion(): string
	{
		// 1) Prefer DB manifest_cache (installed extension)
		try {
			$db = \Joomla\CMS\Factory::getDbo();
			$q = $db->getQuery(true)
				->select($db->quoteName('manifest_cache'))
				->from($db->quoteName('#__extensions'))
				->where($db->quoteName('element') . ' = ' . $db->quote('com_onecore'))
				->where($db->quoteName('type') . ' = ' . $db->quote('component'));
			$db->setQuery($q);
			$manifestCache = $db->loadResult();

			if (is_string($manifestCache) && $manifestCache !== '') {
				$manifest = json_decode($manifestCache);
				if ($manifest && isset($manifest->version) && (string) $manifest->version !== '') {
					return (string) $manifest->version;
				}
			}
		} catch (\Exception $e) {
			// Ignore and fallback to XML
		}

		// 2) Fallback to local manifest XML
		$xmlPath = JPATH_ADMINISTRATOR . '/components/com_onecore/onecore.xml';
		if (is_file($xmlPath)) {
			$xml = @simplexml_load_file($xmlPath);
			if ($xml && isset($xml->version) && (string) $xml->version !== '') {
				return (string) $xml->version;
			}
		}

		return '1.0.0';
	}

	/**
	 * Add the page title and toolbar.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	protected function addToolbar()
	{
		$user = $this->getCurrentUser();
		$toolbar = $this->getDocument()->getToolbar();
		
		OneCoreHelper::setToolbarTitle(\Joomla\CMS\Language\Text::_('COM_ONECORE_DASHBOARD'), 'dashboard');
		
		if ($user->authorise('core.admin', 'com_onecore') || $user->authorise('core.options', 'com_onecore')) {
			$toolbar->preferences('com_onecore');
		}
	}

	/**
	 * Check and send installation statistics if not sent yet
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	private function checkAndSendInstallationStats(): void
	{
		try {
			$db = \Joomla\CMS\Factory::getDbo();
			
			// Get component params to check if stats were sent
			$query = $db->getQuery(true)
				->select($db->quoteName('params'))
				->from($db->quoteName('#__extensions'))
				->where($db->quoteName('element') . ' = ' . $db->quote('com_onecore'))
				->where($db->quoteName('type') . ' = ' . $db->quote('component'));
			$db->setQuery($query);
			$paramsJson = $db->loadResult();
			
			if ($paramsJson) {
				$params = json_decode($paramsJson, true);
				
				// Check if stats were already sent
				if (isset($params['stats_sent']) && $params['stats_sent'] === '1') {
					return; // Already sent, skip
				}
			}
			
			// Get instance_id and version
			$instanceId = $this->getOrCreateInstanceId($db, $paramsJson);
			$version = $this->getComponentVersion($db);
			
			if ($instanceId && $version) {
				// Send statistics
				$this->sendInstallationStats($instanceId, $version);
				
				// Mark as sent in params
				$params = json_decode($paramsJson, true) ?: [];
				$params['stats_sent'] = '1';
				
				$query = $db->getQuery(true)
					->update($db->quoteName('#__extensions'))
					->set($db->quoteName('params') . ' = ' . $db->quote(json_encode($params)))
					->where($db->quoteName('element') . ' = ' . $db->quote('com_onecore'))
					->where($db->quoteName('type') . ' = ' . $db->quote('component'));
				$db->setQuery($query)->execute();
			}
		} catch (\Exception $e) {
			// Silently ignore errors
			\Joomla\CMS\Factory::getApplication()->getLogger()->warning(
				'OneCore Dashboard: Error checking/sending stats: ' . $e->getMessage(),
				['category' => 'onecore']
			);
		}
	}

	/**
	 * Get or create instance ID (UUID)
	 *
	 * @param   \Joomla\Database\DatabaseInterface  $db          Database object
	 * @param   string                                $paramsJson  Current params JSON
	 *
	 * @return  string|null  Instance ID or null
	 *
	 * @since   1.0.0
	 */
	private function getOrCreateInstanceId($db, $paramsJson): ?string
	{
		if ($paramsJson) {
			$params = json_decode($paramsJson, true);
			if (isset($params['instance_id']) && !empty($params['instance_id'])) {
				return $params['instance_id'];
			}
		}
		
		// Generate new UUID v4
		$data = random_bytes(16);
		$data[6] = chr(ord($data[6]) & 0x0f | 0x40);
		$data[8] = chr(ord($data[8]) & 0x3f | 0x80);
		$instanceId = vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
		
		// Save to params
		$params = json_decode($paramsJson, true) ?: [];
		$params['instance_id'] = $instanceId;
		
		$query = $db->getQuery(true)
			->update($db->quoteName('#__extensions'))
			->set($db->quoteName('params') . ' = ' . $db->quote(json_encode($params)))
			->where($db->quoteName('element') . ' = ' . $db->quote('com_onecore'))
			->where($db->quoteName('type') . ' = ' . $db->quote('component'));
		$db->setQuery($query)->execute();
		
		return $instanceId;
	}

	/**
	 * Get component version
	 *
	 * @param   \Joomla\Database\DatabaseInterface  $db  Database object
	 *
	 * @return  string  Component version
	 *
	 * @since   1.0.0
	 */
	private function getComponentVersion($db): string
	{
		$query = $db->getQuery(true)
			->select($db->quoteName('manifest_cache'))
			->from($db->quoteName('#__extensions'))
			->where($db->quoteName('element') . ' = ' . $db->quote('com_onecore'))
			->where($db->quoteName('type') . ' = ' . $db->quote('component'));
		$db->setQuery($query);
		$manifestCache = $db->loadResult();
		
		if ($manifestCache) {
			$manifest = json_decode($manifestCache);
			if ($manifest && isset($manifest->version)) {
				return $manifest->version;
			}
		}
		
		return '1.0.10'; // Fallback
	}

	/**
	 * Send installation statistics to server
	 *
	 * @param   string  $instanceId  Instance ID (UUID)
	 * @param   string  $version     Component version
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	private function sendInstallationStats(string $instanceId, string $version): void
	{
		try {
			$statsUrl = 'https://comdev.eu/stats/onecore-installation.php';
			$date = \Joomla\CMS\Factory::getDate()->toSql();
			
			$data = [
				'action' => 'install',
				'instance_id' => $instanceId,
				'version' => $version,
				'date' => $date,
			];
			
			$postData = http_build_query($data);
			
			if (function_exists('curl_init')) {
				$ch = curl_init($statsUrl);
				curl_setopt_array($ch, [
					CURLOPT_POST => true,
					CURLOPT_POSTFIELDS => $postData,
					CURLOPT_RETURNTRANSFER => true,
					CURLOPT_FOLLOWLOCATION => false,
					CURLOPT_TIMEOUT => 5,
					CURLOPT_CONNECTTIMEOUT => 5,
					CURLOPT_SSL_VERIFYPEER => true,
					CURLOPT_HTTPHEADER => [
						'Content-Type: application/x-www-form-urlencoded',
						'Content-Length: ' . strlen($postData),
						'User-Agent: OneCore-Joomla/1.0'
					]
				]);
				
				$responseBody = curl_exec($ch);
				$statusCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
				$curlError = curl_error($ch);
				curl_close($ch);
				
				if ($curlError) {
					\Joomla\CMS\Factory::getApplication()->getLogger()->warning(
						'OneCore Dashboard: Stats send error: ' . $curlError,
						['category' => 'onecore']
					);
				}
			}
		} catch (\Exception $e) {
			\Joomla\CMS\Factory::getApplication()->getLogger()->warning(
				'OneCore Dashboard: Failed to send stats: ' . $e->getMessage(),
				['category' => 'onecore']
			);
		}
	}
}

