<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\View\Content;

use Comdev\Component\Onecore\Administrator\Helper\OneCoreHelper;
use Joomla\CMS\MVC\View\HtmlView as BaseHtmlView;
use Joomla\CMS\Toolbar\ToolbarHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Content view class.
 *
 * @since  1.0.0
 */
class HtmlView extends BaseHtmlView
{
	/**
	 * The Form object
	 *
	 * @var  \Joomla\CMS\Form\Form
	 */
	protected $form;

	/**
	 * The active item
	 *
	 * @var  object
	 */
	protected $item;

	/**
	 * The model state
	 *
	 * @var  \Joomla\Registry\Registry
	 */
	protected $state;

	/**
	 * An array of items (for list view)
	 *
	 * @var  array
	 */
	protected $items;

	/**
	 * The pagination object (for list view)
	 *
	 * @var  \Joomla\CMS\Pagination\Pagination
	 */
	protected $pagination;

	/**
	 * Form object for search filters (for list view)
	 *
	 * @var  \Joomla\CMS\Form\Form
	 */
	public $filterForm;

	/**
	 * The active search filters (for list view)
	 *
	 * @var  array
	 */
	public $activeFilters;

	/**
	 * Method to get the model object
	 * Override to avoid undefined array key error
	 *
	 * @param   string  $name  The name of the model (optional)
	 *
	 * @return  \Joomla\CMS\MVC\Model\BaseDatabaseModel|null  The model object or null
	 *
	 * @since   1.0.0
	 */
	public function getModel($name = null)
	{
		if ($name === null) {
			$name = $this->_defaultModel;
		}

		$key = strtolower($name);
		
		// Check if model exists before accessing array
		if (isset($this->_models[$key])) {
			return $this->_models[$key];
		}
		
		return null;
	}

	/**
	 * Display the view
	 *
	 * @param   string  $tpl  The name of the template file to parse
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public function display($tpl = null)
	{
		$app = \Joomla\CMS\Factory::getApplication();
		$layout = $app->getInput()->getCmd('layout', 'default');

		// View only checks layout, Controller handles tasks
		if ($layout === 'edit') {
			$this->displayEdit($tpl);
		} elseif ($layout === 'modal') {
			$this->displayModal($tpl);
		} else {
			$this->displayList($tpl);
		}
	}

	/**
	 * Display the edit form
	 *
	 * @param   string  $tpl  The name of the template file to parse
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	protected function displayEdit($tpl = null)
	{
		// Create model directly (same approach as Controller) to avoid undefined array key error
		$className = 'Comdev\\Component\\Onecore\\Administrator\\Model\\Content\\ItemModel';
		$model = null;
		
		if (class_exists($className)) {
			$component = \Joomla\CMS\Factory::getApplication()->bootComponent('com_onecore');
			$factory = $component->getMVCFactory();
			
			$config = ['ignore_request' => false];
			$model = new $className($config, $factory);
			
			// Set required dependencies
			if ($model instanceof \Joomla\Database\DatabaseAwareInterface) {
				$model->setDatabase(\Joomla\CMS\Factory::getContainer()->get(\Joomla\Database\DatabaseInterface::class));
			}
			if ($model instanceof \Joomla\CMS\Form\FormFactoryAwareInterface) {
				$model->setFormFactory(\Joomla\CMS\Factory::getContainer()->get(\Joomla\CMS\Form\FormFactoryInterface::class));
			}
			if ($model instanceof \Joomla\Event\DispatcherAwareInterface) {
				$model->setDispatcher(\Joomla\CMS\Factory::getContainer()->get(\Joomla\Event\DispatcherInterface::class));
			}
		}

		if (!$model) {
			throw new \RuntimeException('Model Item not found');
		}

		// Ensure form paths are registered
		\Joomla\CMS\Form\Form::addFormPath(JPATH_ADMINISTRATOR . '/components/com_onecore/forms');
		
		$this->form = $model->getForm();
		$this->item = $model->getItem();
		$this->state = $model->getState();

		// Custom fields (definitions + values) - limited to selected categories
		$selectedCategories = [];
		if (is_object($this->item) && isset($this->item->categories) && is_array($this->item->categories)) {
			$selectedCategories = $this->item->categories;
		}
		$stateData = \Joomla\CMS\Factory::getApplication()->getUserState('com_onecore.edit.content.data', []);
		if (is_array($stateData) && isset($stateData['categories']) && is_array($stateData['categories'])) {
			$selectedCategories = $stateData['categories'];
		}

		$this->customfields = method_exists($model, 'getCustomFieldDefinitionsForCategories')
			? $model->getCustomFieldDefinitionsForCategories($selectedCategories)
			: (method_exists($model, 'getCustomFieldDefinitions') ? $model->getCustomFieldDefinitions() : []);

		// All custom fields + category tree for dynamic show/hide on add/edit without reload
		$this->customfieldsAll = method_exists($model, 'getCustomFieldDefinitions') ? $model->getCustomFieldDefinitions() : [];
		$this->categoriesTree = [];
		$this->rootCategoryIds = [];
		try {
			$db = \Joomla\CMS\Factory::getDbo();
			$q = $db->createQuery()
				->select([$db->quoteName('id'), $db->quoteName('lft'), $db->quoteName('rgt'), $db->quoteName('parent_id'), $db->quoteName('level')])
				->from($db->quoteName('#__one_categories'))
				->where($db->quoteName('extension') . ' = ' . $db->quote('com_onecore'));
			$db->setQuery($q);
			$rows = $db->loadObjectList() ?: [];
			foreach ($rows as $r) {
				$this->categoriesTree[(int) $r->id] = ['lft' => (int) $r->lft, 'rgt' => (int) $r->rgt];
				// Identify ROOT categories (parent_id = 0, level = 0)
				if ((int) $r->parent_id === 0 && (int) $r->level === 0) {
					$this->rootCategoryIds[] = (int) $r->id;
				}
			}
		} catch (\Exception $e) {
			$this->categoriesTree = [];
			$this->rootCategoryIds = [];
		}
		$this->customfieldValues = [];
		if (method_exists($model, 'getCustomFieldValues')) {
			$this->customfieldValues = $model->getCustomFieldValues((int) ($this->item->id ?? 0));
		}
		// If there is unsaved form data in user state, prefer its custom field values
		if (is_array($stateData) && isset($stateData['customfields']) && is_array($stateData['customfields'])) {
			$this->customfieldValues = $stateData['customfields'];
		}

		// Check for errors
		if ($errors = $model->getErrors()) {
			foreach ($errors as $error) {
				\Joomla\CMS\Factory::getApplication()->enqueueMessage($error, 'error');
			}
		}

		// Ensure item is always an object
		if (!is_object($this->item)) {
			$this->item = (object) ['id' => 0];
		}

		// Ensure form is loaded
		if (!$this->form) {
			\Joomla\CMS\Factory::getApplication()->enqueueMessage('Form could not be loaded', 'error');
			$this->form = false;
		}

		$this->addToolbar();

		parent::display($tpl);
	}

	/**
	 * Display the modal view
	 *
	 * @param   string  $tpl  The name of the template file to parse
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	protected function displayModal($tpl = null)
	{
		$model = $this->getModel('Content');
		
		if ($model) {
			$this->items = $model->getItems() ?: [];
			$this->pagination = $model->getPagination();
			$this->state = $model->getState();
			$this->filterForm = $model->getFilterForm();
			$this->activeFilters = $model->getActiveFilters();
		} else {
			$this->items = [];
			$this->pagination = null;
			$this->state = new \Joomla\Registry\Registry();
			$this->filterForm = null;
			$this->activeFilters = [];
		}

		// Ensure filterForm is never null - create empty form with minimal XML
		if (!$this->filterForm) {
			$formFactory = \Joomla\CMS\Factory::getContainer()->get(\Joomla\CMS\Form\FormFactoryInterface::class);
			$this->filterForm = $formFactory->createForm('filter_contents', ['control' => '']);
			// Load minimal XML structure
			$xml = new \SimpleXMLElement('<form><fields name="filter"></fields><fields name="list"></fields></form>');
			$this->filterForm->load($xml);
		}

		parent::display($tpl);
	}

	/**
	 * Display the list view
	 *
	 * @param   string  $tpl  The name of the template file to parse
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	protected function displayList($tpl = null)
	{
		$model = $this->getModel('Content');
		
		if ($model) {
			$this->items = $model->getItems() ?: [];
			$this->pagination = $model->getPagination();
			$this->state = $model->getState();
			$this->filterForm = $model->getFilterForm();
			$this->activeFilters = $model->getActiveFilters();
		} else {
			$this->items = [];
			$this->pagination = null;
			$this->state = new \Joomla\Registry\Registry();
			$this->filterForm = null;
			$this->activeFilters = [];
		}

		// Ensure filterForm is never null - create empty form with minimal XML
		if (!$this->filterForm) {
			$formFactory = \Joomla\CMS\Factory::getContainer()->get(\Joomla\CMS\Form\FormFactoryInterface::class);
			$this->filterForm = $formFactory->createForm('filter_contents', ['control' => '']);
			// Load minimal XML structure
			$xml = new \SimpleXMLElement('<form><fields name="filter"></fields><fields name="list"></fields></form>');
			$this->filterForm->load($xml);
		}

		$this->addToolbarList();

		parent::display($tpl);
	}

	/**
	 * Method to get the form object
	 *
	 * @return  \Joomla\CMS\Form\Form  The form object
	 *
	 * @since   1.0.0
	 */
	public function getForm()
	{
		return $this->form;
	}

	/**
	 * Add the page title and toolbar for edit form.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	protected function addToolbar()
	{
		$isNew = ($this->item->id ?? 0) == 0;

		OneCoreHelper::setToolbarTitle($isNew ? \Joomla\CMS\Language\Text::_('COM_ONECORE_CONTENT_NEW') : \Joomla\CMS\Language\Text::_('COM_ONECORE_CONTENT_EDIT'), 'file-alt');

		// Apply button stays in edit view (like com_banners)
		ToolbarHelper::apply('content.apply', 'JTOOLBAR_APPLY');
		
		// Save button goes to list view (like com_banners)
		ToolbarHelper::save('content.save', 'JTOOLBAR_SAVE');
		
		if (!$isNew) {
			ToolbarHelper::save2new('content.save2new');
		}
		
		ToolbarHelper::cancel('content.cancel', $isNew ? 'JTOOLBAR_CANCEL' : 'JTOOLBAR_CLOSE');
	}

	/**
	 * Add the page title and toolbar for list view.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	protected function addToolbarList()
	{
		OneCoreHelper::setToolbarTitle(\Joomla\CMS\Language\Text::_('COM_ONECORE_CONTENT'), 'file-alt');

		$canDo = \Joomla\CMS\Helper\ContentHelper::getActions('com_onecore');

		if ($canDo->get('core.create')) {
			ToolbarHelper::addNew('content.add');
		}

		if ($canDo->get('core.edit.state')) {
			ToolbarHelper::publish('content.publish', 'JTOOLBAR_PUBLISH', true);
			ToolbarHelper::unpublish('content.unpublish', 'JTOOLBAR_UNPUBLISH', true);
			ToolbarHelper::custom('content.featured', 'featured.png', 'featured_f2.png', 'JFEATURED', true);
			ToolbarHelper::custom('content.unfeatured', 'unfeatured.png', 'unfeatured_f2.png', 'JUNFEATURED', true);
		}

		if ($canDo->get('core.delete')) {
			ToolbarHelper::deleteList('JGLOBAL_CONFIRM_DELETE', 'content.delete');
		}
	}
}
