<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Model\Events;

use Joomla\CMS\MVC\Model\ListModel;
use Joomla\Database\ParameterType;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Events model class.
 *
 * @since  1.0.0
 */
class EventsModel extends ListModel
{
	/**
	 * Constructor.
	 *
	 * @param   array  $config  An optional associative array of configuration settings.
	 *
	 * @since   1.0.0
	 */
	public function __construct($config = [])
	{
		if (empty($config['filter_fields'])) {
			$config['filter_fields'] = [
				'id', 'a.id',
				'title', 'a.title',
				'alias', 'a.alias',
				'published', 'a.published',
				'featured', 'a.featured',
				'ordering', 'a.ordering',
				'access', 'a.access',
				'created', 'a.created',
				'created_by', 'a.created_by',
				'event_start', 'a.event_start',
				'event_end', 'a.event_end',
				'category_id', 'a.category_id',
			];
		}

		parent::__construct($config);
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * @param   string  $ordering   An optional ordering field.
	 * @param   string  $direction  An optional direction (asc|desc).
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	protected function populateState($ordering = 'a.created', $direction = 'DESC')
	{
		parent::populateState($ordering, $direction);
	}

	/**
	 * Method to build an SQL query to load the list data.
	 *
	 * @return  \Joomla\Database\DatabaseQuery
	 *
	 * @since   1.0.0
	 */
	protected function getListQuery()
	{
		$db = $this->getDatabase();
		$query = $db->getQuery(true);

		// Join with categories and recurrences (for recurring yes/no)
		$query->select(
			$this->getState(
				'list.select',
				'a.*, c.title AS category_title, r.id AS recurrence_id'
			)
		)
			->from($db->quoteName('#__one_events', 'a'))
			->join('LEFT', $db->quoteName('#__one_categories', 'c'), $db->quoteName('c.id') . ' = ' . $db->quoteName('a.category_id'))
			->join('LEFT', $db->quoteName('#__one_event_recurrences', 'r'), $db->quoteName('r.event_id') . ' = ' . $db->quoteName('a.id'));

		// Filter by published state
		$published = $this->getState('filter.published');

		if (is_numeric($published)) {
			$query->where($db->quoteName('a.published') . ' = :published')
				->bind(':published', $published, ParameterType::INTEGER);
		} elseif ($published === '') {
			$query->where('(' . $db->quoteName('a.published') . ' IN (0, 1))');
		}

		// Filter by category
		$categoryId = $this->getState('filter.category_id');

		if (is_numeric($categoryId)) {
			$query->where($db->quoteName('a.category_id') . ' = :category_id')
				->bind(':category_id', $categoryId, ParameterType::INTEGER);
		}

		// Filter by search
		$search = $this->getState('filter.search');

		if (!empty($search)) {
			$search = '%' . str_replace(' ', '%', $db->escape(trim($search), true) . '%');
			$query->where(
				'(' . $db->quoteName('a.title') . ' LIKE :search1 OR ' .
				$db->quoteName('a.alias') . ' LIKE :search2)'
			)
				->bind([':search1', ':search2'], $search);
		}

		// Add the list ordering clause
		$orderCol = $this->state->get('list.ordering', 'a.ordering');
		$orderDirn = $this->state->get('list.direction', 'ASC');

		$query->order($db->escape($orderCol) . ' ' . $db->escape($orderDirn));

		return $query;
	}

	/**
	 * Method to get a list of items.
	 *
	 * @return  mixed  An array of objects on success, false on failure.
	 *
	 * @since   1.0.0
	 */
	public function getItems()
	{
		$items = parent::getItems();

		if ($items) {
			foreach ($items as $item) {
				// Extract first image from images field
				$item->thumbnail = $this->getFirstImage($item->images ?? '');
			}
		}

		return $items;
	}

	/**
	 * Method to toggle the featured setting of one or more records.
	 *
	 * @param   array    &$pks   A list of the primary keys to change.
	 * @param   integer  $value  The value of the featured state.
	 *
	 * @return  boolean  True on success.
	 *
	 * @since   1.0.0
	 */
	public function featured(&$pks, $value = 1)
	{
		$user = $this->getCurrentUser();
		$table = $this->getTable();
		$pks = (array) $pks;

		// Access checks
		foreach ($pks as $i => $pk) {
			$table->reset();

			if ($table->load($pk)) {
				if (!$this->canEditState($table)) {
					// Prune items that you can't change
					unset($pks[$i]);
					$this->setError(\Joomla\CMS\Language\Text::_('JLIB_APPLICATION_ERROR_EDITSTATE_NOT_PERMITTED'));
					return false;
				}

				// If the table is checked out by another user, drop it
				if ($table->hasField('checked_out') && $table->checked_out && ($table->checked_out != $user->id)) {
					// Prune items that you can't change
					unset($pks[$i]);
				}

				// Prune items that are already at the given state
				if (property_exists($table, 'featured') && ($table->featured ?? $value) == $value) {
					unset($pks[$i]);
				}
			}
		}

		// Check if there are items to change
		if (!\count($pks)) {
			return true;
		}

		// Update the featured state
		$db = $this->getDatabase();
		$query = $db->getQuery(true)
			->update($db->quoteName('#__one_events'))
			->set($db->quoteName('featured') . ' = :featured')
			->whereIn($db->quoteName('id'), $pks)
			->bind(':featured', $value, ParameterType::INTEGER);

		$db->setQuery($query);

		try {
			$db->execute();
		} catch (\RuntimeException $e) {
			$this->setError($e->getMessage());
			return false;
		}

		// Clear the component's cache
		$this->cleanCache();

		return true;
	}

	/**
	 * Extract first image from images JSON field.
	 *
	 * @param   string  $imagesJson  JSON string with images data
	 *
	 * @return  string|null  First image path or null
	 *
	 * @since   1.0.0
	 */
	protected function getFirstImage($imagesJson)
	{
		if (empty($imagesJson)) {
			return null;
		}

		if (is_array($imagesJson)) {
			if (isset($imagesJson[0]) && is_array($imagesJson[0])) {
				foreach ($imagesJson as $imageData) {
					if (isset($imageData['image']) && !empty($imageData['image'])) {
						$imagePath = $imageData['image'];
						if (strpos($imagePath, '#') !== false) {
							$imagePath = explode('#', $imagePath)[0];
						}
						return $imagePath;
					}
				}
			}
			if (isset($imagesJson['image']) && !empty($imagesJson['image'])) {
				$imagePath = $imagesJson['image'];
				if (strpos($imagePath, '#') !== false) {
					$imagePath = explode('#', $imagePath)[0];
				}
				return $imagePath;
			}
			return null;
		}

		if (!is_string($imagesJson)) {
			return null;
		}

		$trimmed = trim($imagesJson);
		if (!empty($trimmed) && !preg_match('/^[\{\[]/', $trimmed)) {
			if (strpos($trimmed, '#') !== false) {
				$trimmed = explode('#', $trimmed)[0];
			}
			return $trimmed;
		}

		try {
			// First, try to decode as JSON array (subform format: [{"image":"path"}, ...])
			$images = json_decode($imagesJson, true);
			
			if (is_array($images) && json_last_error() === JSON_ERROR_NONE) {
				// If it's a subform array (repeatable-table format)
				if (isset($images[0]) && is_array($images[0])) {
					foreach ($images as $imageData) {
						if (isset($imageData['image']) && !empty($imageData['image'])) {
							$imagePath = $imageData['image'];
							// Handle new Joomla 4+ format: path#joomlaImage://...
							if (strpos($imagePath, '#') !== false) {
								$imagePath = explode('#', $imagePath)[0];
							}
							return $imagePath;
						}
					}
				}
				// If it's a flat array with 'image' key
				if (isset($images['image']) && !empty($images['image'])) {
					$imagePath = $images['image'];
					// Handle new Joomla 4+ format: path#joomlaImage://...
					if (strpos($imagePath, '#') !== false) {
						$imagePath = explode('#', $imagePath)[0];
					}
					return $imagePath;
				}
				// Check for any key containing 'image'
				foreach ($images as $key => $value) {
					if (is_string($key) && stripos($key, 'image') !== false && !empty($value) && is_string($value)) {
						$imagePath = $value;
						// Handle new Joomla 4+ format: path#joomlaImage://...
						if (strpos($imagePath, '#') !== false) {
							$imagePath = explode('#', $imagePath)[0];
						}
						return $imagePath;
					}
				}
			}
			
			// If JSON decode failed, try Registry format (INI-like format)
			try {
				$registry = new \Joomla\Registry\Registry($imagesJson);
				$images = $registry->toArray();
				
				if (is_array($images) && !empty($images)) {
					// Look for image fields in Registry format
					foreach ($images as $key => $value) {
						if (is_string($key) && stripos($key, 'image') !== false && !empty($value)) {
							// Check if it's an array with 'image' key
							if (is_array($value) && isset($value['image'])) {
								$imagePath = $value['image'];
								// Handle new Joomla 4+ format: path#joomlaImage://...
								if (strpos($imagePath, '#') !== false) {
									$imagePath = explode('#', $imagePath)[0];
								}
								return $imagePath;
							}
							// Or direct value
							if (is_string($value)) {
								$imagePath = $value;
								// Handle new Joomla 4+ format: path#joomlaImage://...
								if (strpos($imagePath, '#') !== false) {
									$imagePath = explode('#', $imagePath)[0];
								}
								return $imagePath;
							}
						}
					}
					
					// Also check for numeric keys (subform array in Registry)
					foreach ($images as $key => $value) {
						if (is_numeric($key) && is_array($value) && isset($value['image']) && !empty($value['image'])) {
							$imagePath = $value['image'];
							// Handle new Joomla 4+ format: path#joomlaImage://...
							if (strpos($imagePath, '#') !== false) {
								$imagePath = explode('#', $imagePath)[0];
							}
							return $imagePath;
						}
					}
					
					// Check if Registry has direct image path
					if (isset($images['image']) && !empty($images['image']) && is_string($images['image'])) {
						$imagePath = $images['image'];
						// Handle new Joomla 4+ format: path#joomlaImage://...
						if (strpos($imagePath, '#') !== false) {
							$imagePath = explode('#', $imagePath)[0];
						}
						return $imagePath;
					}
				}
			} catch (\Exception $e) {
				// Registry parsing failed, continue
			}
		} catch (\Exception $e) {
			// If parsing fails, return null
		}

		return null;
	}

	/**
	 * Method to get a table object, load it if necessary.
	 *
	 * @param   string  $name     The table name. Optional.
	 * @param   string  $prefix   The class prefix. Optional.
	 * @param   array   $options  Configuration array for model. Optional.
	 *
	 * @return  \Joomla\CMS\Table\Table  A Table object
	 *
	 * @since   1.0.0
	 */
	public function getTable($name = 'Event', $prefix = 'Administrator', $options = [])
	{
		return parent::getTable($name, $prefix, $options);
	}

	/**
	 * Method to test whether a record can have its state changed.
	 *
	 * @param   object  $record  A record object.
	 *
	 * @return  boolean  True if allowed to change the state of the record. Defaults to the permission for the component.
	 *
	 * @since   1.0.0
	 */
	protected function canEditState($record)
	{
		return $this->getCurrentUser()->authorise('core.edit.state', 'com_onecore');
	}

	/**
	 * Method to change the published state of one or more records.
	 *
	 * @param   array    &$pks   A list of the primary keys to change.
	 * @param   integer  $value  The value of the published state.
	 *
	 * @return  boolean  True on success.
	 *
	 * @since   1.0.0
	 */
	public function publish(&$pks, $value = 1)
	{
		$user = $this->getCurrentUser();
		$table = $this->getTable();
		$pks = (array) $pks;

		// Access checks
		foreach ($pks as $i => $pk) {
			$table->reset();

			if ($table->load($pk)) {
				if (!$this->canEditState($table)) {
					unset($pks[$i]);
					$this->setError(\Joomla\CMS\Language\Text::_('JLIB_APPLICATION_ERROR_EDITSTATE_NOT_PERMITTED'));
					return false;
				}

				if ($table->hasField('checked_out') && $table->checked_out && ($table->checked_out != $user->id)) {
					unset($pks[$i]);
				}

				$publishedColumnName = $table->getColumnAlias('published');
				if (property_exists($table, $publishedColumnName) && ($table->$publishedColumnName ?? $value) == $value) {
					unset($pks[$i]);
				}
			}
		}

		if (!\count($pks)) {
			return true;
		}

		if (!$table->publish($pks, $value, $user->id)) {
			$this->setError($table->getError());
			return false;
		}

		$this->cleanCache();

		return true;
	}
}
