<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Model\Event;

use Joomla\CMS\MVC\Model\AdminModel;
use Joomla\CMS\Table\Table;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Event item model class.
 *
 * @since  1.0.0
 */
class ItemModel extends AdminModel
{
	/**
	 * The model name (overrides auto-detection from class name)
	 * Must be 'item' for FormController to find ID in state
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	protected $name = 'item';

	/**
	 * The prefix to use with controller messages.
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	protected $text_prefix = 'COM_ONECORE_EVENTS';

	/**
	 * Method to get a table object, load it if necessary.
	 *
	 * @param   string  $name     The table name. Optional.
	 * @param   string  $prefix   The class prefix. Optional.
	 * @param   array   $options  Configuration array for model. Optional.
	 *
	 * @return  Table  A Table object
	 *
	 * @since   1.0.0
	 */
	public function getTable($name = 'Event', $prefix = 'Administrator', $options = [])
	{
		return parent::getTable($name, $prefix, $options);
	}

	/**
	 * Method to get the record form.
	 *
	 * @param   array    $data      Data for the form.
	 * @param   boolean  $loadData  True if the form is to load its own data (default case), false if not.
	 *
	 * @return  \Joomla\CMS\Form\Form|boolean  A Form object on success, false on failure
	 *
	 * @since   1.0.0
	 */
	public function getForm($data = [], $loadData = true)
	{
		\Joomla\CMS\Form\Form::addFormPath(JPATH_ADMINISTRATOR . '/components/com_onecore/forms');
		\Joomla\CMS\Form\Form::addFieldPath(JPATH_ADMINISTRATOR . '/components/com_onecore/src/Field');
		
		$form = $this->loadForm(
			'com_onecore.event',
			'events',
			['control' => 'jform', 'load_data' => $loadData]
		);

		if (empty($form)) {
			$formFactory = \Joomla\CMS\Factory::getContainer()->get(\Joomla\CMS\Form\FormFactoryInterface::class);
			$form = $formFactory->createForm('com_onecore.event', ['control' => 'jform']);
			
			if ($form->loadFile('events', false)) {
				if ($loadData) {
					$formData = $this->loadFormData();
					$form->bind($formData);
				}
			} else {
				$this->setError('Form file events.xml not found');
				return false;
			}
		}

		return $form;
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	protected function populateState()
	{
		$app = \Joomla\CMS\Factory::getApplication();
		$pk = $app->getInput()->getInt('id');
		$this->setState($this->getName() . '.id', $pk);
	}

	/**
	 * Method to get the data that should be injected in the form.
	 *
	 * @return  mixed  The data for the form.
	 *
	 * @since   1.0.0
	 */
	protected function loadFormData()
	{
		$app = \Joomla\CMS\Factory::getApplication();
		$data = $app->getUserState('com_onecore.edit.event.data', []);

		if (empty($data)) {
			$data = $this->getItem();
		}

		$this->preprocessData('com_onecore.event', $data);

		return $data;
	}

	/**
	 * Prepare and sanitise the table prior to saving.
	 *
	 * @param   \Joomla\CMS\Table\Table  $table  The Table object
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	protected function prepareTable($table)
	{
		$date = \Joomla\CMS\Factory::getDate()->toSql();
		$user = \Joomla\CMS\Factory::getApplication()->getIdentity();

		if (empty($table->created) || $table->created === '0000-00-00 00:00:00') {
			$table->created = $date;
		}

		if (property_exists($table, 'modified')) {
			$table->modified = $date;
		}

		if (empty($table->id)) {
			if (property_exists($table, 'created_by') && empty($table->created_by)) {
				$table->created_by = $user->id;
			}
		} else {
			if (property_exists($table, 'modified_by')) {
				$table->modified_by = $user->id;
			}
		}

		if (property_exists($table, 'params') && empty($table->params)) {
			$table->params = '{}';
		}

		if (property_exists($table, 'metakey') && empty($table->metakey)) {
			$table->metakey = '';
		}

		if (property_exists($table, 'metadesc') && empty($table->metadesc)) {
			$table->metadesc = '';
		}

		if (property_exists($table, 'metadata') && empty($table->metadata)) {
			$table->metadata = '{}';
		}

		if (property_exists($table, 'ordering') && $table->ordering === null) {
			$table->ordering = 0;
		}

		if (property_exists($table, 'hits') && $table->hits === null) {
			$table->hits = 0;
		}

		if (property_exists($table, 'checked_out') && $table->checked_out === null) {
			$table->checked_out = 0;
		}

		if (property_exists($table, 'checked_out_time') && empty($table->checked_out_time)) {
			$table->checked_out_time = null;
		}

		// Handle images field - convert array to JSON if needed
		if (property_exists($table, 'images')) {
			if (is_array($table->images)) {
				$table->images = json_encode($table->images);
			} elseif (is_string($table->images) && !empty($table->images)) {
				$decoded = json_decode($table->images, true);
				if (json_last_error() !== JSON_ERROR_NONE) {
					try {
						$registry = new \Joomla\Registry\Registry($table->images);
						$imagesArray = $registry->toArray();
						if (is_array($imagesArray)) {
							$table->images = json_encode($imagesArray);
						} else {
							$table->images = '';
						}
					} catch (\Exception $e) {
						$table->images = '';
					}
				}
			} elseif (empty($table->images)) {
				$table->images = '';
			}
		}

		// Handle category_id
		if (property_exists($table, 'category_id') && ($table->category_id === null || $table->category_id === '')) {
			$table->category_id = null;
		}

		// Handle recur_event
		if (property_exists($table, 'recur_event') && empty($table->recur_event)) {
			$table->recur_event = '';
		}

		if (property_exists($table, 'publish_up') && empty($table->publish_up)) {
			$table->publish_up = null;
		}

		if (property_exists($table, 'publish_down') && empty($table->publish_down)) {
			$table->publish_down = null;
		}

		// Handle event_end - convert empty string to NULL
		if (property_exists($table, 'event_end')) {
			if (empty($table->event_end) || $table->event_end === '' || $table->event_end === '0000-00-00 00:00:00') {
				$table->event_end = null;
			}
		}

		// Handle event_start - ensure it's not empty
		if (property_exists($table, 'event_start') && empty($table->event_start)) {
			$table->event_start = $date; // Default to current date if not set
		}
	}

	/**
	 * Method to save the form data.
	 *
	 * @param   array  $data  The form data.
	 *
	 * @return  boolean  True on success, False on error.
	 *
	 * @since   1.0.0
	 */
	public function save($data)
	{
		$app = \Joomla\CMS\Factory::getApplication();
		$input = $app->getInput();

		// Automatic handling of alias for empty fields (for both new and existing records)
		if (\in_array($input->get('task'), ['apply', 'save', 'save2new'])) {
			if (empty($data['alias']) || $data['alias'] == null || trim($data['alias']) == '') {
				if ($app->get('unicodeslugs') == 1) {
					$data['alias'] = \Joomla\CMS\Filter\OutputFilter::stringUrlUnicodeSlug($data['title']);
				} else {
					$data['alias'] = \Joomla\CMS\Filter\OutputFilter::stringURLSafe($data['title']);
				}

				$table = $this->getTable();

				// Check if alias already exists (excluding current record if editing)
				$existingId = isset($data['id']) ? (int) $data['id'] : 0;
				if ($table->load(['alias' => $data['alias']])) {
					if ($table->id != $existingId) {
						$msg = \Joomla\CMS\Language\Text::_('COM_ONECORE_SAVE_WARNING');
					}
				}

				[$title, $alias] = $this->generateNewTitle(0, $data['alias'], $data['title']);
				$data['alias'] = $alias;

				if (isset($msg)) {
					$app->enqueueMessage($msg, 'warning');
				}
			}
		}

		// Extract address data before saving
		$addressData = [];
		$addressFields = [
			'address',
			'address_street',
			'address_street_number',
			'address_postal_code',
			'address_city',
			'address_country',
			'latitude',
			'longitude'
		];
		
		foreach ($addressFields as $field) {
			if (isset($data[$field])) {
				$addressData[$field] = $data[$field];
				unset($data[$field]);
			}
		}

		// Extract categories before saving (they go to junction table, not main table)
		$categories = [];
		if (isset($data['categories'])) {
			$categories = is_array($data['categories']) ? $data['categories'] : [$data['categories']];
			// Filter out empty values and 0
			$categories = array_filter(array_map('intval', $categories), function($catId) {
				return $catId > 0;
			});
			unset($data['categories']);
		}

		// Extract content before saving (they go to junction table, not main table)
		$content = [];
		if (isset($data['content'])) {
			$content = is_array($data['content']) ? $data['content'] : [$data['content']];
			// Filter out empty values and 0
			$content = array_filter(array_map('intval', $content), function($contentId) {
				return $contentId > 0;
			});
			unset($data['content']);
		}

		// Extract recurrence_enabled before saving
		$recurrenceEnabled = 0;
		if (isset($data['recurrence_enabled'])) {
			$recurrenceEnabled = (int) $data['recurrence_enabled'];
			unset($data['recurrence_enabled']);
		}

		// Extract recurrence data before saving (they go to separate table)
		$recurrenceData = [];
		$recurrenceFields = [
			'recurrence_frequency',
			'recurrence_interval',
			'recurrence_byday',
			'recurrence_until_date',
			'recurrence_count',
			'recurrence_exceptions'
		];
		
		foreach ($recurrenceFields as $field) {
			if (isset($data[$field])) {
				$recurrenceData[$field] = $data[$field];
				unset($data[$field]);
			}
		}

		// Validate event_end >= event_start
		if (!empty($data['event_start']) && !empty($data['event_end'])) {
			$eventStart = $data['event_start'];
			$eventEnd = $data['event_end'];
			
			// Convert to DateTime objects for comparison
			try {
				$startDate = new \DateTime($eventStart);
				$endDate = new \DateTime($eventEnd);
				
				if ($endDate < $startDate) {
					$this->setError(\Joomla\CMS\Language\Text::_('COM_ONECORE_EVENT_END_VALIDATION_ERROR'));
					$app->enqueueMessage(\Joomla\CMS\Language\Text::_('COM_ONECORE_EVENT_END_VALIDATION_ERROR'), 'error');
					return false;
				}
			} catch (\Exception $e) {
				// Invalid date format, let parent::save() handle it
			}
		}

		// Save the main event record
		$isNew = empty($data['id']) || (int) $data['id'] === 0;
		
		// Get table BEFORE save to check ID after
		$table = $this->getTable();
		$key = $table->getKeyName();
		
		$result = parent::save($data);

		if ($result) {
			// AdminModel::save() sets ID in state as getName() . '.id'
			// getName() now returns 'item' (not 'eventitem'), so state key is 'item.id'
			$eventId = (int) $this->getState($this->getName() . '.id');
			
			// Check if table has ID set (insertObject should set it)
			if (isset($table->$key) && (int) $table->$key > 0) {
				$eventId = (int) $table->$key;
				// Ensure state is set
				if ($eventId != $this->getState($this->getName() . '.id')) {
					$this->setState($this->getName() . '.id', $eventId);
				}
			}
			
			// If AdminModel didn't set ID, try to get it from table
			if ($eventId <= 0) {
				$eventId = (int) ($table->$key ?? 0);
				if ($eventId > 0) {
					$this->setState($this->getName() . '.id', $eventId);
				}
			}
			
			// Fallback: get from data (for existing records)
			if ($eventId <= 0) {
				$eventId = (int) ($data['id'] ?? 0);
			}
			
			// Fallback for new records: query database directly for last insert ID
			if ($eventId <= 0 && $isNew) {
				$db = $this->getDatabase();
				// Try insertid() first
				$insertId = (int) $db->insertid();
				
				// If insertid() returns 0, try querying the table directly
				if ($insertId <= 0) {
					$query = $db->createQuery()
						->select('MAX(' . $db->quoteName($key) . ')')
						->from($db->quoteName($table->getTableName()));
					$db->setQuery($query);
					$maxId = (int) $db->loadResult();
					
					// If max ID is greater than what we had, use it
					if ($maxId > 0) {
						$insertId = $maxId;
					}
				}
				
				if ($insertId > 0) {
					$eventId = $insertId;
					// Update table and state
					$table->$key = $eventId;
					$this->setState($this->getName() . '.id', $eventId);
				}
			}
			
			if ($eventId > 0) {
				// Save address data to separate table
				try {
					$db = $this->getDatabase();
					$now = \Joomla\CMS\Factory::getDate()->toSql();
					
					$query = $db->createQuery()
						->select($db->quoteName('id'))
						->from($db->quoteName('#__one_event_addresses'))
						->where($db->quoteName('event_id') . ' = :eventId')
						->bind(':eventId', $eventId, \Joomla\Database\ParameterType::INTEGER);
					$db->setQuery($query);
					$existingAddressId = $db->loadResult();
					
					$latitude = null;
					$longitude = null;
					
					if (!empty($addressData['latitude'])) {
						$latitude = is_numeric($addressData['latitude']) ? (string) $addressData['latitude'] : null;
					}
					
					if (!empty($addressData['longitude'])) {
						$longitude = is_numeric($addressData['longitude']) ? (string) $addressData['longitude'] : null;
					}
					
					$addressRecord = [
						'event_id' => $eventId,
						'address' => $addressData['address'] ?? '',
						'address_street' => $addressData['address_street'] ?? '',
						'address_street_number' => $addressData['address_street_number'] ?? '',
						'address_postal_code' => $addressData['address_postal_code'] ?? '',
						'address_city' => $addressData['address_city'] ?? '',
						'address_country' => $addressData['address_country'] ?? '',
						'latitude' => $latitude,
						'longitude' => $longitude,
						'modified' => $now
					];
					
					if ($existingAddressId) {
						$query = $db->createQuery()
							->update($db->quoteName('#__one_event_addresses'))
							->set($db->quoteName('address') . ' = :address')
							->set($db->quoteName('address_street') . ' = :address_street')
							->set($db->quoteName('address_street_number') . ' = :address_street_number')
							->set($db->quoteName('address_postal_code') . ' = :address_postal_code')
							->set($db->quoteName('address_city') . ' = :address_city')
							->set($db->quoteName('address_country') . ' = :address_country')
							->set($db->quoteName('latitude') . ' = :latitude')
							->set($db->quoteName('longitude') . ' = :longitude')
							->set($db->quoteName('modified') . ' = :modified')
							->where($db->quoteName('id') . ' = :id')
							->bind(':address', $addressRecord['address'])
							->bind(':address_street', $addressRecord['address_street'])
							->bind(':address_street_number', $addressRecord['address_street_number'])
							->bind(':address_postal_code', $addressRecord['address_postal_code'])
							->bind(':address_city', $addressRecord['address_city'])
							->bind(':address_country', $addressRecord['address_country'])
							->bind(':latitude', $addressRecord['latitude'], \Joomla\Database\ParameterType::STRING)
							->bind(':longitude', $addressRecord['longitude'], \Joomla\Database\ParameterType::STRING)
							->bind(':modified', $addressRecord['modified'])
							->bind(':id', $existingAddressId, \Joomla\Database\ParameterType::INTEGER);
						$db->setQuery($query);
						$db->execute();
					} else {
						$addressRecord['created'] = $now;
						$query = $db->createQuery()
							->insert($db->quoteName('#__one_event_addresses'))
							->columns([
								$db->quoteName('event_id'),
								$db->quoteName('address'),
								$db->quoteName('address_street'),
								$db->quoteName('address_street_number'),
								$db->quoteName('address_postal_code'),
								$db->quoteName('address_city'),
								$db->quoteName('address_country'),
								$db->quoteName('latitude'),
								$db->quoteName('longitude'),
								$db->quoteName('created'),
								$db->quoteName('modified')
							])
							->values(':eventId, :address, :address_street, :address_street_number, :address_postal_code, :address_city, :address_country, :latitude, :longitude, :created, :modified')
							->bind(':eventId', $addressRecord['event_id'], \Joomla\Database\ParameterType::INTEGER)
							->bind(':address', $addressRecord['address'])
							->bind(':address_street', $addressRecord['address_street'])
							->bind(':address_street_number', $addressRecord['address_street_number'])
							->bind(':address_postal_code', $addressRecord['address_postal_code'])
							->bind(':address_city', $addressRecord['address_city'])
							->bind(':address_country', $addressRecord['address_country'])
							->bind(':latitude', $addressRecord['latitude'], \Joomla\Database\ParameterType::STRING)
							->bind(':longitude', $addressRecord['longitude'], \Joomla\Database\ParameterType::STRING)
							->bind(':created', $addressRecord['created'])
							->bind(':modified', $addressRecord['modified']);
						$db->setQuery($query);
						$db->execute();
					}
				} catch (\Exception $e) {
					$app->enqueueMessage('Address data could not be saved: ' . $e->getMessage(), 'warning');
				}

				// Save categories to junction table
				try {
					$db = $this->getDatabase();
					
					// Delete existing category associations
					$query = $db->createQuery()
						->delete($db->quoteName('#__one_event_categories'))
						->where($db->quoteName('event_id') . ' = :eventId')
						->bind(':eventId', $eventId, \Joomla\Database\ParameterType::INTEGER);
					$db->setQuery($query);
					$db->execute();

					// Insert new category associations
					if (!empty($categories)) {
						foreach ($categories as $categoryId) {
							if ($categoryId > 0) {
								$query = $db->createQuery()
									->insert($db->quoteName('#__one_event_categories'))
									->columns([$db->quoteName('event_id'), $db->quoteName('category_id')])
									->values(':eventId, :categoryId')
									->bind(':eventId', $eventId, \Joomla\Database\ParameterType::INTEGER)
									->bind(':categoryId', $categoryId, \Joomla\Database\ParameterType::INTEGER);
								$db->setQuery($query);
								$db->execute();
							}
						}
					}
				} catch (\Exception $e) {
					$app->enqueueMessage('Categories could not be saved: ' . $e->getMessage(), 'warning');
				}

				// Save content associations to junction table
				try {
					$db = $this->getDatabase();
					
					// Delete existing content associations
					$query = $db->createQuery()
						->delete($db->quoteName('#__one_content_events'))
						->where($db->quoteName('event_id') . ' = :eventId')
						->bind(':eventId', $eventId, \Joomla\Database\ParameterType::INTEGER);
					$db->setQuery($query);
					$db->execute();

					// Insert new content associations
					if (!empty($content)) {
						foreach ($content as $contentId) {
							if ($contentId > 0) {
								$query = $db->createQuery()
									->insert($db->quoteName('#__one_content_events'))
									->columns([$db->quoteName('event_id'), $db->quoteName('content_id')])
									->values(':eventId, :contentId')
									->bind(':eventId', $eventId, \Joomla\Database\ParameterType::INTEGER)
									->bind(':contentId', $contentId, \Joomla\Database\ParameterType::INTEGER);
								$db->setQuery($query);
								$db->execute();
							}
						}
					}
				} catch (\Exception $e) {
					$app->enqueueMessage('Content associations could not be saved: ' . $e->getMessage(), 'warning');
				}

				// Save recurrence data to separate table
				try {
					$db = $this->getDatabase();
					
					// If recurrence_enabled is 0, delete existing recurrence record
					if ($recurrenceEnabled == 0) {
						$query = $db->createQuery()
							->delete($db->quoteName('#__one_event_recurrences'))
							->where($db->quoteName('event_id') . ' = :eventId')
							->bind(':eventId', $eventId, \Joomla\Database\ParameterType::INTEGER);
						$db->setQuery($query);
						$db->execute();
					} elseif ($recurrenceEnabled == 1) {
						// Check if recurrence data exists (at least one field filled)
						$hasRecurrenceData = false;
						foreach ($recurrenceData as $value) {
							if (!empty($value)) {
								$hasRecurrenceData = true;
								break;
							}
						}
						
						if ($hasRecurrenceData) {
						// Check if recurrence record already exists
						$query = $db->createQuery()
							->select($db->quoteName('id'))
							->from($db->quoteName('#__one_event_recurrences'))
							->where($db->quoteName('event_id') . ' = :eventId')
							->bind(':eventId', $eventId, \Joomla\Database\ParameterType::INTEGER);
						$db->setQuery($query);
						$existingRecurrenceId = $db->loadResult();
						
						// Process exceptions - convert comma-separated string to JSON array
						$exceptions = null;
						if (!empty($recurrenceData['recurrence_exceptions'])) {
							$exceptionsStr = trim($recurrenceData['recurrence_exceptions']);
							if (!empty($exceptionsStr)) {
								$exceptionsArray = array_map('trim', explode(',', $exceptionsStr));
								$exceptionsArray = array_filter($exceptionsArray);
								if (!empty($exceptionsArray)) {
									$exceptions = json_encode($exceptionsArray);
								}
							}
						}
						
						// Process until_date - convert empty string to null
						$untilDate = !empty($recurrenceData['recurrence_until_date']) ? $recurrenceData['recurrence_until_date'] : null;
						if ($untilDate === '' || $untilDate === '0000-00-00' || $untilDate === '0000-00-00 00:00:00') {
							$untilDate = null;
						}
						
						// Process count - convert empty string to null
						$count = !empty($recurrenceData['recurrence_count']) ? (int) $recurrenceData['recurrence_count'] : null;
						
						// Process interval - default to 1
						$interval = !empty($recurrenceData['recurrence_interval']) ? (int) $recurrenceData['recurrence_interval'] : 1;
						
						// Process frequency - default to 'daily'
						$frequency = !empty($recurrenceData['recurrence_frequency']) ? $recurrenceData['recurrence_frequency'] : 'daily';
						
						// Process byday - convert to null if empty
						$byday = !empty($recurrenceData['recurrence_byday']) ? $recurrenceData['recurrence_byday'] : null;
						
						if ($existingRecurrenceId) {
							// Update existing recurrence record
							$query = $db->createQuery()
								->update($db->quoteName('#__one_event_recurrences'))
								->set($db->quoteName('frequency') . ' = :frequency')
								->set($db->quoteName('interval') . ' = :interval')
								->set($db->quoteName('byday') . ' = :byday')
								->set($db->quoteName('until_date') . ' = :untilDate')
								->set($db->quoteName('count') . ' = :count')
								->set($db->quoteName('exceptions') . ' = :exceptions')
								->where($db->quoteName('event_id') . ' = :eventId')
								->bind(':frequency', $frequency)
								->bind(':interval', $interval, \Joomla\Database\ParameterType::INTEGER)
								->bind(':byday', $byday)
								->bind(':untilDate', $untilDate)
								->bind(':count', $count, \Joomla\Database\ParameterType::INTEGER)
								->bind(':exceptions', $exceptions)
								->bind(':eventId', $eventId, \Joomla\Database\ParameterType::INTEGER);
							$db->setQuery($query);
							$db->execute();
						} else {
							// Insert new recurrence record
							$query = $db->createQuery()
								->insert($db->quoteName('#__one_event_recurrences'))
								->columns([
									$db->quoteName('event_id'),
									$db->quoteName('frequency'),
									$db->quoteName('interval'),
									$db->quoteName('byday'),
									$db->quoteName('until_date'),
									$db->quoteName('count'),
									$db->quoteName('exceptions')
								])
								->values(':eventId, :frequency, :interval, :byday, :untilDate, :count, :exceptions')
								->bind(':eventId', $eventId, \Joomla\Database\ParameterType::INTEGER)
								->bind(':frequency', $frequency)
								->bind(':interval', $interval, \Joomla\Database\ParameterType::INTEGER)
								->bind(':byday', $byday)
								->bind(':untilDate', $untilDate)
								->bind(':count', $count, \Joomla\Database\ParameterType::INTEGER)
								->bind(':exceptions', $exceptions);
							$db->setQuery($query);
							$db->execute();
						}
					}
				}
				} catch (\Exception $e) {
					$app->enqueueMessage('Recurrence data could not be saved: ' . $e->getMessage(), 'warning');
				}

				// Save custom field values
				// Get values directly from input (form sends as jform[customfields][fieldId])
				$inputCustomFields = $input->get('jform', [], 'array')['customfields'] ?? [];
				$customValues = [];
				
				if (is_array($inputCustomFields) && !empty($inputCustomFields)) {
					// Normalize keys to integers for consistency
					foreach ($inputCustomFields as $k => $v) {
						$key = is_numeric($k) ? (int) $k : $k;
						// For multiselect, values come as arrays (from name="jform[customfields][fieldId][]")
						// Ensure we preserve arrays for multiselect fields
						if (is_array($v)) {
							// Filter out empty values
							$v = array_filter($v, function($item) {
								return $item !== '' && $item !== null;
							});
							// Only store if not empty
							if (!empty($v)) {
								$customValues[$key] = array_values($v); // Re-index array
							}
						} else {
							$customValues[$key] = $v;
						}
					}
				}
				
				if (!empty($customValues) || !empty($this->getState('customfields'))) {
					try {
						$this->ensureCustomFieldsSchema();
						$db = $this->getDatabase();

						// Get selected categories for filtering custom fields
						$selectedCategories = $categories ?: [];
						if (empty($selectedCategories) && is_object($this->getItem($eventId)) && property_exists($this->getItem($eventId), 'categories')) {
							$selectedCategories = $this->getItem($eventId)->categories ?: [];
						}

						// Get custom field definitions for selected categories
						$definitions = $this->getCustomFieldDefinitionsForCategories($selectedCategories);

						// Remove existing values first
						$query = $db->createQuery()
							->delete($db->quoteName('#__one_customfield_values'))
							->where($db->quoteName('event_id') . ' = :eventId')
							->bind(':eventId', $eventId, \Joomla\Database\ParameterType::INTEGER);
						$db->setQuery($query);
						$db->execute();

						$now = \Joomla\CMS\Factory::getDate()->toSql();

						// Get all field IDs that were submitted from $customValues
						$submittedFieldIds = array_values(array_filter(array_map('intval', array_keys($customValues))));
						
						if (!empty($submittedFieldIds)) {
							// Get allowed groups first (ROOT + selected categories)
							$allowedGroupIds = [];
							
							// Always allow groups assigned to ROOT EVENTS category (for events) - title='ROOT EVENTS', entity_type='events' or NULL
							$rootGroupQuery = $db->createQuery()
								->select($db->quoteName('g.id'))
								->from($db->quoteName('#__one_customfield_groups', 'g'))
								->leftJoin(
									$db->quoteName('#__one_categories', 'gcat'),
									$db->quoteName('gcat.id') . ' = ' . $db->quoteName('g.category_id')
								)
								->where($db->quoteName('g.published') . ' = 1')
								->where('(' . $db->quoteName('g.entity_type') . ' = ' . $db->quote('events') . ' OR ' . $db->quoteName('g.entity_type') . ' IS NULL)')
								->where($db->quoteName('gcat.title') . ' = ' . $db->quote('ROOT EVENTS'))
								->where($db->quoteName('gcat.extension') . ' = ' . $db->quote('com_onecore'));
							$db->setQuery($rootGroupQuery);
							$rootGroupIds = $db->loadColumn() ?: [];
							$allowedGroupIds = array_merge($allowedGroupIds, $rootGroupIds);
							
							// Also allow groups for selected categories (if any)
							if (!empty($selectedCategories)) {
								$groupQuery = $db->createQuery()
									->select($db->quoteName('g.id'))
									->from($db->quoteName('#__one_customfield_groups', 'g'))
									->innerJoin(
										$db->quoteName('#__one_categories', 'csel'),
										$db->quoteName('csel.id') . ' IN (' . implode(',', array_map('intval', $selectedCategories)) . ')'
									)
									->leftJoin(
										$db->quoteName('#__one_categories', 'gcat'),
										$db->quoteName('gcat.id') . ' = ' . $db->quoteName('g.category_id')
									)
									->where($db->quoteName('g.published') . ' = 1')
									->where('(' . $db->quoteName('g.entity_type') . ' = ' . $db->quote('events') . ' OR ' . $db->quoteName('g.entity_type') . ' IS NULL)')
									// Exclude ROOT EVENTS groups (title='ROOT EVENTS') as they're already included
									->where($db->quoteName('gcat.title') . ' != ' . $db->quote('ROOT EVENTS'))
									->where($db->quoteName('gcat.lft') . ' <= ' . $db->quoteName('csel.lft'))
									->where($db->quoteName('gcat.rgt') . ' >= ' . $db->quoteName('csel.rgt'))
									->group($db->quoteName('g.id'));
								$db->setQuery($groupQuery);
								$categoryGroupIds = $db->loadColumn() ?: [];
								$allowedGroupIds = array_unique(array_merge($allowedGroupIds, $categoryGroupIds));
							}
							
							// Get all fields from allowed groups that were submitted
							if (!empty($allowedGroupIds)) {
								$fieldsQuery = $db->createQuery()
									->select($db->quoteName('f.id'))
									->from($db->quoteName('#__one_customfields', 'f'))
									->where($db->quoteName('f.published') . ' = 1')
									->whereIn($db->quoteName('f.group_id'), $allowedGroupIds)
									->whereIn($db->quoteName('f.id'), $submittedFieldIds);
								$db->setQuery($fieldsQuery);
								$allowedFieldIds = $db->loadColumn() ?: [];
							} else {
								$allowedFieldIds = [];
							}
						} else {
							// If no submitted fields, use definitions as fallback
							$allowedFieldIds = [];
							foreach ($definitions as $def) {
								$fid = (int) ($def->id ?? 0);
								if ($fid > 0) {
									$allowedFieldIds[] = $fid;
								}
							}
						}

						// Insert current values (skip empty) - only for allowed fields
						foreach ($submittedFieldIds as $fieldId) {
							// Get value from $customValues (already normalized with integer keys)
							$val = $customValues[$fieldId] ?? null;
							
							// Fallback: try string key
							if ($val === null) {
								$val = $customValues[(string) $fieldId] ?? null;
							}
							
							// Additional fallback: check input directly
							if ($val === null) {
								$inputVal = $input->get('jform', [], 'array')['customfields'][$fieldId] ?? 
								            $input->get('jform', [], 'array')['customfields'][(string) $fieldId] ?? null;
								if ($inputVal !== null) {
									// Handle arrays for multiselect
									if (is_array($inputVal)) {
										$inputVal = array_filter($inputVal, function($item) {
											return $item !== '' && $item !== null;
										});
										if (!empty($inputVal)) {
											$val = array_values($inputVal);
										}
									} else {
										$val = $inputVal;
									}
								}
							}
							
							// Get field type first to handle values correctly
							$fieldType = null;
							foreach ($definitions as $def) {
								if ((int) ($def->id ?? 0) === $fieldId) {
									$fieldType = $def->type ?? 'input';
									break;
								}
							}
							
							// If not found in definitions, query directly from DB
							if ($fieldType === null) {
								try {
									$q = $db->createQuery()
										->select($db->quoteName('type'))
										->from($db->quoteName('#__one_customfields'))
										->where($db->quoteName('id') . ' = :fieldId')
										->bind(':fieldId', $fieldId, \Joomla\Database\ParameterType::INTEGER);
									$db->setQuery($q);
									$fieldType = $db->loadResult() ?: 'input';
								} catch (\Exception $e) {
									$fieldType = 'input'; // Default fallback
								}
							}
							
							// Handle different value types based on field type
							$originalVal = $val;
							if (is_array($val)) {
								// For multiselect, store as JSON array
								if ($fieldType === 'multiselect') {
									$val = json_encode(array_filter(array_map('trim', $val)));
								} else {
									// For other array values (e.g., textarea), join with newlines
									$val = implode("\n", $val);
								}
							} elseif ($val === null) {
								$val = '';
							} else {
								$val = (string) $val;
							}
							
							// Check if field is allowed for these categories
							if (!in_array($fieldId, $allowedFieldIds, true)) {
								continue; // Skip fields not allowed for these categories
							}
							
							// Skip only if completely empty
							if ($fieldType === 'textarea') {
								if ($val === '' || $val === null) {
									continue;
								}
							} elseif ($fieldType === 'multiselect') {
								if (!is_array($originalVal) || empty(array_filter(array_map('trim', $originalVal)))) {
									continue;
								}
							} else {
								if (trim($val) === '') {
									continue;
								}
							}

							// Use raw SQL with proper NULL handling
							// MySQL requires actual NULL value, not string 'NULL'
							// Use REPLACE INTO to handle duplicates (based on UNIQUE KEY idx_event_field)
							$sql = 'REPLACE INTO ' . $db->quoteName('#__one_customfield_values') . 
								' (' . $db->quoteName('content_id') . ', ' . 
								$db->quoteName('event_id') . ', ' . 
								$db->quoteName('field_id') . ', ' . 
								$db->quoteName('value') . ', ' . 
								$db->quoteName('created') . ') VALUES (' .
								'NULL, ' .
								(int) $eventId . ', ' .
								(int) $fieldId . ', ' .
								$db->quote($val) . ', ' .
								$db->quote($now) . ')';
							$db->setQuery($sql);
							$db->execute();
						}
					} catch (\Exception $e) {
						// Non-fatal: still consider main save successful
						$app->enqueueMessage('Custom field values could not be saved: ' . $e->getMessage(), 'warning');
					}
				}
			}

			$app->setUserState('com_onecore.edit.event.data', null);
		}

		return $result;
	}

	/**
	 * Method to get a single record.
	 *
	 * @param   integer  $pk  The id of the primary key.
	 *
	 * @return  \stdClass|false  Object on success, false on failure.
	 *
	 * @since   1.0.0
	 */
	public function getItem($pk = null)
	{
		$item = parent::getItem($pk);

		if (!$item) {
			$item = (object) ['id' => 0, 'images' => [], 'categories' => []];
		} elseif ($item->id) {
			// Decode images field if it's a JSON string
			if (property_exists($item, 'images') && !empty($item->images) && is_string($item->images)) {
				$decoded = json_decode($item->images, true);
				if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
					$item->images = $decoded;
				} else {
					try {
						$registry = new \Joomla\Registry\Registry($item->images);
						$item->images = $registry->toArray();
					} catch (\Exception $e) {
						$item->images = [];
					}
				}
			} elseif (empty($item->images)) {
				$item->images = [];
			}

			// Load categories from junction table
			$db = $this->getDatabase();
			$query = $db->createQuery()
				->select($db->quoteName('category_id'))
				->from($db->quoteName('#__one_event_categories'))
				->where($db->quoteName('event_id') . ' = :eventId')
				->bind(':eventId', $item->id, \Joomla\Database\ParameterType::INTEGER);
			$db->setQuery($query);
			$categories = $db->loadColumn();
			
			$item->categories = $categories ?: [];

			// Load content associations from junction table
			$query = $db->createQuery()
				->select($db->quoteName('content_id'))
				->from($db->quoteName('#__one_content_events'))
				->where($db->quoteName('event_id') . ' = :eventId')
				->bind(':eventId', $item->id, \Joomla\Database\ParameterType::INTEGER);
			$db->setQuery($query);
			$content = $db->loadColumn();
			
			$item->content = $content ?: [];

			// Load address data from separate table
			$query = $db->createQuery()
				->select([
					$db->quoteName('address'),
					$db->quoteName('address_street'),
					$db->quoteName('address_street_number'),
					$db->quoteName('address_postal_code'),
					$db->quoteName('address_city'),
					$db->quoteName('address_country'),
					$db->quoteName('latitude'),
					$db->quoteName('longitude')
				])
				->from($db->quoteName('#__one_event_addresses'))
				->where($db->quoteName('event_id') . ' = :eventId')
				->bind(':eventId', $item->id, \Joomla\Database\ParameterType::INTEGER);
			$db->setQuery($query);
			$addressData = $db->loadObject();
			
			if ($addressData) {
				$item->address = $addressData->address ?? '';
				$item->address_street = $addressData->address_street ?? '';
				$item->address_street_number = $addressData->address_street_number ?? '';
				$item->address_postal_code = $addressData->address_postal_code ?? '';
				$item->address_city = $addressData->address_city ?? '';
				$item->address_country = $addressData->address_country ?? '';
				$item->latitude = $addressData->latitude ?? '';
				$item->longitude = $addressData->longitude ?? '';
			} else {
				$item->address = '';
				$item->address_street = '';
				$item->address_street_number = '';
				$item->address_postal_code = '';
				$item->address_city = '';
				$item->address_country = '';
				$item->latitude = '';
				$item->longitude = '';
			}

			// Load recurrence data from separate table
			$query = $db->createQuery()
				->select([
					$db->quoteName('frequency'),
					$db->quoteName('interval'),
					$db->quoteName('byday'),
					$db->quoteName('until_date'),
					$db->quoteName('count'),
					$db->quoteName('exceptions')
				])
				->from($db->quoteName('#__one_event_recurrences'))
				->where($db->quoteName('event_id') . ' = :eventId')
				->bind(':eventId', $item->id, \Joomla\Database\ParameterType::INTEGER);
			$db->setQuery($query);
			$recurrenceData = $db->loadObject();
			
			if ($recurrenceData) {
				$item->recurrence_enabled = 1;
				$item->recurrence_frequency = $recurrenceData->frequency ?? 'daily';
				$item->recurrence_interval = $recurrenceData->interval ?? 1;
				$item->recurrence_byday = $recurrenceData->byday ?? '';
				$item->recurrence_until_date = $recurrenceData->until_date ?? '';
				$item->recurrence_count = $recurrenceData->count ?? null;
				
				// Decode exceptions JSON to comma-separated string
				if (!empty($recurrenceData->exceptions)) {
					$exceptionsArray = json_decode($recurrenceData->exceptions, true);
					if (is_array($exceptionsArray)) {
						$item->recurrence_exceptions = implode(',', $exceptionsArray);
					} else {
						$item->recurrence_exceptions = '';
					}
				} else {
					$item->recurrence_exceptions = '';
				}
			} else {
				$item->recurrence_enabled = 0;
				$item->recurrence_frequency = 'daily';
				$item->recurrence_interval = 1;
				$item->recurrence_byday = '';
				$item->recurrence_until_date = '';
				$item->recurrence_count = null;
				$item->recurrence_exceptions = '';
			}
		} else {
			// New record
			if (!property_exists($item, 'images') || empty($item->images)) {
				$item->images = [];
			}
			$item->categories = [];
			$item->address = '';
			$item->address_street = '';
			$item->address_street_number = '';
			$item->address_postal_code = '';
			$item->address_city = '';
			$item->address_country = '';
			$item->latitude = '';
			$item->longitude = '';
		}

		return $item;
	}

	/**
	 * Method to change the title & alias.
	 *
	 * @param   integer  $parentId  The id of the parent (not used for events, kept for compatibility).
	 * @param   string   $alias     The alias.
	 * @param   string   $title     The title.
	 *
	 * @return  array  Contains the modified title and alias.
	 *
	 * @since   1.0.0
	 */
	protected function generateNewTitle($parentId, $alias, $title)
	{
		$table = $this->getTable();

		while ($table->load(['alias' => $alias])) {
			if ($title == $table->title) {
				$title = \Joomla\String\StringHelper::increment($title);
			}

			$alias = \Joomla\String\StringHelper::increment($alias, 'dash');
		}

		return [$title, $alias];
	}

	/**
	 * Ensure custom fields tables exist (same as Content ItemModel).
	 *
	 * @return  void
	 *
	 * @since  1.0.0
	 */
	private function ensureCustomFieldsSchema(): void
	{
		$db = $this->getDatabase();
		$prefix = $db->getPrefix();

		try {
			$tables = $db->getTableList();
		} catch (\Exception $e) {
			return;
		}

		// Tables should already exist from Content model
		// Just ensure they're accessible
		if (!in_array($prefix . 'one_customfield_values', $tables, true)) {
			// If table doesn't exist, try to create it (should be created by Content model)
			return;
		}
	}

	/**
	 * Get all custom field definitions (all groups).
	 *
	 * @return  array
	 *
	 * @since  1.0.0
	 */
	public function getCustomFieldDefinitions(): array
	{
		$this->ensureCustomFieldsSchema();

		$db = $this->getDatabase();

		// Get all published groups for events (entity_type='events' or NULL for backward compatibility)
		$groupQuery = $db->createQuery()
			->select($db->quoteName('id'))
			->from($db->quoteName('#__one_customfield_groups'))
			->where($db->quoteName('published') . ' = 1')
			->where('(' . $db->quoteName('entity_type') . ' = ' . $db->quote('events') . ' OR ' . $db->quoteName('entity_type') . ' IS NULL)');

		$db->setQuery($groupQuery);
		$groupIds = $db->loadColumn() ?: [];

		if (empty($groupIds)) {
			return [];
		}

		// Get all fields from all published groups, including group's category_id
		$query = $db->createQuery()
			->select([
				$db->quoteName('f.id'),
				$db->quoteName('f.group_id'),
				$db->quoteName('g.category_id', 'category_id'),
				$db->quoteName('f.title'),
				$db->quoteName('f.type'),
				$db->quoteName('f.required'),
				$db->quoteName('f.height'),
				$db->quoteName('f.options'),
				$db->quoteName('f.ordering'),
				$db->quoteName('f.display_label'),
				$db->quoteName('f.label_position'),
				$db->quoteName('f.searchable'),
				$db->quoteName('f.pos_items'),
				$db->quoteName('f.pos_item'),
				$db->quoteName('f.pos_module'),
				$db->quoteName('f.published'),
			])
			->from($db->quoteName('#__one_customfields', 'f'))
			->leftJoin(
				$db->quoteName('#__one_customfield_groups', 'g'),
				$db->quoteName('g.id') . ' = ' . $db->quoteName('f.group_id')
			)
			->where($db->quoteName('f.published') . ' = 1')
			->whereIn($db->quoteName('f.group_id'), $groupIds)
			->order($db->quoteName('f.group_id') . ' ASC, ' . $db->quoteName('f.ordering') . ' ASC, ' . $db->quoteName('f.id') . ' ASC');

		$db->setQuery($query);

		try {
			return $db->loadObjectList() ?: [];
		} catch (\Exception $e) {
			return [];
		}
	}

	/**
	 * Load published custom field definitions limited to selected categories.
	 *
	 * @param   array  $categoryIds  Category ids
	 *
	 * @return  array
	 *
	 * @since  1.0.0
	 */
	public function getCustomFieldDefinitionsForCategories(array $categoryIds): array
	{
		$this->ensureCustomFieldsSchema();

		$categoryIds = array_values(array_filter(array_map('intval', $categoryIds)));

		$db = $this->getDatabase();
		$groupIds = [];

		// Always get groups assigned to ROOT EVENTS category (for events) - title='ROOT EVENTS', entity_type='events' or NULL
		$rootGroupQuery = $db->createQuery()
			->select($db->quoteName('g.id'))
			->from($db->quoteName('#__one_customfield_groups', 'g'))
			->leftJoin(
				$db->quoteName('#__one_categories', 'gcat'),
				$db->quoteName('gcat.id') . ' = ' . $db->quoteName('g.category_id')
			)
			->where($db->quoteName('g.published') . ' = 1')
			->where('(' . $db->quoteName('g.entity_type') . ' = ' . $db->quote('events') . ' OR ' . $db->quoteName('g.entity_type') . ' IS NULL)')
			->where($db->quoteName('gcat.title') . ' = ' . $db->quote('ROOT EVENTS'))
			->where($db->quoteName('gcat.extension') . ' = ' . $db->quote('com_onecore'));

		$db->setQuery($rootGroupQuery);
		try {
			$rootGroupIds = $db->loadColumn() ?: [];
			$groupIds = array_merge($groupIds, $rootGroupIds);
		} catch (\Exception $e) {
			// Ignore errors, continue with category-specific groups
		}

		// Get groups for selected categories (if any)
		if (!empty($categoryIds)) {
			$groupQuery = $db->createQuery()
				->select($db->quoteName('g.id'))
				->from($db->quoteName('#__one_customfield_groups', 'g'))
				->innerJoin(
					$db->quoteName('#__one_categories', 'csel'),
					$db->quoteName('csel.id') . ' IN (' . implode(',', $categoryIds) . ')'
				)
				->leftJoin(
					$db->quoteName('#__one_categories', 'gcat'),
					$db->quoteName('gcat.id') . ' = ' . $db->quoteName('g.category_id')
				)
				->where($db->quoteName('g.published') . ' = 1')
				->where('(' . $db->quoteName('g.entity_type') . ' = ' . $db->quote('events') . ' OR ' . $db->quoteName('g.entity_type') . ' IS NULL)')
				// Inheritance: group category is ancestor of selected category (or equals)
				// Exclude ROOT EVENTS groups (title='ROOT EVENTS') as they're already included above
				->where($db->quoteName('gcat.title') . ' != ' . $db->quote('ROOT EVENTS'))
				->where('(' . $db->quoteName('gcat.lft') . ' <= ' . $db->quoteName('csel.lft') . ' AND ' . $db->quoteName('gcat.rgt') . ' >= ' . $db->quoteName('csel.rgt') . ')')
				->group($db->quoteName('g.id'));

			$db->setQuery($groupQuery);
			try {
				$categoryGroupIds = $db->loadColumn() ?: [];
				$groupIds = array_unique(array_merge($groupIds, $categoryGroupIds));
			} catch (\Exception $e) {
				// Ignore errors, use ROOT groups only
			}
		}

		// If no groups found, return empty array
		if (empty($groupIds)) {
			return [];
		}

		// Get all fields from the found groups, including group's category_id
		$fieldsQuery = $db->createQuery()
			->select([
				$db->quoteName('f.id'),
				$db->quoteName('f.group_id'),
				$db->quoteName('g.category_id', 'category_id'),
				$db->quoteName('f.title'),
				$db->quoteName('f.type'),
				$db->quoteName('f.required'),
				$db->quoteName('f.height'),
				$db->quoteName('f.options'),
				$db->quoteName('f.ordering'),
				$db->quoteName('f.display_label'),
				$db->quoteName('f.label_position'),
				$db->quoteName('f.searchable'),
				$db->quoteName('f.pos_items'),
				$db->quoteName('f.pos_item'),
				$db->quoteName('f.pos_module'),
				$db->quoteName('f.published'),
			])
			->from($db->quoteName('#__one_customfields', 'f'))
			->leftJoin(
				$db->quoteName('#__one_customfield_groups', 'g'),
				$db->quoteName('g.id') . ' = ' . $db->quoteName('f.group_id')
			)
			->where($db->quoteName('f.published') . ' = 1')
			->whereIn($db->quoteName('f.group_id'), $groupIds)
			->order($db->quoteName('f.group_id') . ' ASC, ' . $db->quoteName('f.ordering') . ' ASC, ' . $db->quoteName('f.id') . ' ASC');

		$db->setQuery($fieldsQuery);

		try {
			return $db->loadObjectList() ?: [];
		} catch (\Exception $e) {
			return [];
		}
	}

	/**
	 * Load custom field values for event item.
	 * Note: Uses same table as content, but with event_id stored in content_id column
	 * (we could create separate table later if needed).
	 *
	 * @param   int  $eventId  Event id
	 *
	 * @return  array  field_id => value
	 *
	 * @since  1.0.0
	 */
	public function getCustomFieldValues(int $eventId): array
	{
		$this->ensureCustomFieldsSchema();

		if ($eventId <= 0) {
			return [];
		}

		// For now, we'll use a separate query approach
		// Check if there's a customfield_event_values table, otherwise use content_id column
		// For simplicity, we'll use a different table name pattern or add event_id column check
		// But for now, let's use a marker approach or separate table lookup
		
		// Actually, for events, we should check if we need a separate table
		// For now, let's try using the same table but with a prefix/identifier
		// Or we can use negative event IDs temporarily (not ideal)
		
		// Best approach: check if table has event_id column, otherwise use content_id
		// For now, we'll assume events use the same structure but we need to track separately
		
		// Let's check the table structure first
		$db = $this->getDatabase();
		
		// Try to check if there's an event-specific values table
		$prefix = $db->getPrefix();
		try {
			$tables = $db->getTableList();
			$hasEventValuesTable = in_array($prefix . 'one_customfield_event_values', $tables, true);
		} catch (\Exception $e) {
			$hasEventValuesTable = false;
		}
		
		// Use event_id column to distinguish from content
		$query = $db->createQuery()
			->select([$db->quoteName('field_id'), $db->quoteName('value')])
			->from($db->quoteName('#__one_customfield_values'))
			->where($db->quoteName('event_id') . ' = :eventId')
			->bind(':eventId', $eventId, \Joomla\Database\ParameterType::INTEGER);

		$db->setQuery($query);

		try {
			$rows = $db->loadObjectList() ?: [];
			$values = [];
			// Get field types to determine if value should be parsed as JSON
			$fieldTypes = [];
			if (!empty($rows)) {
				$fieldIds = array_map(function($row) { return (int) $row->field_id; }, $rows);
				$typeQuery = $db->createQuery()
					->select([$db->quoteName('id'), $db->quoteName('type')])
					->from($db->quoteName('#__one_customfields'))
					->whereIn($db->quoteName('id'), $fieldIds);
				$db->setQuery($typeQuery);
				$typeRows = $db->loadObjectList() ?: [];
				foreach ($typeRows as $typeRow) {
					$fieldTypes[(int) $typeRow->id] = (string) $typeRow->type;
				}
			}
			
			foreach ($rows as $row) {
				$fieldId = (int) $row->field_id;
				$fieldType = $fieldTypes[$fieldId] ?? '';
				$value = (string) $row->value;
				
				// For multiselect, parse JSON array
				if ($fieldType === 'multiselect' && !empty($value)) {
					$decoded = json_decode($value, true);
					$values[$fieldId] = is_array($decoded) ? $decoded : [$value];
				} else {
					$values[$fieldId] = $value;
				}
			}
			return $values;
		} catch (\Exception $e) {
			return [];
		}
	}
}
