<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Model\Dashboard;

use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Model\BaseDatabaseModel;
use Joomla\Database\ParameterType;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Dashboard model class.
 *
 * @since  1.0.0
 */
class DashboardModel extends BaseDatabaseModel
{
	/**
	 * Get statistics for dashboard
	 *
	 * @return  array
	 *
	 * @since   1.0.0
	 */
	public function getStatistics()
	{
		$db = $this->getDatabase();
		$stats = [];

		try {
			// Get total content count (all content, excluding trashed)
			$query = $db->getQuery(true)
				->select('COUNT(*)')
				->from($db->quoteName('#__one_content'))
				->where($db->quoteName('published') . ' != -2'); // Exclude trashed (-2)
			$db->setQuery($query);
			$result = $db->loadResult();
			$stats['total_content'] = (int) $result;
		} catch (\Exception $e) {
			Factory::getApplication()->getLogger()->error(
				'OneCore Dashboard: Error getting total content count: ' . $e->getMessage(),
				['category' => 'onecore', 'exception' => $e]
			);
			$stats['total_content'] = 0;
		}

		try {
			// Get published content count
			$query = $db->getQuery(true)
				->select('COUNT(*)')
				->from($db->quoteName('#__one_content'))
				->where($db->quoteName('published') . ' = 1');
			$db->setQuery($query);
			$result = $db->loadResult();
			$stats['published_content'] = (int) $result;
		} catch (\Exception $e) {
			Factory::getApplication()->getLogger()->error(
				'OneCore Dashboard: Error getting published content count: ' . $e->getMessage(),
				['category' => 'onecore', 'exception' => $e]
			);
			$stats['published_content'] = 0;
		}

		try {
			// Get total categories count
			$query = $db->getQuery(true)
				->select('COUNT(*)')
				->from($db->quoteName('#__one_categories'));
			$db->setQuery($query);
			$stats['total_categories'] = (int) $db->loadResult();
		} catch (\Exception $e) {
			$stats['total_categories'] = 0;
		}

		try {
			// Get total events count (all events, excluding trashed)
			$query = $db->getQuery(true)
				->select('COUNT(*)')
				->from($db->quoteName('#__one_events'))
				->where($db->quoteName('published') . ' != -2'); // Exclude trashed (-2)
			$db->setQuery($query);
			$result = $db->loadResult();
			$stats['total_events'] = (int) $result;
		} catch (\Exception $e) {
			Factory::getApplication()->getLogger()->error(
				'OneCore Dashboard: Error getting total events count: ' . $e->getMessage(),
				['category' => 'onecore', 'exception' => $e]
			);
			$stats['total_events'] = 0;
		}

		try {
			// Get featured content count (featured = 1, published = 1)
			$query = $db->getQuery(true)
				->select('COUNT(*)')
				->from($db->quoteName('#__one_content'))
				->where($db->quoteName('featured') . ' = 1')
				->where($db->quoteName('published') . ' = 1');
			$db->setQuery($query);
			$result = $db->loadResult();
			$stats['featured_content'] = (int) $result;
		} catch (\Exception $e) {
			Factory::getApplication()->getLogger()->error(
				'OneCore Dashboard: Error getting featured content count: ' . $e->getMessage(),
				['category' => 'onecore', 'exception' => $e]
			);
			$stats['featured_content'] = 0;
		}

		try {
			// Get custom fields count
			$query = $db->getQuery(true)
				->select('COUNT(*)')
				->from($db->quoteName('#__one_customfields'))
				->where($db->quoteName('published') . ' = 1');
			$db->setQuery($query);
			$stats['custom_fields'] = (int) $db->loadResult();
		} catch (\Exception $e) {
			$stats['custom_fields'] = 0;
		}

		try {
			// Get custom field groups count
			$query = $db->getQuery(true)
				->select('COUNT(*)')
				->from($db->quoteName('#__one_customfield_groups'))
				->where($db->quoteName('published') . ' = 1');
			$db->setQuery($query);
			$stats['custom_field_groups'] = (int) $db->loadResult();
		} catch (\Exception $e) {
			$stats['custom_field_groups'] = 0;
		}

		try {
			// Get recent content (last 7 days)
			$date = Factory::getDate();
			$date->modify('-7 days');
			$dateSql = $date->toSql();
			$query = $db->getQuery(true)
				->select('COUNT(*)')
				->from($db->quoteName('#__one_content'))
				->where($db->quoteName('created') . ' >= :date')
				->bind(':date', $dateSql, ParameterType::STRING);
			$db->setQuery($query);
			$stats['recent_content'] = (int) $db->loadResult();
		} catch (\Exception $e) {
			$stats['recent_content'] = 0;
		}

		try {
			// Get all users count (users who created content)
			$query = $db->getQuery(true)
				->select('COUNT(DISTINCT ' . $db->quoteName('created_by') . ')')
				->from($db->quoteName('#__one_content'))
				->where($db->quoteName('created_by') . ' > 0');
			$db->setQuery($query);
			$result = $db->loadResult();
			$stats['all_users'] = (int) $result;
		} catch (\Exception $e) {
			Factory::getApplication()->getLogger()->error(
				'OneCore Dashboard: Error getting all users count: ' . $e->getMessage(),
				['category' => 'onecore', 'exception' => $e]
			);
			$stats['all_users'] = 0;
		}

		try {
			// Get listings per day for last 7 days
			$stats['listings_per_day'] = [];
			for ($i = 6; $i >= 0; $i--) {
				$date = Factory::getDate();
				$date->modify('-' . $i . ' days');
				$dateStart = clone $date;
				$dateStart->setTime(0, 0, 0);
				$dateEnd = clone $date;
				$dateEnd->setTime(23, 59, 59);
				
				$dateStartSql = $dateStart->toSql();
				$dateEndSql = $dateEnd->toSql();
				
				$query = $db->getQuery(true)
					->select('COUNT(*)')
					->from($db->quoteName('#__one_content'))
					->where($db->quoteName('created') . ' >= :dateStart')
					->where($db->quoteName('created') . ' <= :dateEnd')
					->bind(':dateStart', $dateStartSql, ParameterType::STRING)
					->bind(':dateEnd', $dateEndSql, ParameterType::STRING);
				$db->setQuery($query);
				$count = (int) $db->loadResult();
				
				$stats['listings_per_day'][] = [
					'day' => $date->format('D'),
					'date' => $date->format('M j'),
					'count' => $count
				];
			}
		} catch (\Exception $e) {
			Factory::getApplication()->getLogger()->error(
				'OneCore Dashboard: Error getting listings per day: ' . $e->getMessage(),
				['category' => 'onecore', 'exception' => $e]
			);
			$stats['listings_per_day'] = [];
		}

		// Get component version and license info
		try {
			// Read installed component version from database (most reliable)
			$manifestCache = null;
			$manifest = null;

			$q = $db->getQuery(true)
				->select($db->quoteName('manifest_cache'))
				->from($db->quoteName('#__extensions'))
				->where($db->quoteName('element') . ' = ' . $db->quote('com_onecore'))
				->where($db->quoteName('type') . ' = ' . $db->quote('component'));
			$db->setQuery($q);
			$manifestCache = $db->loadResult();

			if (!empty($manifestCache) && is_string($manifestCache)) {
				$manifest = json_decode($manifestCache);
			}

			$stats['component_version'] = ($manifest && isset($manifest->version) && !empty($manifest->version))
				? (string) $manifest->version
				: '1.0.0';

			// Fallback: read version directly from manifest XML (if DB value missing)
			if ($stats['component_version'] === '1.0.0') {
				$xmlPath = JPATH_ADMINISTRATOR . '/components/com_onecore/onecore.xml';
				if (is_file($xmlPath)) {
					$xml = @simplexml_load_file($xmlPath);
					if ($xml && isset($xml->version) && (string) $xml->version !== '') {
						$stats['component_version'] = (string) $xml->version;
					}
				}
			}
			
			// Get license information
			$licenseHelper = \Comdev\Component\Onecore\Administrator\Helper\LicenseHelper::class;
			$stats['license_type'] = $licenseHelper::getLicenseType();
			$stats['license_info'] = $licenseHelper::getLicenseInfo();
			$stats['component_version_name'] = $licenseHelper::getComponentVersionName();
		} catch (\Exception $e) {
			$stats['component_version'] = '1.0.0';
			$stats['license_type'] = 'free';
			$stats['component_version_name'] = 'OneCore';
			$stats['license_info'] = [
				'key' => '',
				'status' => 'inactive',
				'is_active' => false,
				'type' => 'free',
				'version_name' => 'OneCore',
				'last_check' => 0,
				'last_check_formatted' => ''
			];
		}

		return $stats;
	}
}

