<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Model\Customfields;

use Joomla\CMS\MVC\Model\ListModel;
use Joomla\Database\ParameterType;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Customfields model class.
 *
 * @since  1.0.4
 */
class CustomfieldsModel extends ListModel
{
	/**
	 * Ensure custom fields tables exist (runtime fallback).
	 *
	 * @return void
	 *
	 * @since  1.0.5
	 */
	private function ensureTables(): void
	{
		$db = $this->getDatabase();
		$prefix = $db->getPrefix();

		try {
			$existing = $db->getTableList();
		} catch (\Exception $e) {
			return;
		}

		$need = [
			$prefix . 'one_customfields',
			$prefix . 'one_customfield_values',
			$prefix . 'one_customfield_categories',
		];

		if (empty(array_diff($need, $existing))) {
			return;
		}

		$queries = [
			"CREATE TABLE IF NOT EXISTS `{$prefix}one_customfields` (
				`id` int NOT NULL AUTO_INCREMENT,
				`title` varchar(255) NOT NULL DEFAULT '',
				`type` varchar(20) NOT NULL DEFAULT 'input',
				`required` tinyint NOT NULL DEFAULT 0,
				`height` int NOT NULL DEFAULT 0,
				`ordering` int NOT NULL DEFAULT 0,
				`display_label` tinyint NOT NULL DEFAULT 1,
				`label_position` varchar(10) NOT NULL DEFAULT 'next',
				`searchable` tinyint NOT NULL DEFAULT 0,
				`pos_items` varchar(20) NOT NULL DEFAULT 'none',
				`pos_item` varchar(20) NOT NULL DEFAULT 'none',
				`pos_module` varchar(20) NOT NULL DEFAULT 'none',
				`published` tinyint NOT NULL DEFAULT 1,
				`created` datetime NOT NULL,
				`modified` datetime,
				PRIMARY KEY (`id`),
				KEY `idx_published` (`published`),
				KEY `idx_ordering` (`ordering`),
				KEY `idx_searchable` (`searchable`),
				KEY `idx_type` (`type`)
			) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 DEFAULT COLLATE=utf8mb4_unicode_ci",
			"CREATE TABLE IF NOT EXISTS `{$prefix}one_customfield_categories` (
				`field_id` int NOT NULL,
				`category_id` int NOT NULL,
				PRIMARY KEY (`field_id`, `category_id`),
				KEY `idx_field_id` (`field_id`),
				KEY `idx_category_id` (`category_id`)
			) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 DEFAULT COLLATE=utf8mb4_unicode_ci",
		];

		foreach ($queries as $sql) {
			try {
				$db->setQuery($sql)->execute();
			} catch (\Exception $e) {
				// ignore
			}
		}
	}

	/**
	 * Constructor.
	 *
	 * @param   array  $config  An optional associative array of configuration settings.
	 *
	 * @since   1.0.4
	 */
	public function __construct($config = [])
	{
		if (empty($config['filter_fields'])) {
			$config['filter_fields'] = [
				'id', 'a.id',
				'title', 'a.title',
				'published', 'a.published',
				'type', 'a.type',
				'ordering', 'a.ordering',
				'searchable', 'a.searchable',
				'group_id', 'a.group_id',
				'group_title', 'g.title',
			];
		}

		parent::__construct($config);
	}

	/**
	 * Build an SQL query to load the list data.
	 *
	 * @return  \Joomla\Database\DatabaseQuery
	 *
	 * @since   1.0.4
	 */
	protected function getListQuery()
	{
		$db = $this->getDatabase();
		$query = $db->getQuery(true);

		$this->ensureTables();

		$query->select(
			$this->getState(
				'list.select',
				'a.*'
			)
		)
			->select($db->quoteName('g.title', 'group_title'))
			->select($db->quoteName('g.id', 'group_id'))
			->from($db->quoteName('#__one_customfields', 'a'))
			->leftJoin($db->quoteName('#__one_customfield_groups', 'g'), $db->quoteName('g.id') . ' = ' . $db->quoteName('a.group_id'));

		// Filter by published state
		$published = $this->getState('filter.published');
		if (is_numeric($published)) {
			$query->where($db->quoteName('a.published') . ' = :published')
				->bind(':published', (int) $published, ParameterType::INTEGER);
		}

		// Filter by type
		$type = $this->getState('filter.type');
		if (is_string($type) && $type !== '') {
			$query->where($db->quoteName('a.type') . ' = :type')
				->bind(':type', $type);
		}

		// Filter by search
		$search = $this->getState('filter.search');
		if (!empty($search)) {
			$search = '%' . str_replace(' ', '%', $db->escape(trim($search), true) . '%');
			$query->where($db->quoteName('a.title') . ' LIKE :search')
				->bind(':search', $search);
		}

		// Ordering
		$orderCol = $this->state->get('list.ordering', 'a.ordering');
		$orderDirn = $this->state->get('list.direction', 'ASC');
		$query->order($db->escape($orderCol) . ' ' . $db->escape($orderDirn));

		return $query;
	}

	/**
	 * Publish/unpublish records (simple bulk update).
	 *
	 * @param   array  $pks    Record ids.
	 * @param   int    $value  New published value.
	 *
	 * @return  bool
	 *
	 * @since  1.0.4
	 */
	public function publish(array $pks, int $value): bool
	{
		$pks = array_filter(array_map('intval', $pks));

		if (empty($pks)) {
			return false;
		}

		$db = $this->getDatabase();
		$query = $db->getQuery(true)
			->update($db->quoteName('#__one_customfields'))
			->set($db->quoteName('published') . ' = :published')
			->whereIn($db->quoteName('id'), $pks)
			->bind(':published', $value, ParameterType::INTEGER);

		$db->setQuery($query);
		$db->execute();

		return true;
	}

	/**
	 * Delete records (values will cascade via FK).
	 * Blocks delete if any field is used by listings (has values in content/events).
	 *
	 * @param   array  $pks  Record ids.
	 *
	 * @return  bool
	 *
	 * @since  1.0.4
	 */
	public function delete(array $pks): bool
	{
		$pks = array_filter(array_map('intval', $pks));

		if (empty($pks)) {
			return false;
		}

		$db = $this->getDatabase();

		// Check if any of these fields have values (content or events)
		$checkQuery = $db->createQuery()
			->select('1')
			->from($db->quoteName('#__one_customfield_values'))
			->whereIn($db->quoteName('field_id'), $pks);
		$db->setQuery($checkQuery);

		try {
			$used = (int) $db->loadResult();
		} catch (\Throwable $e) {
			$used = 0;
		}

		if ($used) {
			$this->setError(\Joomla\CMS\Language\Text::_('COM_ONECORE_CUSTOMFIELD_DELETE_IN_USE'));
			return false;
		}

		$query = $db->getQuery(true)
			->delete($db->quoteName('#__one_customfields'))
			->whereIn($db->quoteName('id'), $pks);

		$db->setQuery($query);
		$db->execute();

		return true;
	}
}

