<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Model\Customfieldgroups;

use Joomla\CMS\MVC\Model\ListModel;
use Joomla\Database\ParameterType;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Customfieldgroups model class.
 *
 * @since  1.0.9
 */
class CustomfieldgroupsModel extends ListModel
{
	/**
	 * Constructor.
	 *
	 * @param   array  $config  An optional associative array of configuration settings.
	 *
	 * @since   1.0.9
	 */
	public function __construct($config = [])
	{
		if (empty($config['filter_fields'])) {
			$config['filter_fields'] = [
				'id', 'g.id',
				'title', 'g.title',
				'category_id', 'g.category_id',
				'ordering', 'g.ordering',
				'published', 'g.published',
			];
		}

		parent::__construct($config);
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * @param   string  $ordering   An optional ordering field.
	 * @param   string  $direction  An optional direction (asc|desc).
	 *
	 * @return  void
	 *
	 * @since   1.0.9
	 */
	protected function populateState($ordering = 'g.ordering', $direction = 'ASC')
	{
		$app = \Joomla\CMS\Factory::getApplication();

		$search = $app->getUserStateFromRequest($this->context . '.filter.search', 'filter_search', '', 'string');
		$this->setState('filter.search', $search);

		$categoryId = $app->getUserStateFromRequest($this->context . '.filter.category_id', 'filter_category_id', '', 'string');
		$this->setState('filter.category_id', $categoryId);

		$published = $app->getUserStateFromRequest($this->context . '.filter.published', 'filter_published', '', 'string');
		$this->setState('filter.published', $published);

		parent::populateState($ordering, $direction);
	}

	/**
	 * Method to get a store id based on model configuration state.
	 *
	 * @param   string  $id  A prefix for the store id.
	 *
	 * @return  string  A store id.
	 *
	 * @since   1.0.9
	 */
	protected function getStoreId($id = '')
	{
		$id .= ':' . $this->getState('filter.search');
		$id .= ':' . $this->getState('filter.category_id');
		$id .= ':' . $this->getState('filter.published');

		return parent::getStoreId($id);
	}

	/**
	 * Build an SQL query to load the list data.
	 *
	 * @return  \Joomla\Database\QueryInterface
	 *
	 * @since   1.0.9
	 */
	protected function getListQuery()
	{
		$db    = $this->getDatabase();
		$query = $db->createQuery();

		$query->select(
			$db->quoteName(
				[
					'g.id',
					'g.category_id',
					'g.entity_type',
					'g.title',
					'g.description',
					'g.show_title',
					'g.show_description',
					'g.pos_items',
					'g.pos_item',
					'g.pos_module',
					'g.ordering',
					'g.published',
					'g.created',
					'g.modified',
				]
			)
		)
			->from($db->quoteName('#__one_customfield_groups', 'g'));

		// Join on category table
		$query->select($db->quoteName('c.title', 'category_title'))
			->leftJoin($db->quoteName('#__one_categories', 'c'), $db->quoteName('c.id') . ' = ' . $db->quoteName('g.category_id'));

		// Filter by search
		$search = $this->getState('filter.search');

		if (!empty($search)) {
			if (stripos($search, 'id:') === 0) {
				$query->where($db->quoteName('g.id') . ' = ' . (int) substr($search, 3));
			} else {
				$search = '%' . $db->escape($search, true) . '%';
				$query->where('(' . $db->quoteName('g.title') . ' LIKE :search1)')
					->bind(':search1', $search);
			}
		}

		// Filter by category
		$categoryId = $this->getState('filter.category_id');

		if (is_numeric($categoryId)) {
			$query->where($db->quoteName('g.category_id') . ' = :categoryId')
				->bind(':categoryId', $categoryId, ParameterType::INTEGER);
		}

		// Filter by published state
		$published = $this->getState('filter.published');

		if ($published !== '' && $published !== null) {
			$query->where($db->quoteName('g.published') . ' = :published')
				->bind(':published', $published, ParameterType::INTEGER);
		}

		// Add the list ordering clause.
		$orderCol  = $this->state->get('list.ordering', 'g.ordering');
		$orderDirn = $this->state->get('list.direction', 'ASC');

		$query->order($db->escape($orderCol) . ' ' . $db->escape($orderDirn));

		return $query;
	}

	/**
	 * Publish/unpublish records (simple bulk update).
	 *
	 * @param   array  $pks    An array of record primary keys.
	 * @param   int    $value  New published value.
	 *
	 * @return  bool  True on success.
	 *
	 * @since  1.0.9
	 */
	public function publish(array $pks, int $value): bool
	{
		if (empty($pks)) {
			return false;
		}

		$db = $this->getDatabase();
		$query = $db->createQuery()
			->update($db->quoteName('#__one_customfield_groups'))
			->set($db->quoteName('published') . ' = :published')
			->whereIn($db->quoteName('id'), $pks)
			->bind(':published', $value, ParameterType::INTEGER);

		$db->setQuery($query);

		try {
			$db->execute();
			return true;
		} catch (\Exception $e) {
			$this->setError($e->getMessage());
			return false;
		}
	}

	/**
	 * Delete records.
	 * Blocks delete if any group is used by listings (has field values in content/events).
	 *
	 * @param   array  $pks  An array of record primary keys.
	 *
	 * @return  bool  True on success.
	 *
	 * @since  1.0.9
	 */
	public function delete(array $pks): bool
	{
		if (empty($pks)) {
			return false;
		}

		$db = $this->getDatabase();

		// Get field IDs that belong to these groups
		$fieldIdsQuery = $db->createQuery()
			->select($db->quoteName('id'))
			->from($db->quoteName('#__one_customfields'))
			->whereIn($db->quoteName('group_id'), array_map('intval', $pks));
		$db->setQuery($fieldIdsQuery);
		$fieldIds = $db->loadColumn() ?: [];

		if (!empty($fieldIds)) {
			// Check if any of these fields have values (content or events)
			$checkQuery = $db->createQuery()
				->select('1')
				->from($db->quoteName('#__one_customfield_values'))
				->whereIn($db->quoteName('field_id'), array_map('intval', $fieldIds));
			$db->setQuery($checkQuery);

			try {
				$used = (int) $db->loadResult();
			} catch (\Throwable $e) {
				$used = 0;
			}

			if ($used) {
				$this->setError(\Joomla\CMS\Language\Text::_('COM_ONECORE_CUSTOMFIELDGROUP_DELETE_IN_USE'));
				return false;
			}
		}

		$query = $db->createQuery()
			->delete($db->quoteName('#__one_customfield_groups'))
			->whereIn($db->quoteName('id'), $pks);

		$db->setQuery($query);

		try {
			$db->execute();
			return true;
		} catch (\Exception $e) {
			$this->setError($e->getMessage());
			return false;
		}
	}
}
