<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Helper;

use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Layout\FileLayout;
use Comdev\Component\Onecore\Administrator\Helper\LicenseHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * OneCore component helper.
 *
 * @since  1.0.0
 */
class OneCoreHelper
{
	/**
	 * Ensure admin menu entry exists for Custom Fields (adds to DB if missing).
	 *
	 * @return void
	 *
	 * @since  1.0.5
	 */
	private static function ensureAdminMenuCustomFields(): void
	{
		$app = Factory::getApplication();

		if (!$app->isClient('administrator')) {
			return;
		}

		$user = $app->getIdentity();
		if (!$user || !$user->authorise('core.admin')) {
			return;
		}

		try {
			$db = Factory::getDbo();

			$link = 'index.php?option=com_onecore&view=customfields';

			// Exists?
			$query = $db->getQuery(true)
				->select($db->quoteName('id'))
				->from($db->quoteName('#__menu'))
				->where($db->quoteName('client_id') . ' = 1')
				->where($db->quoteName('type') . ' = ' . $db->quote('component'))
				->where($db->quoteName('link') . ' = ' . $db->quote($link))
				->setLimit(1);
			$db->setQuery($query);
			$exists = (int) $db->loadResult();

			if ($exists > 0) {
				return;
			}

			// Find parent (any existing com_onecore admin menu item)
			$query = $db->getQuery(true)
				->select($db->quoteName('id'))
				->from($db->quoteName('#__menu'))
				->where($db->quoteName('client_id') . ' = 1')
				->where($db->quoteName('type') . ' = ' . $db->quote('component'))
				->where($db->quoteName('link') . ' LIKE ' . $db->quote('index.php?option=com_onecore%'))
				->order($db->quoteName('id') . ' ASC')
				->setLimit(1);
			$db->setQuery($query);
			$parentId = (int) $db->loadResult();

			if ($parentId <= 0) {
				return;
			}

			// Component id
			$query = $db->getQuery(true)
				->select($db->quoteName('extension_id'))
				->from($db->quoteName('#__extensions'))
				->where($db->quoteName('type') . ' = ' . $db->quote('component'))
				->where($db->quoteName('element') . ' = ' . $db->quote('com_onecore'))
				->setLimit(1);
			$db->setQuery($query);
			$componentId = (int) $db->loadResult();

			if ($componentId <= 0) {
				return;
			}

			/** @var \Joomla\CMS\Table\Menu $table */
			$table = \Joomla\CMS\Table\Table::getInstance('Menu');
			$table->setLocation($parentId, 'last-child');

			$table->menutype = 'main';
			$table->title = 'COM_ONECORE_CUSTOMFIELDS';
			$table->alias = 'com-onecore-customfields';
			$table->note = '';
			$table->link = $link;
			$table->type = 'component';
			$table->published = 1;
			$table->parent_id = $parentId;
			$table->component_id = $componentId;
			$table->client_id = 1;
			$table->access = 1;
			$table->language = '*';
			$table->params = '{}';
			$table->img = 'class:tag';
			$table->home = 0;

			$table->check();
			$table->store();
		} catch (\Exception $e) {
			// ignore
		}
	}

	/**
	 * Set toolbar title with OneCore logo
	 *
	 * @param   string  $title  The title text
	 * @param   string  $icon   The icon name (optional)
	 *
	 * @return  void
	 *
	 * @since   1.0.0
	 */
	public static function setToolbarTitle($title, $icon = '')
	{
		self::ensureAdminMenuCustomFields();

		// Use standard layout for title
		$layout = new FileLayout('joomla.toolbar.title');
		$html = $layout->render(['title' => $title, 'icon' => $icon]);
		
		$app = Factory::getApplication();
		$app->JComponentTitle = $html;
		$pageTitle = strip_tags($title) . ' - ' . $app->get('sitename') . ' - ' . Text::_('JADMINISTRATION');
		$app->getDocument()->setTitle($pageTitle);
		
		// Add logo to subhead-container via JavaScript
		$wa = $app->getDocument()->getWebAssetManager();
		
		// Get logo URL based on license status
		$isLicenseActive = LicenseHelper::isLicenseActive();
		$logoFile = $isLicenseActive ? 'onecorepluslogo.png' : 'logo_h.png';
		$logoAlt = $isLicenseActive ? 'OneCore Plus' : 'OneCore';
		$logoUrl = \Joomla\CMS\Uri\Uri::root(true) . '/administrator/components/com_onecore/assets/' . $logoFile;
		
		$script = "
		document.addEventListener('DOMContentLoaded', function() {
			const subheadContainer = document.getElementById('subhead-container');
			if (subheadContainer) {
				const row = subheadContainer.querySelector('.row');
				if (row) {
					const col = row.querySelector('.col-md-12');
					if (col) {
						// Check if logo already exists
						if (!col.querySelector('.onecore-logo-toolbar')) {
							const logo = document.createElement('div');
							logo.className = 'onecore-logo-wrapper';
							const logoUrl = '" . $logoUrl . "';
							const logoAlt = '" . $logoAlt . "';
							logo.innerHTML = '<span class=\"onecore-logo-toolbar\"><img src=\"' + logoUrl + '\" alt=\"' + logoAlt + '\" /></span>';
							// Insert before nav element if it exists, otherwise at the beginning
							const nav = col.querySelector('nav');
							if (nav) {
								col.insertBefore(logo, nav);
							} else {
								col.insertBefore(logo, col.firstChild);
							}
						}
					}
				}
			}
		});
		";
		
		$wa->addInlineScript($script, ['name' => 'onecore-logo'], ['type' => 'module']);
		
		$css = "
		.onecore-logo-toolbar {
			display: inline-flex;
			align-items: center;
			vertical-align: middle;
		}
		.onecore-logo-toolbar img {
			height: auto;
			width: 200px;
			display: block;
			max-width: 200px;
		}
		.onecore-logo-wrapper {
			display: flex;
			align-items: center;
			flex-shrink: 0;
			margin-right: auto;
		}
		#subhead-container .row .col-md-12 {
			display: flex;
			align-items: center;
			justify-content: space-between;
			flex-wrap: wrap;
		}
		#subhead-container .row .col-md-12 > nav {
			display: flex;
			align-items: center;
			margin-left: auto;
		}
		";
		
		$wa->addInlineStyle($css, ['name' => 'onecore-logo-styles']);
	}
}
