<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Administrator\Controller;

use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Controller\BaseController;
use Joomla\CMS\Response\JsonResponse;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Component Controller
 *
 * @since  1.0.0
 */
class DisplayController extends BaseController
{
	/**
	 * The default view.
	 *
	 * @var    string
	 * @since  1.0.0
	 */
	protected $default_view = 'dashboard';

	/**
	 * Get chart data via AJAX
	 *
	 * @return  void
	 *
	 * @since   1.0.10
	 */
	public function getChartData()
	{
		try {
			// Check token
			$this->checkToken('get');

			// Set format to json
			$this->app->setHeader('Content-Type', 'application/json; charset=utf-8');

			// Load model using MVCFactory
			$factory = $this->app->bootComponent('com_onecore')->getMVCFactory();
			$model = $factory->createModel('Dashboard', 'Administrator', ['ignore_request' => true]);
			
			if (!$model) {
				// Fallback: try direct loading
				$className = 'Comdev\\Component\\Onecore\\Administrator\\Model\\Dashboard\\DashboardModel';
				if (!class_exists($className)) {
					$modelPath = JPATH_ADMINISTRATOR . '/components/com_onecore/src/Model/Dashboard/DashboardModel.php';
					if (file_exists($modelPath)) {
						require_once $modelPath;
					}
				}
				
				if (class_exists($className)) {
					$db = Factory::getDbo();
					$model = new $className(['dbo' => $db]);
				}
			}
			
			if (!$model || !method_exists($model, 'getStatistics')) {
				throw new \Exception('Dashboard model could not be loaded. Model: ' . ($model ? get_class($model) : 'null'));
			}
			
			$stats = $model->getStatistics();

			// Prepare chart data
			$chartData = [
				'labels' => [],
				'data' => [],
				'max' => 0
			];

			if (!empty($stats['listings_per_day']) && is_array($stats['listings_per_day'])) {
				foreach ($stats['listings_per_day'] as $dayData) {
					if (isset($dayData['day']) && isset($dayData['count'])) {
						$chartData['labels'][] = $dayData['day'];
						$chartData['data'][] = (int) $dayData['count'];
						if ($dayData['count'] > $chartData['max']) {
							$chartData['max'] = (int) $dayData['count'];
						}
					}
				}
			}

			// Ensure max is at least 1 to avoid division by zero
			if ($chartData['max'] == 0) {
				$chartData['max'] = 1;
			}

			// Return JSON response - JsonResponse wraps data in 'data' property
			$response = new JsonResponse($chartData, null, false, true);
			echo $response;
			$this->app->close();

		} catch (\Exception $e) {
			// Log error
			Factory::getApplication()->getLogger()->error(
				'OneCore Chart Data Error: ' . $e->getMessage(),
				['category' => 'onecore', 'exception' => $e]
			);

			// Return error response
			$this->app->setHeader('Status', '500 Internal Server Error', true);
			echo new JsonResponse([
				'success' => false,
				'message' => 'Error loading chart data: ' . $e->getMessage(),
				'data' => [
					'labels' => [],
					'data' => [],
					'max' => 1
				]
			]);
			$this->app->close();
		}
	}
}
