<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

\defined('_JEXEC') or die;

use Comdev\Component\Onecore\Site\Helper\RouteHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Date\Date;

/** @var \Comdev\Component\Onecore\Site\View\Events\HtmlView $this */

$app = Factory::getApplication();
$input = $app->getInput();
$currentSort = $input->getString('sort', 'date');

// Format date helper
function formatEventDate($dateString) {
	if (empty($dateString) || $dateString === '0000-00-00 00:00:00') {
		return '';
	}
	try {
		$date = new Date($dateString);
		return HTMLHelper::_('date', $date, Text::_('DATE_FORMAT_LC3'));
	} catch (\Exception $e) {
		return $dateString;
	}
}

// Format time helper
function formatEventTime($dateString) {
	if (empty($dateString) || $dateString === '0000-00-00 00:00:00') {
		return '';
	}
	try {
		$date = new Date($dateString);
		return HTMLHelper::_('date', $date, 'H:i');
	} catch (\Exception $e) {
		return '';
	}
}
?>
<div class="com-onecore-events-wrapper one_container">
	<?php if ($this->params->get('show_page_heading')) : ?>
		<div class="page-header one_header">
			<h1 class="one_page_heading"><?php echo $this->escape($this->params->get('page_heading', Text::_('COM_ONECORE_EVENTS'))); ?></h1>
		</div>
	<?php endif; ?>

	<!-- Map Section -->
	<?php 
	$showMap = (int) $this->params->get('show_map', 1) === 1;
	$hasMapData = $showMap && !empty($this->allMapItems);
	?>
	<?php if ($hasMapData) : ?>
		<div class="onecore-events-map-container mb-4" style="height: 400px; border: 1px solid #ddd; border-radius: 4px; position: relative;">
			<div id="onecore-events-map" style="width: 100%; height: 100%;"></div>
		</div>
	<?php endif; ?>

	<!-- Filters and Search -->
	<div class="onecore-events-filters mb-4 d-flex justify-content-between align-items-center flex-wrap gap-3">
		<div class="onecore-events-title">
			<h2><?php echo Text::_('COM_ONECORE_UPCOMING_EVENTS'); ?></h2>
		</div>
		<div class="onecore-events-controls d-flex align-items-center gap-2 flex-wrap">
			<!-- Sort -->
			<div class="onecore-events-sort">
				<label for="onecore-events-sort" class="visually-hidden"><?php echo Text::_('COM_ONECORE_SORT_BY'); ?></label>
				<select id="onecore-events-sort" name="sort" class="form-select form-select-sm">
					<option value="date" <?php echo $currentSort === 'date' ? 'selected' : ''; ?>>
						<?php echo Text::_('COM_ONECORE_SORT_BY_DATE'); ?>
					</option>
					<option value="newest" <?php echo $currentSort === 'newest' ? 'selected' : ''; ?>>
						<?php echo Text::_('COM_ONECORE_SORT_NEWEST'); ?>
					</option>
					<option value="oldest" <?php echo $currentSort === 'oldest' ? 'selected' : ''; ?>>
						<?php echo Text::_('COM_ONECORE_SORT_OLDEST'); ?>
					</option>
					<option value="a-z" <?php echo $currentSort === 'a-z' ? 'selected' : ''; ?>>
						<?php echo Text::_('COM_ONECORE_SORT_A_Z'); ?>
					</option>
					<option value="z-a" <?php echo $currentSort === 'z-a' ? 'selected' : ''; ?>>
						<?php echo Text::_('COM_ONECORE_SORT_Z_A'); ?>
					</option>
				</select>
			</div>
		</div>
	</div>

	<!-- Events List -->
	<?php if (!empty($this->items)) : ?>
		<div class="onecore-events-list">
			<?php foreach ($this->items as $item) : ?>
				<?php
				$eventStartDate = formatEventDate($item->event_start ?? '');
				$eventEndDate = formatEventDate($item->event_end ?? '');
				$eventStartTime = formatEventTime($item->event_start ?? '');
				$eventEndTime = formatEventTime($item->event_end ?? '');
				$eventUrl = Route::_(RouteHelper::getEventRoute($item->id . ':' . $item->alias));
				?>
				<div class="onecore-event-item mb-4 d-flex align-items-start gap-3<?php echo !empty($item->featured) ? ' onecore-event-item--featured' : ''; ?>" data-event-id="<?php echo (int) $item->id; ?>">
					<!-- Event Image -->
					<?php if (!empty($item->image)) : ?>
						<a href="<?php echo $eventUrl; ?>" class="onecore-event-image-link" style="width: 150px; height: 150px; flex-shrink: 0; display: block;">
							<img src="<?php echo htmlspecialchars(Uri::root() . $item->image, ENT_COMPAT, 'UTF-8'); ?>" 
								 alt="<?php echo htmlspecialchars($item->title, ENT_COMPAT, 'UTF-8'); ?>"
								 style="width: 100%; height: 100%; object-fit: cover; border-radius: 4px; transition: opacity 0.3s ease;">
						</a>
					<?php endif; ?>

					<!-- Event Details -->
					<div class="onecore-event-details flex-grow-1">
						<!-- Date -->
						<?php if (!empty($eventStartDate)) : ?>
							<div class="onecore-event-date mb-0" style="font-size: 1.2em; color: #6c757d; font-weight: 500;">
								<?php echo htmlspecialchars($eventStartDate, ENT_COMPAT, 'UTF-8'); ?>
							</div>
						<?php endif; ?>

						<!-- Title -->
						<h3 class="onecore-event-title mb-2 mt-2">
							<a href="<?php echo $eventUrl; ?>" style="text-decoration: none; color: inherit;">
								<?php echo htmlspecialchars($item->title, ENT_COMPAT, 'UTF-8'); ?>
								<?php if (!empty($item->featured)) : ?>
									<span class="badge bg-danger ms-2" title="<?php echo Text::_('JFEATURED'); ?>" style="font-size: 0.5em; padding: 0.25em 0.5em;">
										<span class="icon-star" aria-hidden="true"></span>
										<?php echo Text::_('JFEATURED'); ?>
									</span>
								<?php endif; ?>
							</a>
						</h3>

						<!-- Custom Fields Under Title -->
						<?php if (!empty($item->customfields_under_title)) : ?>
							<?php foreach ($item->customfields_under_title as $group) : ?>
								<div class="one_customfield_group one_customfield_group--under-title mb-2">
									<?php if (!empty($group->fields)) : ?>
										<div class="one_customfields one_customfields--under-title">
											<?php foreach ($group->fields as $cf) : ?>
												<div class="one_customfield">
													<?php if (!empty($cf->display_label)) : ?>
														<?php if (($cf->label_position ?? 'next') === 'over') : ?>
															<div class="one_customfield_label"><?php echo htmlspecialchars((string) $cf->title, ENT_QUOTES, 'UTF-8'); ?></div>
															<div class="one_customfield_value"><?php echo nl2br(htmlspecialchars((string) $cf->value, ENT_QUOTES, 'UTF-8')); ?></div>
														<?php else : ?>
															<span class="one_customfield_label"><?php echo htmlspecialchars((string) $cf->title, ENT_QUOTES, 'UTF-8'); ?>:</span>
															<span class="one_customfield_value"><?php echo nl2br(htmlspecialchars((string) $cf->value, ENT_QUOTES, 'UTF-8')); ?></span>
														<?php endif; ?>
													<?php else : ?>
														<span class="one_customfield_value"><?php echo nl2br(htmlspecialchars((string) $cf->value, ENT_QUOTES, 'UTF-8')); ?></span>
													<?php endif; ?>
												</div>
											<?php endforeach; ?>
										</div>
									<?php endif; ?>
								</div>
							<?php endforeach; ?>
						<?php endif; ?>

						<!-- Description -->
						<?php if (!empty($item->description)) : ?>
							<div class="onecore-event-description mb-2">
								<?php echo \Joomla\CMS\HTML\HTMLHelper::_('string.truncate', strip_tags($item->description), 150); ?>
							</div>
						<?php endif; ?>

						<!-- Custom Fields Under Intro -->
						<?php if (!empty($item->customfields_under_intro)) : ?>
							<?php foreach ($item->customfields_under_intro as $group) : ?>
								<div class="one_customfield_group one_customfield_group--under-intro mb-2">
									<?php if (!empty($group->fields)) : ?>
										<div class="one_customfields one_customfields--under-intro">
											<?php foreach ($group->fields as $cf) : ?>
												<div class="one_customfield">
													<?php if (!empty($cf->display_label)) : ?>
														<?php if (($cf->label_position ?? 'next') === 'over') : ?>
															<div class="one_customfield_label"><?php echo htmlspecialchars((string) $cf->title, ENT_QUOTES, 'UTF-8'); ?></div>
															<div class="one_customfield_value"><?php echo nl2br(htmlspecialchars((string) $cf->value, ENT_QUOTES, 'UTF-8')); ?></div>
														<?php else : ?>
															<span class="one_customfield_label"><?php echo htmlspecialchars((string) $cf->title, ENT_QUOTES, 'UTF-8'); ?>:</span>
															<span class="one_customfield_value"><?php echo nl2br(htmlspecialchars((string) $cf->value, ENT_QUOTES, 'UTF-8')); ?></span>
														<?php endif; ?>
													<?php else : ?>
														<span class="one_customfield_value"><?php echo nl2br(htmlspecialchars((string) $cf->value, ENT_QUOTES, 'UTF-8')); ?></span>
													<?php endif; ?>
												</div>
											<?php endforeach; ?>
										</div>
									<?php endif; ?>
								</div>
							<?php endforeach; ?>
						<?php endif; ?>

						<!-- Location and Time -->
						<div class="onecore-event-meta d-flex flex-wrap gap-3 mb-2">
							<?php if (!empty($item->address)) : ?>
								<div class="onecore-event-location d-flex align-items-center gap-1">
									<span>📍</span>
									<span><?php echo htmlspecialchars($item->address, ENT_COMPAT, 'UTF-8'); ?></span>
								</div>
							<?php endif; ?>
							<?php if (!empty($eventStartTime) && !empty($eventEndTime)) : ?>
								<div class="onecore-event-time d-flex align-items-center gap-1">
									<span>🕐</span>
									<span><?php echo htmlspecialchars($eventStartTime . '-' . $eventEndTime, ENT_COMPAT, 'UTF-8'); ?></span>
								</div>
							<?php endif; ?>
						</div>

						<!-- Categories -->
						<?php if (!empty($item->categories)) : ?>
							<div class="onecore-event-categories mb-2">
								<?php foreach ($item->categories as $category) : ?>
									<span class="badge bg-secondary me-1"><?php echo htmlspecialchars($category->title, ENT_COMPAT, 'UTF-8'); ?></span>
								<?php endforeach; ?>
							</div>
						<?php endif; ?>
					</div>
				</div>
			<?php endforeach; ?>
		</div>

		<!-- Pagination -->
		<?php if ($this->pagination->pagesTotal > 1) : ?>
			<div class="onecore-events-pagination mt-4">
				<?php echo $this->pagination->getPagesLinks(); ?>
			</div>
		<?php endif; ?>
	<?php else : ?>
		<div class="alert alert-info">
			<?php echo Text::_('COM_ONECORE_NO_EVENTS_FOUND'); ?>
		</div>
	<?php endif; ?>
</div>

<?php
// Load map script if we have map items and map is enabled
$showMap = (int) $this->params->get('show_map', 1) === 1;
if ($showMap && !empty($this->allMapItems)) :
	$wa = $app->getDocument()->getWebAssetManager();
	$wa->registerAndUseStyle('leaflet', 'https://unpkg.com/leaflet@1.9.4/dist/leaflet.css', [], ['rel' => 'stylesheet']);
	$wa->registerAndUseScript('leaflet', 'https://unpkg.com/leaflet@1.9.4/dist/leaflet.js', [], ['defer' => true]);
	
	// Prepare map data
	$mapData = [];
	foreach ($this->allMapItems as $mapItem) {
		$eventDate = formatEventDate($mapItem->event_start ?? '');
		$mapData[] = [
			'id' => (int) $mapItem->id,
			'title' => htmlspecialchars($mapItem->title, ENT_COMPAT, 'UTF-8'),
			'date' => htmlspecialchars($eventDate, ENT_COMPAT, 'UTF-8'),
			'lat' => (float) $mapItem->latitude,
			'lng' => (float) $mapItem->longitude,
		];
	}
	
	$defaultZoom = (int) $this->params->get('map_default_zoom', 13);
	$script = "
	(function() {
		'use strict';
		
		var mapData = " . json_encode($mapData) . ";
		var defaultZoom = " . $defaultZoom . ";
		
		function initMap() {
			if (typeof L === 'undefined') {
				setTimeout(initMap, 100);
				return;
			}
			
			var mapContainer = document.getElementById('onecore-events-map');
			if (!mapContainer || mapContainer.dataset.initialized) return;
			mapContainer.dataset.initialized = 'true';
			
			if (mapData.length === 0) return;
			
			var map;
			var bounds = [];
			
			// Create map
			if (mapData.length === 1) {
				map = L.map(mapContainer).setView([mapData[0].lat, mapData[0].lng], defaultZoom);
			} else {
				// Calculate bounds for multiple markers
				mapData.forEach(function(item) {
					bounds.push([item.lat, item.lng]);
				});
				var boundsObj = L.latLngBounds(bounds);
				map = L.map(mapContainer).fitBounds(boundsObj, {padding: [50, 50]});
			}
			
			L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
				attribution: '&copy; <a href=\"https://www.openstreetmap.org/copyright\">OpenStreetMap</a> contributors',
				maxZoom: 19
			}).addTo(map);
			
			// Add markers with popups
			mapData.forEach(function(item) {
				var marker = L.marker([item.lat, item.lng]).addTo(map);
				
				var popupContent = '<div style=\"min-width: 200px;\">';
				popupContent += '<b style=\"display: block; margin-bottom: 5px; font-size: 14px;\">' + item.title + '</b>';
				if (item.date) {
					popupContent += '<span style=\"color: #666; font-size: 12px;\">' + item.date + '</span>';
				}
				popupContent += '</div>';
				
				marker.bindPopup(popupContent);
			});
		}
		
		document.addEventListener('DOMContentLoaded', initMap);
		if (document.readyState === 'complete' || document.readyState === 'interactive') {
			setTimeout(initMap, 100);
		}
	})();
	";
	
	$wa->addInlineScript($script, ['name' => 'onecore-events-map'], ['type' => 'module']);
endif;

// Add search and sort handlers
$script = "
document.addEventListener('DOMContentLoaded', function() {
	var searchInput = document.getElementById('onecore-events-search-input');
	var sortSelect = document.getElementById('onecore-events-sort');
	
	function updateUrl() {
		var url = new URL(window.location.href);
		var search = searchInput ? searchInput.value.trim() : '';
		var sort = sortSelect ? sortSelect.value : 'date';
		
		if (search) {
			url.searchParams.set('search', search);
		} else {
			url.searchParams.delete('search');
		}
		
		if (sort && sort !== 'date') {
			url.searchParams.set('sort', sort);
		} else {
			url.searchParams.delete('sort');
		}
		
		url.searchParams.delete('limitstart');
		window.location.href = url.toString();
	}
	
	if (searchInput) {
		var searchTimeout;
		searchInput.addEventListener('input', function() {
			clearTimeout(searchTimeout);
			searchTimeout = setTimeout(function() {
				updateUrl();
			}, 500);
		});
		
		searchInput.addEventListener('keypress', function(e) {
			if (e.key === 'Enter') {
				e.preventDefault();
				updateUrl();
			}
		});
	}
	
	if (sortSelect) {
		sortSelect.addEventListener('change', function() {
			updateUrl();
		});
	}
});
";

$wa = $app->getDocument()->getWebAssetManager();
$wa->addInlineScript($script, ['name' => 'onecore-events-filters'], ['type' => 'module']);

$css = "
.onecore-events-map-container {
	margin-bottom: 2rem;
}
.onecore-events-filters {
	margin-bottom: 2rem;
}
.onecore-events-title h2 {
	margin: 0;
	font-size: 1.5rem;
	font-weight: 600;
}
.onecore-event-item {
	padding: 1rem;
	border: 1px solid #dee2e6;
	border-radius: 4px;
	background: #fff;
}
.onecore-event-item--featured {
	border: 2px solid #dc3545;
	border-radius: 4px;
}
.onecore-event-item:hover {
	box-shadow: 0 2px 8px rgba(0,0,0,0.1);
}
.onecore-event-title h3 {
	font-size: 1.25rem;
	font-weight: 600;
	margin: 0;
}
.onecore-event-title a:hover {
	text-decoration: underline !important;
}
.onecore-event-image-link:hover img {
	opacity: 0.8;
}
@media (max-width: 768px) {
	.onecore-event-item {
		flex-direction: column;
	}
	.onecore-event-image-link {
		width: 100% !important;
		height: 200px !important;
	}
}
";

$wa->addInlineStyle($css, ['name' => 'onecore-events-styles']);
?>
