<?php

/**
 * @package     Comdev.Component
 * @subpackage  com_onecore
 *
 * @copyright   (C) 2026 Comdev. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace Comdev\Component\Onecore\Site\View\Item;

use Comdev\Component\Onecore\Site\Helper\CustomFieldsHelper;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\View\HtmlView as BaseHtmlView;
use Joomla\CMS\MVC\View\GenericDataException;
use Joomla\CMS\Router\Route;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * HTML Item View class for the OneCore component
 *
 * @since  1.0.0
 */
class HtmlView extends BaseHtmlView
{
	/**
	 * The item object
	 *
	 * @var  \stdClass
	 */
	protected $item;

	/**
	 * The page parameters
	 *
	 * @var    \Joomla\Registry\Registry|null
	 */
	protected $params = null;

	/**
	 * The model state
	 *
	 * @var   \Joomla\Registry\Registry
	 */
	protected $state;

	/**
	 * Execute and display a template script.
	 *
	 * @param   string  $tpl  The name of the template file to parse; automatically searches through the template paths.
	 *
	 * @return  void
	 */
	public function display($tpl = null)
	{
		$app = Factory::getApplication();

		/** @var \Comdev\Component\Onecore\Site\Model\ItemModel $model */
		$model       = $this->getModel();
		$this->item  = $model->getItem();
		$this->state = $model->getState();

		// Check for errors.
		if (\count($errors = $model->getErrors())) {
			throw new GenericDataException(implode("\n", $errors), 500);
		}

		// Check if item exists
		if (!$this->item) {
			throw new GenericDataException(Text::_('COM_ONECORE_ERROR_ITEM_NOT_FOUND'), 404);
		}

		// Attach custom fields for item context ("item") in all positions
		$cid = (int) ($this->item->id ?? 0);
		if ($cid > 0) {
			$underTitle = CustomFieldsHelper::getFieldsForContentIds([$cid], 'item', 'under_title');
			$underIntro = CustomFieldsHelper::getFieldsForContentIds([$cid], 'item', 'under_intro');
			$sidebar = CustomFieldsHelper::getFieldsForContentIds([$cid], 'item', 'sidebar');

			$this->item->customfields_under_title = $underTitle[$cid] ?? [];
			$this->item->customfields_under_intro = $underIntro[$cid] ?? [];
			$this->item->customfields_sidebar = $sidebar[$cid] ?? [];
		}

		// Create a shortcut for $item.
		$item = $this->item;

		// Add router helpers.
		$item->slug = $item->alias ? ($item->id . ':' . $item->alias) : $item->id;

		// Load the parameters.
		$this->params = $this->state->get('params');
		$active       = $app->getMenu()->getActive();
		$temp         = clone $this->params;

		// Merge item params. If this is single-item view, menu params override item params
		if (
			$active
			&& $active->component == 'com_onecore'
			&& isset($active->query['view'], $active->query['id'])
			&& $active->query['view'] == 'item'
			&& $active->query['id'] == $item->id
		) {
			// Load layout from active query (in case it is an alternative menu item)
			if (isset($active->query['layout'])) {
				$this->setLayout($active->query['layout']);
			}

			// Merge so that the menu item params take priority
			$item->params->merge($temp);
		} else {
			// Merge the menu item params with the item params so that the item params take priority
			$temp->merge($item->params);
			$item->params = $temp;

			// Check for alternative layouts
			if ($layout = $item->params->get('item_layout')) {
				$this->setLayout($layout);
			}
		}

		// Check the view access to the item
		$user = Factory::getUser();
		$groups = $user->getAuthorisedViewLevels();
		
		// Check if user has access to this item's access level
		if (!in_array($item->access, $groups) && !$user->authorise('core.admin')) {
			if ($this->params->get('show_noauth', '0') == '0') {
				$app->enqueueMessage(Text::_('JERROR_ALERTNOAUTHOR'), 'error');
				$app->setHeader('status', 403, true);
				return;
			}
		}

		$this->_prepareDocument();

		parent::display($tpl);
	}

	/**
	 * Prepare the document (title, meta, canonical) for item view.
	 *
	 * @return  void
	 */
	protected function _prepareDocument(): void
	{
		$app        = Factory::getApplication();
		$item       = $this->item;
		$compParams = ComponentHelper::getParams('com_onecore');

		$this->params->def('page_heading', $this->params->get('page_title', $item->title ?? ''));

		$title = $this->params->get('page_title', '');
		if (empty($title)) {
			$title = $item->title ?? $app->get('sitename');
		}
		if ($app->get('sitename_pagetitles', 0) == 1) {
			$title = Text::sprintf('JPAGETITLE', $app->get('sitename'), $title);
		} elseif ($app->get('sitename_pagetitles', 0) == 2) {
			$title = Text::sprintf('JPAGETITLE', $title, $app->get('sitename'));
		}
		$this->getDocument()->setTitle($title);

		$metaDesc = trim((string) ($item->metadesc ?? ''));
		if ($metaDesc === '' && $this->params->get('menu-meta_description')) {
			$metaDesc = $this->params->get('menu-meta_description');
		}
		if ($metaDesc === '' && (int) $compParams->get('meta_desc_fallback', 1) === 1) {
			$intro = trim(strip_tags((string) ($item->introtext ?? '')));
			if ($intro !== '') {
				$metaDesc = mb_strlen($intro) > 160 ? mb_substr($intro, 0, 157) . '...' : $intro;
			}
		}
		if ($metaDesc !== '') {
			$this->getDocument()->setDescription($metaDesc);
		}

		$metaKey = trim((string) ($item->metakey ?? ''));
		if ($metaKey === '' && $this->params->get('menu-meta_keywords')) {
			$metaKey = $this->params->get('menu-meta_keywords');
		}
		if ($metaKey !== '') {
			$this->getDocument()->setMetadata('keywords', $metaKey);
		}

		$robots = $this->params->get('robots', '');
		if ($robots === '' && !empty($item->metadata)) {
			$meta = is_string($item->metadata) ? json_decode($item->metadata, true) : $item->metadata;
			if (!empty($meta['robots'])) {
				$robots = $meta['robots'];
			}
		}
		if ($robots === '' && $compParams->get('robots_fallback') !== '') {
			$robots = $compParams->get('robots_fallback');
		}
		if ($robots !== '') {
			$this->getDocument()->setMetadata('robots', $robots);
		}

		if ((int) $compParams->get('sef_add_canonical', 1) === 1) {
			$canonical = Route::_(
				'index.php?option=com_onecore&view=item&id=' . $item->slug,
				true,
				Route::TLS_IGNORE,
				true
			);
			$this->getDocument()->addHeadLink($canonical, 'canonical');
		}
	}
}
